<?php
 /**
 * Jamroom Redis Cache module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2003 - 2022 Talldude Networks, LLC.
 * @noinspection PhpExpressionResultUnusedInspection
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Open a Session
 * @param $path string
 * @param $name string
 * @return bool
 */
function _jrRedis_redis_session_open($path, $name)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_open', array($path, $name));
    }
    return true;
}

/**
 * Close a Session
 * @return bool
 */
function _jrRedis_redis_session_close()
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_close', array());
    }
    return true;
}

/**
 * Read an active Session
 * @param $sid String Current Session ID
 * @return string
 */
function _jrRedis_redis_session_read($sid)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_read', array($sid));
    }
    if ($rds = jrRedis_key_connect('session', md5($sid))) {
        $pfx = jrRedis_get_key_prefix();
        jrRedis_start_timer('access');
        try {
            $_vl = $rds->multi(Redis::PIPELINE)->get("{$pfx}{$sid}")->get("{$pfx}s{$sid}")->exec();
            jrRedis_stop_timer('access');
            // Do we have to sync session?
            if (isset($_vl[1]) && $_vl[1] == 1) {
                // Yes - we need to session sync
                jrRedis_start_timer('access');
                try {
                    $rds->del("{$pfx}s{$sid}");
                }
                catch (Exception $e) {
                    jrRedis_record_event('write_error');
                }
                jrRedis_stop_timer('access');
                jrCore_set_flag('user_session_sync', 1);
            }
            $val = $_vl[0];
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('write_error');
            return jrRedis_fallback('session', 'session_read', array($sid));
        }
        jrRedis_stop_timer('access');
        jrRedis_disconnect($rds);
        if ($val && strlen($val) > 0) {
            return $val;
        }
        return '';
    }
    return jrRedis_fallback('session', 'session_read', array($sid));
}

/**
 * Write an existing User session
 * @param $sid string Session ID to write to
 * @param $val string Session Value to save (set automatically by PHP)
 * @return bool
 */
function _jrRedis_redis_session_write($sid, $val)
{
    global $_post;
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_write', array($sid, $val));
    }
    if ($rds = jrRedis_key_connect('session', md5($sid))) {

        // p = IP address
        // u = user ID
        // d = profile ID
        // q = quota_id
        // g = user group
        // a = action
        // n = user name
        // v = user agent
        // t = update epoch

        $_data = array(
            'p' => jrCore_get_ip(),
            'u' => (isset($_SESSION['_user_id']) && is_numeric($_SESSION['_user_id'])) ? intval($_SESSION['_user_id']) : 0,
            'd' => (isset($_SESSION['_profile_id']) && is_numeric($_SESSION['_profile_id'])) ? intval($_SESSION['_profile_id']) : 0,
            'q' => (isset($_SESSION['profile_quota_id']) && is_numeric($_SESSION['profile_quota_id'])) ? intval($_SESSION['profile_quota_id']) : 0,
            'v' => (!empty($_SERVER['HTTP_USER_AGENT'])) ? $_SERVER['HTTP_USER_AGENT'] : '',
            't' => time()
        );
        if (!empty($_SESSION['user_name'])) {
            $_data['n'] = $_SESSION['user_name'];
            $_data['g'] = 'user';
            if (!empty($_SESSION['user_group'])) {
                $_data['g'] = $_SESSION['user_group'];
            }
            $exp = (int) (jrCore_get_config_value('jrUser', 'session_expire_min', 360) * 60);
        }
        else {
            if ($_data['n'] = jrUser_get_bot_name()) {
                $_data['u'] = $_data['n'];
                $exp        = jrCore_get_config_value('jrUser', 'session_expire_bot', 900);
            }
            else {
                // Logged out user - default to 15 minutes
                $_data['n'] = '';
                $_data['u'] = 0;
                $exp        = jrCore_get_config_value('jrUser', 'session_expire_visitor', 900);
            }
        }
        if (!jrCore_get_flag('jruser_ignore_action')) {
            if (!isset($_post['_uri']) || strlen($_post['_uri']) === 0) {
                $_data['a'] = '/';
            }
            elseif (jrCore_is_ajax_request()) {
                $_data['a'] = parse_url(jrCore_get_local_referrer(), PHP_URL_PATH);
                if (empty($_data['a'])) {
                    $_data['a'] = '/';
                }
                else {
                    $_data['a'] = jrCore_strip_emoji($_data['a'], false);
                }
            }
            elseif (jrCore_is_view_request()) {
                $_data['a'] = jrCore_strip_emoji($_post['_uri'], false);
            }
        }

        // Let other modules add additional session meta data that will be returned in get_online_info
        $_data = jrCore_trigger_event('jrRedis', 'session_meta_data', $_data);

        // 15 minute sessions for logged out users and bots
        $pfx = jrRedis_get_key_prefix();
        $key = jrRedis_get_distribution_key($sid);
        $sdt = jrRedis_array_to_string($_data);
        jrRedis_start_timer('access');

        // s[sid] = will be set if session SYNC flag is on
        // [pfx]d[dk] = $sid -> json(data)
        try {
            $rds->multi()->setEx("{$pfx}{$sid}", $exp, $val)->hSet("{$pfx}d{$key}", $sid, $sdt)->exec();
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('write_error');
            jrRedis_disconnect($rds);
            return jrRedis_fallback('session', 'session_write', array($sid, $val));
        }

        jrRedis_stop_timer('access');
        jrRedis_disconnect($rds);
        return true;
    }
    return jrRedis_fallback('session', 'session_write', array($sid, $val));
}

/**
 * Destroy an active session
 * @param $sid string SessionID
 * @return bool
 */
function _jrRedis_redis_session_destroy($sid)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_destroy', array($sid));
    }
    if ($rds = jrRedis_key_connect('session', md5($sid))) {
        $key = jrRedis_get_distribution_key($sid);
        $pfx = jrRedis_get_key_prefix();
        jrRedis_start_timer('access');
        try {
            $rds->multi(Redis::PIPELINE)->del("{$pfx}{$sid}")->del("{$pfx}s{$sid}")->hDel("{$pfx}d{$key}", $sid)->exec();
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('write_error');
            jrRedis_disconnect($rds);
            return jrRedis_fallback('session', 'session_destroy', array($sid));
        }
        jrRedis_stop_timer('access');
        jrRedis_disconnect($rds);
        return true;
    }
    return jrRedis_fallback('session', 'session_destroy', array($sid));
}

/**
 * Garbage collection for sessions
 * @param $max integer length of time session can be valid for
 * @return bool
 */
function _jrRedis_redis_session_collect($max)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_collect', array($max));
    }
    // GC handled in minute maintenance listener
    return true;
}

/**
 * Delete an individual session ID
 * @param string $sid Session ID to remove
 * @return mixed
 */
function _jrRedis_redis_session_delete_session_id($sid)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_delete_session', array($sid));
    }
    return _jrRedis_redis_session_destroy($sid);
}

/**
 * Remove all sessions for a specific user id
 * @param mixed $user_id mixed User ID or array of User ID's
 * @param string $exclude_sid if set to a session ID, dot remove it
 * @return mixed
 */
function _jrRedis_redis_session_remove($user_id, $exclude_sid = '')
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_remove', array($user_id));
    }
    $_sv = jrRedis_get_configured_servers_by_type('session');
    if ($_sv && is_array($_sv)) {
        $tot = 0;
        $uid = (int) $user_id;
        $pfx = jrRedis_get_key_prefix();
        $_ky = jrRedis_get_all_distribution_keys();
        foreach ($_sv as $server_id => $config) {
            if ($rds = jrRedis_connect('session', $server_id)) {

                // s[sid] = will be set if session SYNC flag is on
                // [pfx]d[dk] = $sid -> json(data)

                // p = IP address
                // u = user ID
                // d = profile ID
                // q = quota_id
                // g = user group
                // a = action
                // n = user name
                // v = user agent
                // t = update epoch

                jrRedis_start_timer('access');
                $rds->multi(Redis::PIPELINE);
                foreach ($_ky as $dky) {
                    $rds->hGetAll("{$pfx}d{$dky}");
                }
                $_sd = $rds->exec();
                if ($_sd && is_array($_sd)) {
                    $rds->multi(Redis::PIPELINE);
                    foreach ($_sd as $_sids) {
                        foreach ($_sids as $sid => $tmp) {
                            if (empty($exclude_sid) || $sid != $exclude_sid) {
                                if ($tmp = jrRedis_string_to_array($tmp)) {
                                    if ($tmp['u'] == $uid) {
                                        $key = jrRedis_get_distribution_key($sid);
                                        $rds->del("{$pfx}{$sid}")->del("{$pfx}s{$sid}")->hDel("{$pfx}d{$key}", $sid);
                                        $tot++;
                                    }
                                }
                            }
                        }
                    }
                    if ($tot > 0) {
                        $rds->exec();
                    }
                    else {
                        $rds->discard();
                    }
                    jrRedis_stop_timer('access');

                }
                jrRedis_disconnect($rds);
            }
        }
        return $tot;
    }
    return jrRedis_fallback('session', 'session_remove', array($user_id));
}

/**
 * Remove all sessions for a specific user id except active session
 * @param $user_id mixed User ID or array of User ID's
 * @param $active_sid string Active session ID
 * @return mixed
 */
function _jrRedis_redis_session_remove_all_other_sessions($user_id, $active_sid)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_remove_all_other_sessions', array($user_id, $active_sid));
    }
    return _jrRedis_redis_session_remove($user_id, $active_sid);
}

/**
 * Check if a given Session ID is a valid Session ID
 * @param $sid string Session ID
 * @return mixed
 */
function _jrRedis_redis_session_is_valid_session($sid)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_is_valid_session', array($sid));
    }
    if ($rds = jrRedis_key_connect('session', md5($sid))) {
        $pfx = jrRedis_get_key_prefix();
        jrRedis_start_timer('access');
        try {
            $tmp = $rds->exists("{$pfx}{$sid}");
        }
        catch (Exception $e) {
            jrRedis_stop_timer('access');
            jrRedis_record_event('read_error');
            jrRedis_disconnect($rds);
            return jrRedis_fallback('session', 'session_is_valid_session', array($sid));
        }
        jrRedis_stop_timer('access');
        jrRedis_disconnect($rds);
        return ($tmp == 1);
    }
    return jrRedis_fallback('session', 'session_is_valid_session', array($sid));
}

/**
 * Set the session_sync flag for a Quota ID
 * @param $quota_id mixed Quota ID or array of Quota IDs
 * @param $state string on|off
 * @return mixed
 */
function _jrRedis_redis_session_sync_for_quota($quota_id, $state)
{
    global $_conf;
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_sync_for_quota', array($quota_id, $state));
    }
    $_qi = array();
    if (jrCore_checktype($quota_id, 'number_nz')) {
        $quota_id       = (int) $quota_id;
        $_qi[$quota_id] = $quota_id;
    }
    elseif (is_array($quota_id)) {
        foreach ($quota_id as $qid) {
            if (jrCore_checktype($qid, 'number_nz')) {
                $qid       = (int) $qid;
                $_qi[$qid] = $qid;
            }
        }
    }
    if (count($_qi) === 0) {
        return false;
    }
    $_sv = jrRedis_get_configured_servers_by_type('session');
    if ($_sv && is_array($_sv)) {
        $exp = (int) ($_conf['jrUser_session_expire_min'] * 60);
        $pfx = jrRedis_get_key_prefix();
        $_ky = jrRedis_get_all_distribution_keys();
        foreach ($_sv as $server_id => $config) {
            if ($rds = jrRedis_connect('session', $server_id)) {
                jrRedis_start_timer('access');

                // s[sid] = will be set if session SYNC flag is on
                // [pfx]d[dk] = $sid -> json(data)

                // p = IP address
                // u = user ID
                // d = profile ID
                // q = quota_id
                // g = user group
                // a = action
                // n = user name
                // v = user agent
                // t = update epoch

                $rds->multi(Redis::PIPELINE);
                foreach ($_ky as $dky) {
                    $rds->hGetAll("{$pfx}d{$dky}");
                }
                $_sk = $rds->exec();
                if ($_sk && is_array($_sk)) {
                    $run = false;
                    $rds->multi(Redis::PIPELINE);
                    foreach ($_sk as $_sids) {
                        foreach ($_sids as $sid => $tmp) {
                            if ($_tmp = jrRedis_string_to_array($tmp)) {
                                $qid = (int) $_tmp['q'];
                                if (isset($_qi[$qid])) {
                                    $rds->setEx("{$pfx}s{$sid}", $exp, 1);
                                    $run = true;
                                }
                            }
                        }
                    }
                    if ($run) {
                        $rds->exec();
                    }
                    else {
                        $rds->discard();
                    }
                }
                jrRedis_disconnect($rds);
                jrRedis_stop_timer('access');
            }
        }
        return true;
    }
    return jrRedis_fallback('session', 'session_sync_for_quota', array($quota_id, $state));
}

/**
 * Set the session_sync flag for a User ID
 * @param $user_id mixed Quota ID or array of Quota IDs
 * @param $state string on|off
 * @return mixed
 */
function _jrRedis_redis_session_sync_for_user_id($user_id, $state)
{
    global $_conf;
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_sync_for_user_id', array($user_id, $state));
    }
    $_us = array();
    if (jrCore_checktype($user_id, 'number_nz')) {
        $_us[$user_id] = $user_id;
    }
    elseif (is_array($user_id)) {
        foreach ($user_id as $uid) {
            if (jrCore_checktype($uid, 'number_nz')) {
                $_us[$uid] = $uid;
            }
        }
    }
    if (count($_us) === 0) {
        return false;
    }

    $_sv = jrRedis_get_configured_servers_by_type('session');
    if ($_sv && is_array($_sv)) {
        $exp = (int) ($_conf['jrUser_session_expire_min'] * 60);
        $pfx = jrRedis_get_key_prefix();
        $_ky = jrRedis_get_all_distribution_keys();
        foreach ($_sv as $server_id => $config) {
            if ($rds = jrRedis_connect('session', $server_id)) {
                jrRedis_start_timer('access');

                // s[sid] = will be set if session SYNC flag is on
                // [pfx]d[dk] = $sid -> json(data)

                // p = IP address
                // u = user ID
                // d = profile ID
                // q = quota_id
                // g = user group
                // a = action
                // n = user name
                // v = user agent
                // t = update epoch

                $rds->multi(Redis::PIPELINE);
                foreach ($_ky as $dky) {
                    $rds->hGetAll("{$pfx}d{$dky}");
                }
                $_sk = $rds->exec();
                if ($_sk && is_array($_sk)) {
                    $run = false;
                    $rds->multi(Redis::PIPELINE);
                    foreach ($_sk as $_sids) {
                        foreach ($_sids as $sid => $tmp) {
                            if ($tmp = jrRedis_string_to_array($tmp)) {
                                if (isset($_us["{$tmp['u']}"])) {
                                    $rds->setEx("{$pfx}s{$sid}", $exp, 1);
                                    $run = true;
                                }
                            }
                        }
                    }
                    if ($run) {
                        $rds->exec();
                    }
                    else {
                        $rds->discard();
                    }
                }
                jrRedis_disconnect($rds);
                jrRedis_stop_timer('access');
            }
        }
        return true;
    }
    return jrRedis_fallback('session', 'session_sync_for_user_id', array($user_id, $state));
}

/**
 * Get number of active online users
 * @param $length int Max number of seconds with no activity a session is considered "active"
 * @param $type string type of user to count = user|bot
 * @return int
 */
function _jrRedis_redis_session_online_user_count($length, $type = 'combined')
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_online_user_count', array($length, $type));
    }
    $cnt = 0;
    $_sv = jrRedis_get_configured_servers_by_type('session');
    if ($_sv && is_array($_sv)) {

        $_us = array();
        $old = (time() - $length);
        $pfx = jrRedis_get_key_prefix();
        $_ky = jrRedis_get_all_distribution_keys();
        foreach ($_sv as $server_id => $config) {
            if ($rds = jrRedis_connect('session', $server_id)) {
                jrRedis_start_timer('session_info');

                // s[sid] = will be set if session SYNC flag is on
                // [pfx]d[dk] = $sid -> json(data)

                // p = IP address
                // u = user ID
                // d = profile ID
                // q = quota_id
                // g = user group
                // a = action
                // n = user name
                // v = user agent
                // t = update epoch

                $rds->multi(Redis::PIPELINE);
                foreach ($_ky as $dky) {
                    $rds->hGetAll("{$pfx}d{$dky}");
                }
                $_rt = $rds->exec();
                jrRedis_stop_timer('session_info');
                jrRedis_disconnect($rds);

                if ($_rt && is_array($_rt)) {
                    foreach ($_rt as $_sids) {
                        foreach ($_sids as $tmp) {
                            if ($tmp = jrRedis_string_to_array($tmp)) {
                                if ($tmp['t'] > $old) {
                                    // This user is ONLINE
                                    switch ($type) {
                                        case 'user':
                                            if (is_numeric($tmp['u']) && $tmp['u'] > 0) {
                                                $_us["{$tmp['u']}"] = 1;
                                            }
                                            break;
                                        case 'visitor':
                                            if ($tmp['u'] == 0) {
                                                $_us["{$tmp['u']}"] = 1;
                                            }
                                            break;
                                        case 'bot':
                                            if (!is_numeric($tmp['u'])) {
                                                $_us["{$tmp['u']}"] = 1;
                                            }
                                            break;
                                        default:
                                            // "combined" user + visitor (no bots)
                                            if (is_numeric($tmp['u'])) {
                                                $_us["{$tmp['u']}"] = 1;
                                            }
                                            break;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        $cnt = count($_us);
    }
    return $cnt;
}

/**
 * Return IDs of active users
 * @param int $length number of seconds of inactivity before user is "offline"
 * @param array $_ids only check the given IDs
 * @return mixed
 */
function _jrRedis_redis_session_online_user_ids($length = 900, $_ids = null)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_online_user_ids', array($length, $_ids));
    }
    $flg = 'jrredis_online_user_ids';
    if (!$_us = jrCore_get_flag($flg)) {
        $_us = array();
        $_sv = jrRedis_get_configured_servers_by_type('session');
        if ($_sv && is_array($_sv)) {
            $old = (time() - $length);
            $pfx = jrRedis_get_key_prefix();
            $_ky = jrRedis_get_all_distribution_keys();
            foreach ($_sv as $server_id => $config) {
                if ($rds = jrRedis_connect('session', $server_id)) {
                    jrRedis_start_timer('session_info');

                    // s[sid] = will be set if session SYNC flag is on
                    // [pfx]d[dk] = $sid -> json(data)

                    // p = IP address
                    // u = user ID
                    // d = profile ID
                    // q = quota_id
                    // g = user group
                    // a = action
                    // n = user name
                    // v = user agent
                    // t = update epoch

                    $rds->multi(Redis::PIPELINE);
                    foreach ($_ky as $dky) {
                        $rds->hGetAll("{$pfx}d{$dky}");
                    }
                    $_rt = $rds->exec();
                    jrRedis_stop_timer('session_info');
                    jrRedis_disconnect($rds);

                    if ($_rt && is_array($_rt)) {
                        foreach ($_rt as $_sids) {
                            foreach ($_sids as $tmp) {
                                if ($tmp = jrRedis_string_to_array($tmp)) {
                                    if ($tmp['t'] > $old) {
                                        if (!isset($_us["{$tmp['u']}"]) || $tmp['t'] > $_us["{$tmp['u']}"]) {
                                            $_us["{$tmp['u']}"] = $tmp['t'];
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        jrCore_set_flag($flg, $_us);
    }
    return $_us;
}

/**
 * Return information about users that are active and online
 * @param int $length include sessions active in last X seconds
 * @param string $search Optional Search string
 * @param array $_options additional flags and options
 * @return array|false
 */
function _jrRedis_redis_session_online_user_info($length = 900, $search = null, $_options = null)
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'session_online_user_info', array($length, $search));
    }

    $_nw = array();
    $_sv = jrRedis_get_configured_servers_by_type('session');
    if ($_sv && is_array($_sv)) {

        $_us = array();
        $_tm = array();
        $old = (time() - $length);
        $pfx = jrRedis_get_key_prefix();
        $_ky = jrRedis_get_all_distribution_keys();
        foreach ($_sv as $server_id => $config) {
            if ($rds = jrRedis_connect('session', $server_id)) {
                jrRedis_start_timer('session_info');

                // s[sid] = will be set if session SYNC flag is on
                // [pfx]d[dk] = $sid -> json(data)

                // p = IP address
                // u = user ID
                // d = profile ID
                // q = quota_id
                // g = user group
                // a = action
                // n = user name
                // v = user agent
                // t = update epoch

                $rds->multi(Redis::PIPELINE);
                foreach ($_ky as $key) {
                    $rds->hGetAll("{$pfx}d{$key}");
                }
                $_rs = $rds->exec();
                jrRedis_stop_timer('session_info');
                jrRedis_disconnect($rds);

                if ($_rs && is_array($_rs)) {

                    foreach ($_rs as $_sids) {
                        foreach ($_sids as $sid => $_inf) {
                            if ($_inf = jrRedis_string_to_array($_inf)) {
                                if ($_inf['t'] > $old) {
                                    $uid = (int) $_inf['u'];
                                    if ($uid === 0) {
                                        // This is a VISITOR (user_id = 0) - use IP address instead
                                        $uid = $_inf['p'];
                                    }
                                    if (!isset($_tm[$uid]) || $_inf['t'] > $_tm[$uid]) {
                                        $_tm[$uid] = $_inf['t'];
                                        if (!is_null($search)) {
                                            if (isset($_inf['n']) && stripos(' ' . $_inf['n'], $search)) {
                                                $_inf['sid'] = $sid;
                                                $_us[$uid]   = $_inf;
                                            }
                                        }
                                        else {
                                            $_inf['sid'] = $sid;
                                            $_us[$uid]   = $_inf;
                                        }
                                    }
                                }
                            }
                        }
                    }

                    // Did we come out of that with users?
                    // If so we need to rename keys
                    if (count($_tm) > 0) {
                        arsort($_tm);
                        foreach ($_tm as $uid => $time) {
                            if (isset($_us[$uid])) {
                                $_nw[$uid] = array(
                                    'session_id'          => $_us[$uid]['sid'],
                                    'session_updated'     => $time,
                                    'session_user_id'     => $uid,
                                    'session_user_name'   => (isset($_us[$uid]['n'])) ? $_us[$uid]['n'] : '',
                                    'session_user_group'  => (isset($_us[$uid]['g'])) ? $_us[$uid]['g'] : '',
                                    'session_profile_id'  => $_us[$uid]['d'],
                                    'session_quota_id'    => $_us[$uid]['q'],
                                    'session_user_ip'     => $_us[$uid]['p'],
                                    'session_user_action' => (!empty($_us[$uid]['a'])) ? $_us[$uid]['a'] : '?',
                                    'session_user_agent'  => (!empty($_us[$uid]['v'])) ? $_us[$uid]['v'] : ''
                                );
                            }
                        }
                        $_nw = jrCore_trigger_event('jrRedis', 'session_online_user_info', $_nw, $_us);
                    }
                }
            }
        }
    }
    return $_nw;
}

/**
 * Redis User sessions minute maintenance plugin
 * @return bool
 */
function _jrRedis_redis_session_minute_maintenance()
{
    jrRedis_remove_expired_sessions();
    return true;
}

/**
 * Remove expired sessions
 * @return int
 */
function jrRedis_remove_expired_sessions()
{
    if (!class_exists('Redis')) {
        return jrRedis_fallback('session', 'remove_expired_sessions', array());
    }
    $num = 0;
    $_sv = jrRedis_get_configured_servers_by_type('session');
    if ($_sv && is_array($_sv)) {

        $pfx = jrRedis_get_key_prefix();
        $_ky = jrRedis_get_all_distribution_keys();
        foreach ($_sv as $server_id => $config) {
            if ($rds = jrRedis_connect('session', $server_id)) {
                jrRedis_start_timer('maintenance');

                // s[sid] = will be set if session SYNC flag is on
                // [pfx]d[dk] = $sid -> json(data)

                // p = IP address
                // u = user ID
                // d = profile ID
                // q = quota_id
                // g = user group
                // a = action
                // n = user name
                // t = update epoch

                $rds->multi(Redis::PIPELINE);
                foreach ($_ky as $dky) {
                    $rds->hGetAll("{$pfx}d{$dky}");
                }
                $_rs = $rds->exec();

                if ($_rs && is_array($_rs)) {

                    // Get the keys that still exist
                    // NOTE: keys that no longer exist have expired
                    $idx = 0;
                    $_id = array();
                    $rds->multi(Redis::PIPELINE);
                    foreach ($_rs as $_sids) {
                        foreach ($_sids as $s => $v) {
                            $rds->exists("{$pfx}{$s}");
                            $_id[$idx] = $s;
                            $idx++;
                        }
                    }
                    $_tm = $rds->exec();

                    $_dl = array();
                    if ($_tm && is_array($_tm)) {
                        foreach ($_tm as $k => $v) {
                            if ($v != 1) {
                                // This sid no longer exists - cleanup
                                $_dl[$_id[$k]] = 1;
                            }
                        }
                    }
                    $del = count($_dl);
                    if ($del > 0) {
                        // Delete session_user and session_data
                        $rds->multi(Redis::PIPELINE);
                        foreach ($_dl as $sid => $ignore) {
                            $dky = jrRedis_get_distribution_key($sid);
                            $rds->del("{$pfx}{$sid}")->del("{$pfx}s{$sid}")->hDel("{$pfx}d{$dky}", $sid);
                        }
                        $rds->exec();
                        $num += $del;
                    }
                }
                jrRedis_stop_timer('maintenance');
                jrRedis_disconnect($rds);
            }
        }
    }
    return $num;
}
