<?php
 /**
 * Jamroom Redis Cache module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2003 - 2022 Talldude Networks, LLC.
 * @noinspection PhpExpressionResultUnusedInspection
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Update the Redis Config cache
 * @return bool
 */
function jrRedis_update_redis_config()
{
    $tbl = jrCore_db_table_name('jrRedis', 'server');
    $req = "SELECT * FROM {$tbl} WHERE server_cache = 'on' OR server_session = 'on'";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if ($_rt && is_array($_rt)) {
        $_nw = array(
            'c' => array(),
            's' => array()
        );
        foreach ($_rt as $s) {
            $sid = (int) $s['server_id'];
            if (isset($s['server_cache']) && $s['server_cache'] == 'on') {
                if (isset($s['server_pass']) && strlen($s['server_pass']) > 0) {
                    $_nw['c'][$sid] = array(trim($s['server_host']), intval($s['server_port']), intval($s['server_base']), trim($s['server_pass']));
                }
                else {
                    $_nw['c'][$sid] = array(trim($s['server_host']), intval($s['server_port']), intval($s['server_base']));
                }
            }
            if (isset($s['server_session']) && $s['server_session'] == 'on') {
                if (isset($s['server_pass']) && strlen($s['server_pass']) > 0) {
                    $_nw['s'][$sid] = array(trim($s['server_host']), intval($s['server_port']), intval($s['server_base']), trim($s['server_pass']));
                }
                else {
                    $_nw['s'][$sid] = array(trim($s['server_host']), intval($s['server_port']), intval($s['server_base']));
                }
            }
        }
        jrCore_set_setting_value('jrRedis', 'config', json_encode($_nw));
        jrCore_delete_config_cache();
    }
    return true;
}

/**
 * Get a Redis Server by ID
 * @param int $server_id
 * @return array|false
 */
function jrRedis_get_server_by_id($server_id)
{
    $sid = (int) $server_id;
    $tbl = jrCore_db_table_name('jrRedis', 'server');
    $req = "SELECT * FROM {$tbl} WHERE server_id = {$sid}";
    return jrCore_db_query($req, 'SINGLE');
}

/**
 * Get configured Redis servers
 * @return array|false
 */
function jrRedis_get_config()
{
    global $_conf;
    if (!$_cf = jrCore_get_flag('jrredis_expanded_config')) {
        // Are we running on a cluster?
        if (jrCore_module_is_active('jrCloudClient')) {
            if ($_nw = jrCloudClient_get_cluster_config()) {
                if (is_array($_nw) && isset($_nw['redis'])) {
                    $_cf = array(
                        'c' => array(),
                        's' => array()
                    );
                    $idx = 0;
                    foreach ($_nw['redis'] as $s) {
                        $prt = 16379;
                        if (isset($s['server_port'])) {
                            $prt = (int) $s['server_port'];
                        }
                        if (isset($s['redis_cache']) && $s['redis_cache'] == 'on') {
                            $_cf['c'][$idx] = array(trim($s['server_host']), $prt, 0, trim($s['server_pass']));
                        }
                        if (isset($s['redis_session']) && $s['redis_session'] == 'on') {
                            $_cf['s'][$idx] = array(trim($s['server_host']), $prt, 0, trim($s['server_pass']));
                        }
                        $idx++;
                    }
                }
            }
        }
        else {
            $_cf = json_decode($_conf['jrRedis_config'], true);
        }
        jrCore_set_flag('jrredis_expanded_config', $_cf);
    }
    return (is_array($_cf)) ? $_cf : false;
}

/**
 * Get Configured redis servers by type (cache|session)
 * @param string $type
 * @return bool|mixed
 */
function jrRedis_get_configured_servers_by_type($type)
{
    if ($_cf = jrRedis_get_config()) {
        $_mp = jrCore_get_flag('jrredis_db_map');
        $key = (isset($_mp[$type][0])) ? $_mp[$type][0] : 'c';
        if (isset($_cf[$key])) {
            return jrCore_trigger_event('jrRedis', 'get_configured_servers', $_cf[$key], array('type' => $type));
        }
    }
    return false;
}

/**
 * Get the database for the type of data
 * @param string $type
 * @return int|mixed
 */
function jrRedis_get_database_for_type($type)
{
    $_map = jrCore_get_flag('jrredis_db_map');
    return (isset($_map[$type])) ? $_map[$type][1] : 0;
}

/**
 * Get the type from a database number
 * @param int $db
 * @return bool|int|string
 */
function jrRedis_get_type_for_database($db)
{
    if ($_map = jrCore_get_flag('jrredis_db_map')) {
        foreach ($_map as $t => $d) {
            if ($d[1] == $db) {
                return $t;
            }
        }
    }
    return false;
}

/**
 * Connect to Redis server based on key distribution
 * @see: https://github.com/phpredis/phpredis/#usage
 * @param string $type type from database map: cache|session|form_session|key_cache|image_cache|counter|flag
 * @param string $key
 * @return bool|object
 */
function jrRedis_key_connect($type, $key)
{
    $db = jrRedis_get_database_for_type($type);
    if ($_tm = jrRedis_get_configured_servers_by_type($type)) {
        $sid = 0;
        $num = count($_tm);
        if ($num > 1) {
            $sid = jrRedis_get_server_for_key($key, $num);
        }
        else {
            if (!isset($_tm[0])) {
                $_tm = array_values($_tm);
            }
        }
        if (isset($_tm[$sid])) {
            $password = (!empty($_tm[$sid][3])) ? $_tm[$sid][3] : false;
            $flag_key = "jr_redis_key_connect_{$sid}_{$_tm[$sid][0]}_{$_tm[$sid][1]}_{$password}_{$db}";
            if (!$rds = jrCore_get_flag($flag_key)) {
                $tmo = floatval(jrCore_get_config_value('jrRedis', 'timeout', '0.50'));
                jrRedis_start_timer('connect');
                try {
                    $rds = new Redis();
                    $rds->connect($_tm[$sid][0], $_tm[$sid][1], $tmo);
                    if ($password) {
                        $rds->auth($password);
                    }
                    $rds->select($db);
                }
                catch (Exception $e) {
                    jrRedis_stop_timer('connect');
                    jrCore_delete_flag($flag_key);
                    jrCore_trigger_event('jrRedis', 'redis_connect_error', array('ip_address' => $_tm[$sid][0], '_server' => $_tm, 'error' => $e->getMessage()));
                    return false;
                }
                jrRedis_stop_timer('connect');
                jrCore_set_flag($flag_key, $rds);
            }
            return $rds;
        }
    }
    return false;
}

/**
 * Connect to Redis server
 * @param string $type session|cache
 * @param int $server_id connect to a specific server ID
 * @return bool|object
 */
function jrRedis_connect($type, $server_id = 0)
{
    $db = jrRedis_get_database_for_type($type);
    if ($_tm = jrRedis_get_configured_servers_by_type($type)) {
        $sid = (int) $server_id;
        if (isset($_tm[$sid])) {
            $password = (!empty($_tm[$sid][3])) ? $_tm[$sid][3] : false;
            $flag_key = "jr_redis_key_connect_{$sid}_{$_tm[$sid][0]}_{$_tm[$sid][1]}_{$password}_{$db}";
            if (!$rds = jrCore_get_flag($flag_key)) {
                $tmo = floatval(jrCore_get_config_value('jrRedis', 'timeout', '0.50'));
                $rds = new Redis();
                jrRedis_start_timer('connect');
                try {
                    $rds->connect($_tm[$sid][0], $_tm[$sid][1], $tmo);
                    if (!empty($_tm[$sid][3])) {
                        $rds->auth($_tm[$sid][3]);
                    }
                    $rds->select($db);
                }
                catch (Exception $e) {
                    jrRedis_stop_timer('connect');
                    jrCore_trigger_event('jrRedis', 'redis_connect_error', array('ip_address' => $_tm[$sid][0], '_server' => $_tm, 'error' => $e->getMessage()));
                    return false;
                }
                jrRedis_stop_timer('connect');
                jrCore_set_flag($flag_key, $rds);
            }
            return $rds;
        }
    }
    return false;
}

/**
 * Disconnect from redis
 * @param object $rds Redis Connection Object
 * @return true
 */
function jrRedis_disconnect($rds)
{
    return true;
}

/**
 * Get the server the key will be placed on
 * @param string $key MD5 key hash
 * @param int $server_count Number of Servers to distribute across
 * @return int
 */
function jrRedis_get_server_for_key($key, $server_count)
{
    $max = (pow(16, 8) - 1);
    $key = base_convert(substr($key, 0, 8), 16, 10);
    if ($key == $max) {
        $key--;
    }
    $tmp = round($max / $server_count);
    return intval(floor($key / $tmp));
}

/**
 * Get a distribution key for a value (returns 0-9a-f)
 * @param string $val
 * @return string
 */
function jrRedis_get_distribution_key($val)
{
    return substr(md5($val), 8, 1);
}

/**
 * Get all distribution keys
 * @return array
 */
function jrRedis_get_all_distribution_keys()
{
    return array('0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f');
}

/**
 * Get the Redis key prefix
 * @return string
 */
function jrRedis_get_key_prefix()
{
    if ($pfx = jrCore_get_config_value('jrRedis', 'key_prefix', false)) {
        if (!empty($pfx)) {
            return $pfx;
        }
    }
    return substr(md5(APP_DIR), 0, 5);
}

/**
 * Get binary of MD5
 * @note This reduces key storage requirement from 32 to 16 bytes
 * @param string $key
 * @return string
 */
function jrRedis_get_packed_key($key)
{
    return $key;
}

/**
 * Start a Redis timer
 * @param string $timer_name
 * @return bool
 */
function jrRedis_start_timer($timer_name)
{
    if (function_exists('jrCloudClient_start_timer')) {
        jrCloudClient_start_timer('jrRedis', $timer_name);
    }
    return true;
}

/**
 * Stop a Redis timer
 * @param string $timer_name
 * @return bool
 */
function jrRedis_stop_timer($timer_name)
{
    if (function_exists('jrCloudClient_stop_timer')) {
        jrCloudClient_stop_timer('jrRedis', $timer_name);
    }
    return true;
}

/**
 * Record a Redis Event
 * @param string $event_name
 * @return bool
 */
function jrRedis_record_event($event_name)
{
    if (function_exists('jrCloudClient_record_event')) {
        jrCloudClient_record_event('jrRedis', $event_name);
    }
    return true;
}

/**
 * Run a fallback function
 * @param string $type
 * @param string $function
 * @param array $_args
 * @return bool|mixed
 */
function jrRedis_fallback($type, $function, $_args)
{
    if (jrCore_get_config_value('jrRedis', 'fallback', 'on') == 'on') {
        jrCore_record_event('redis_fallback');
        switch ($type) {
            case 'session':
                $plug = 'jrUser_mysql';
                break;
            default:
                $plug = 'jrCore_mysql';
                break;
        }
        $func = "_{$plug}_{$function}";
        if (function_exists($func)) {
            return call_user_func_array($func, $_args);
        }
    }
    return false;
}

/**
 * Get string version of array for storing in Redis
 * @param array $array
 * @return string
 */
function jrRedis_array_to_string($array)
{
    if (function_exists('igbinary_serialize')) {
        return igbinary_serialize($array);
    }
    return json_encode($array);
}

/**
 * Decode a string back to an array
 * @param string $string
 * @param bool $assoc set to FALSE for object
 * @return array
 */
function jrRedis_string_to_array($string, $assoc = true)
{
    if (function_exists('igbinary_unserialize') && strpos($string, '{') !== 0) {
        return igbinary_unserialize($string);
    }
    return json_decode($string, $assoc);
}
