<?php
 /**
 * Jamroom Redis Cache module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2003 - 2022 Talldude Networks, LLC.
 * @noinspection PhpExpressionResultUnusedInspection
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Counter: Store a value by key in Redis with optional expiration
 * @see https://redis.io/docs/latest/commands/expire/
 * @param string $module Module to save cache for
 * @param string $key Key to save cache for
 * @param int $value Initial counter value
 * @param int $expire_seconds Number of seconds until key expires
 * @param string $expire_mode Redis expire flag
 * @return int|false Returns count after incrementing on success, false on fail
 * @noinspection PhpExpressionResultUnusedInspection
 */
function jrRedis_redis_set_counter($module, $key, $value, $expire_seconds = 0, $expire_mode = 'NX')
{
    $key = md5($module . $key);
    if ($rds = jrRedis_key_connect('counter', $key)) {
        $key = jrRedis_get_packed_key($key);
        jrRedis_start_timer('counter_write');
        try {
            if (!empty($expire_seconds)) {
                $val = $rds->multi()->incr($key, $value)->expire($key, intval($expire_seconds), $expire_mode)->exec();
            }
            else {
                $val = $rds->incr($key, $value);
            }
        }
        catch (Exception $e) {
            jrRedis_stop_timer('counter_write');
            jrRedis_record_event('write_error');
            jrRedis_disconnect($rds);
            return false;
        }
        jrRedis_stop_timer('counter_write');
        jrRedis_disconnect($rds);
        return intval($val);
    }
    return false;
}

/**
 * Counter: Get an existing key from Redis
 * @param string $module Module to save cache for
 * @param string $key Key to save cache for
 * @return int|false Returns counter value on success, false if counter does not exist
 * @noinspection PhpExpressionResultUnusedInspection
 */
function jrRedis_redis_get_counter($module, $key)
{
    $key = md5($module . $key);
    if ($rds = jrRedis_key_connect('counter', $key)) {
        $key = jrRedis_get_packed_key($key);
        jrRedis_start_timer('counter_read');
        try {
            $val = $rds->get($key);
        }
        catch (Exception $e) {
            jrRedis_stop_timer('counter_read');
            jrRedis_record_event('read_error');
            jrRedis_disconnect($rds);
            return false;
        }
        jrRedis_stop_timer('counter_read');
        jrRedis_disconnect($rds);
        return (!empty($val)) ? intval($val) : false;
    }
    return false;
}

/**
 * Counter: Delete an existing key from Redis
 * @param string $module Module to save cache for
 * @param string $key Key to save cache for
 * @return bool
 * @noinspection PhpExpressionResultUnusedInspection
 */
function jrRedis_redis_delete_counter($module, $key)
{
    $key = md5($module . $key);
    if ($rds = jrRedis_key_connect('counter', $key)) {
        $key = jrRedis_get_packed_key($key);
        jrRedis_start_timer('counter_write');
        try {
            $rds->del($key);
        }
        catch (Exception $e) {
            jrRedis_stop_timer('counter_write');
            jrRedis_record_event('write_error');
            jrRedis_disconnect($rds);
            return false;
        }
        jrRedis_stop_timer('counter_write');
        jrRedis_disconnect($rds);
    }
    return true;
}
