<?php
 /**
 * Jamroom URL Redirection module
 *
 * copyright 2023 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * jrRedirect_meta
 */
function jrRedirect_meta()
{
    return array(
        'name'        => 'URL Redirection',
        'url'         => 'r',
        'version'     => '1.1.4',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Redirect one URL to another with stats + URL shortening',
        'category'    => 'tools',
        'license'     => 'mpl',
        'priority'    => 251
    );
}

/**
 * jrRedirect_init
 */
function jrRedirect_init()
{
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrRedirect', 'browse', array('redirect rule browser', 'Browse, Create, Update, and Delete Redirect rules'));
    jrCore_register_module_feature('jrCore', 'admin_tab', 'jrRedirect', 'browse', 'Redirect Rule Browser');
    jrCore_register_module_feature('jrCore', 'default_admin_view', 'jrRedirect', 'browse');

    jrCore_register_event_listener('jrCore', 'verify_module', 'jrRedirect_verify_module_listener');
    jrCore_register_event_listener('jrCore', 'parsed_template', 'jrRedirect_parsed_template_listener');

    jrCore_register_event_listener('jrProfile', 'item_detail_view', 'jrRedirect_item_detail_view_listener');

    // Replace "regular" URLs with shortened ones in meta tags
    jrCore_register_event_listener('jrCore', 'db_get_item', 'jrRedirect_db_get_item_listener');

    // System reset listener
    jrCore_register_event_listener('jrDeveloper', 'reset_system', 'jrRedirect_reset_system_listener');

    // Graph Support
    $_tmp = array(
        'title'    => 'Redirected URLs by Day',
        'function' => 'jrRedirect_graph_redirects_by_day',
        'group'    => 'admin'
    );
    jrCore_register_module_feature('jrGraph', 'graph_config', 'jrRedirect', 'redirects_by_day', $_tmp);

    return true;
}

//----------------------
// GRAPH
//----------------------

/**
 * Redirects by Day
 * @param $module string Module
 * @param $name string Name of Graph to create
 * @param $_args array Passed in Parameters
 * @return array
 */
function jrRedirect_graph_redirects_by_day($module, $name, $_args)
{
    global $_post;
    $_rs = array(
        '_sets' => array(
            0 => array(
                'label'       => "Redirects",
                'date_format' => '%m/%d/%Y',
                'minTickSize' => "[1, 'day']",
                'type'        => 'line',
                'pointRadius' => 3,
                '_data'       => array()
            )
        )
    );
    // Get our data
    $iid = (int) $_post['id'];
    $tbl = jrCore_db_table_name('jrRedirect', 'stats');
    $req = "SELECT stat_date AS c, stat_value AS v FROM {$tbl} WHERE stat_redirect_id = '{$iid}' ORDER BY stat_date DESC LIMIT {$_args['days']}";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if ($_rt && is_array($_rt)) {
        $_rt = array_reverse($_rt);
        foreach ($_rt as $v) {
            $yr = substr($v['c'], 0, 4);
            $mn = substr($v['c'], 4, 2);
            $dy = substr($v['c'], 6, 2);
            $tm = (string) mktime(0, 0, 0, $mn, $dy, $yr);
            if (!isset($_rs['_sets'][0]['_data']["{$tm}"])) {
                $_rs['_sets'][0]['_data']["{$tm}"] = 0;
            }
            $_rs['_sets'][0]['_data']["{$tm}"] += $v['v'];
        }
    }
    return $_rs;
}

//----------------------
// EVENT LISTENERS
//----------------------

/**
 * System Reset listener
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrRedirect_reset_system_listener($_data, $_user, $_conf, $_args, $event)
{
    $tbl = jrCore_db_table_name('jrRedirect', 'stats');
    jrCore_db_query("TRUNCATE TABLE {$tbl}");
    jrCore_db_query("OPTIMIZE TABLE {$tbl}");
    return $_data;
}

/**
 * Add short URL to item data
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrRedirect_db_get_item_listener($_data, $_user, $_conf, $_args, $event)
{
    if (isset($_conf['jrRedirect_short_urls']) && $_conf['jrRedirect_short_urls'] == 'on') {
        if ($pfx = jrCore_db_get_prefix($_args['module'])) {
            if (isset($_data["{$pfx}_short_url_code"])) {
                $url = "{$_conf['jrCore_base_url']}/" . jrCore_get_module_url('jrRedirect') . "/" . jrCore_get_module_url($_args['module']) . '/' . $_data["{$pfx}_short_url_code"];
                jrCore_set_flag('jrredirect_short_url', $url);
                $_data["{$pfx}_item_short_url"] = $url;
            }
        }
    }
    return $_data;
}

/**
 * Add Template name as HTML comments if enabled
 * @param mixed $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrRedirect_parsed_template_listener($_data, $_user, $_conf, $_args, $event)
{
    // Replace regular item detail URL with short URL if enabled
    if (isset($_conf['jrRedirect_short_urls']) && $_conf['jrRedirect_short_urls'] == 'on' && isset($_args['jr_template']) && $_args['jr_template'] == 'item_detail_meta.tpl') {
        if ($url = jrCore_get_flag('jrredirect_short_url')) {
            // <meta property="og:url" content="http://local.jamroom.net/brian/blog/130/blog-test"/>
            if (strpos($_data, 'og:url')) {
                $_data = preg_replace(',(.*og:url.*content=")[^"]*,', "$1{$url}", $_data);
            }
        }
    }
    return $_data;
}

/**
 * Remove unused config option
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrRedirect_verify_module_listener($_data, $_user, $_conf, $_args, $event)
{
    jrCore_delete_setting('jrRedirect', 'active');
    return $_data;
}

/**
 * Listen for DS Items that need a short URL
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrRedirect_item_detail_view_listener($_data, $_user, $_conf, $_args, $event)
{
    // We check for incoming non-shortened URLs and if it matches a DS item
    // we go ahead and generate a shortened URL for this item
    if (isset($_conf['jrRedirect_short_urls']) && $_conf['jrRedirect_short_urls'] == 'on') {
        if ($pfx = jrCore_db_get_prefix($_args['module'])) {
            if (!isset($_data["{$pfx}_short_url_code"])) {
                jrCore_db_update_item($_args['module'], $_data['_item_id'], array("{$pfx}_short_url_code" => jrCore_create_unique_string(12)));
            }
        }
    }
    return $_data;
}

//----------------------
// FUNCTIONS
//----------------------

/**
 * Count a "hit" on a redirect URL
 * @param $id int Redirect ID
 * @return bool
 */
function jrRedirect_count_hit($id)
{
    $iid = (int) $id;
    $dat = date('Ymd');
    $tbl = jrCore_db_table_name('jrRedirect', 'stats');
    $req = "INSERT INTO {$tbl} (stat_redirect_id, stat_date, stat_value) VALUES ('{$iid}', '{$dat}', 1) ON DUPLICATE KEY UPDATE stat_value = (stat_value + 1)";
    $cnt = jrCore_db_query($req, 'COUNT');
    if ($cnt && $cnt > 0) {
        return true;
    }
    return false;
}
