<?php
 /**
 * Jamroom Profiles module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2021 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * view: create_save
 * @param array $_post Posted Data
 * @param array $_user Viewing User data
 * @param array $_conf Global Config
 */
function view_jrProfile_create_save($_post, $_user, $_conf)
{
    global $_urls;
    jrUser_session_require_login();
    // Make sure user is allowed to create profiles....
    if (!jrUser_is_power_user()) {
        jrUser_not_authorized();
    }
    jrCore_form_validate($_post);

    // Make sure the given profile name does not already exist
    $_rt = jrCore_db_get_item_by_key('jrProfile', 'profile_name', $_post['profile_name']);
    if ($_rt && is_array($_rt)) {
        jrCore_set_form_notice('error', 19);
        jrCore_form_field_hilight('profile_name');
        jrCore_form_result();
    }

    // Make sure it does not exist in the Recycle Bin
    if (jrUser_name_exists_in_recycle_bin($_post['profile_name'])) {
        jrCore_set_form_notice('error', 19);
        jrCore_form_field_hilight('profile_name');
        jrCore_form_result();
    }

    // Make sure user_name is not a banned word...
    if (jrCore_run_module_function('jrBanned_is_banned', 'name', $_post['profile_name'])) {
        jrCore_set_form_notice('error', 20);
        jrCore_form_field_hilight('profile_name');
        jrCore_form_result();
    }

    // Check for an active skin template with that name...
    $_tl = glob(APP_DIR . "/skins/{$_conf['jrCore_active_skin']}/*.tpl");
    $unm = jrCore_url_string($_post['profile_name']);
    foreach ($_tl as $tname) {
        if (strpos($tname, "/{$unm}.tpl")) {
            jrCore_set_form_notice('error', 'There is an active skin page using that name - please try another');
            jrCore_form_field_hilight('profile_name');
            jrCore_form_result();
            break;
        }
    }

    // Make sure it is NOT a module URL
    if (isset($_urls["{$_post['profile_name']}"]) || isset($_urls[$unm])) {
        jrCore_set_form_notice('error', 'There is a module already using that name - please try another');
        jrCore_form_field_hilight('profile_name');
        jrCore_form_result();
    }

    // Make sure we get a good profile_user_id
    if (jrUser_is_admin()) {
        if (!isset($_post['profile_user_id']) || !jrCore_checktype($_post['profile_user_id'], 'number_nz')) {
            jrCore_set_form_notice('error', 'You have entered an invalid profile owner - please search and select a valid profile owner');
            jrCore_form_field_hilight('profile_user_id');
            jrCore_form_result();
        }
        $_vu = jrCore_db_get_item('jrUser', $_post['profile_user_id']);
        if (!is_array($_vu)) {
            jrCore_set_form_notice('error', 'You have entered an invalid profile owner - please search and select a valid profile owner');
            jrCore_form_field_hilight('profile_user_id');
            jrCore_form_result();
        }
    }

    // Validate posted Quota
    $_qut = jrProfile_get_quotas();
    if (!jrUser_is_admin()) {
        // We're a power user and may only have access to selected Quotas
        $key = jrUser_get_profile_home_key('quota_jrUser_power_user_quotas');
        if (strpos($key, ',')) {
            $_all = array();
            foreach (explode(',', $key) as $qid) {
                if (isset($_qut[$qid])) {
                    $_all[$qid] = $_qut[$qid];
                }
            }
            $_qut = $_all;
            unset($_all);
        }
        elseif (jrCore_checktype($key, 'number_nz') && isset($_qut[$key])) {
            $_qut = array($key => $_qut[$key]);
        }
        else {
            jrCore_set_form_notice('error', 'Unable to determine Power User Quota - please contact the system adminstrator');
            jrCore_form_field_hilight('profile_quota_id');
            jrCore_form_result();
        }

        // Let's see how many profiles they have created
        $num = jrProfile_get_user_linked_profiles($_user['_user_id']);
        $max = jrUser_get_profile_home_key('quota_jrUser_power_user_max');
        if ($num && is_array($num) && count($num) >= $max) {
            jrCore_set_form_notice('error', 37);
            jrCore_form_result();
        }
    }

    $qid = (int) $_post['profile_quota_id'];
    if (!isset($_qut[$qid])) {
        jrCore_set_form_notice('error', 31);
        jrCore_form_field_hilight('profile_quota_id');
        jrCore_form_result();
    }

    // Get our posted data - the jrCore_form_get_save_data function will
    // return just those fields that were presented in the form.
    $_rt                    = jrCore_form_get_save_data('jrProfile', 'create', $_post);
    $_rt['profile_url']     = jrCore_url_string($_post['profile_name']);
    $_rt['profile_active']  = 1;
    $_rt['profile_private'] = 1;

    // Create new Profile
    $pid = jrCore_db_create_item('jrProfile', $_rt);
    if (!$pid) {
        jrCore_set_form_notice('error', 18);
        jrCore_form_result();
    }

    // If this is NOT an admin user, setup profile link
    if (jrUser_is_admin()) {
        $uid = $_post['profile_user_id'];
    }
    else {
        $uid = $_user['_user_id'];
        // This is a power user - add to user_active_profile_ids
        $_pr = array();
        if (isset($_SESSION['user_linked_profile_ids']) && strlen($_SESSION['user_linked_profile_ids']) > 0) {
            $_pr = explode(',', $_SESSION['user_linked_profile_ids']);
        }
        $_pr[]                               = $pid;
        $_SESSION['user_linked_profile_ids'] = implode(',', $_pr);
    }

    // Update with new profile id
    if (isset($uid)) {
        $_temp = array();
        $_core = array(
            '_user_id'    => $uid,
            '_profile_id' => $pid
        );
        jrCore_db_update_item('jrProfile', $pid, $_temp, $_core);
        jrProfile_create_user_link($uid, $pid);
    }

    // update the profile_count for the quota
    jrProfile_increment_quota_profile_count($qid);

    // Save any uploaded media files added in by our
    jrCore_save_all_media_files('jrProfile', 'create', $pid, $pid);

    jrCore_logger('INF', "created new profile: {$_post['profile_name']}");
    jrCore_form_delete_session();
    // Redirect to new Profile
    jrCore_form_result("{$_conf['jrCore_base_url']}/{$_rt['profile_url']}");
}
