<?php
 /**
 * Jamroom Profiles module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * Profile Smarty Functions and Modifiers
 * @copyright 2003-2022 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Get the signup quotas - used in template user signup
 * @param array $params
 * @param object $smarty
 * @return array|bool|int|mixed
 */
function smarty_function_jrProfile_get_signup_quota($params, $smarty)
{
    if ($_qt = jrProfile_get_signup_quotas()) {
        if (count($_qt) === 1) {
            // We only have 1 signup quota - return ID
            $_qt = array_keys($_qt);
            $_qt = reset($_qt);
        }
    }
    else {
        $_qt = false;
    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $_qt);
        return '';
    }
    return $_qt;
}

/**
 * Profile Delete button for authorized users
 * @param $params array Function parameters
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrProfile_delete_button($params, $smarty)
{
    // {jrCore_item_delete_button module="jrProfile" view="delete_save" profile_id=$_profile_id item_id=$_profile_id title="Delete this Profile" prompt="Are you sure you want to delete this profile?"}
    global $_conf;
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('profile_id');
    }
    if (!jrUser_is_logged_in()) {
        return '';
    }
    $_ln = jrUser_load_lang_strings();
    $run = false;
    if (jrUser_is_admin()) {
        $run = true;
    }
    elseif (isset($_conf['jrProfile_allow_delete']) && $_conf['jrProfile_allow_delete'] == 'on' && jrProfile_is_profile_owner($params['profile_id'])) {
        // Check if this is a POWER USER - if it is, they can
        // delete any profile that is NOT their home profile
        if (jrUser_is_power_user()) {
            if ($params['profile_id'] != jrUser_get_profile_home_key('_profile_id')) {
                $run = true;
            }
        }
        else {
            $run = true;
        }
    }
    if ($run) {
        $params['module']  = 'jrProfile';
        $params['view']    = 'delete_save';
        $params['item_id'] = $params['profile_id'];
        $params['title']   = $_ln['jrProfile'][39];
        $params['prompt']  = $_ln['jrProfile'][40];
        return smarty_function_jrCore_item_delete_button($params, $smarty);
    }
    return '';
}

/**
 * Disable Profile header for a specific module page
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrProfile_disable_header($params, $smarty)
{
    jrCore_set_flag('jrprofile_disable_option_enabled', 1);
    jrCore_set_flag('jrprofile_disable_header', 1);
    return '';
}

/**
 * Disable Profile sidebar for a specific module page
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrProfile_disable_sidebar($params, $smarty)
{
    jrCore_set_flag('jrprofile_disable_option_enabled', 1);
    jrCore_set_flag('jrprofile_disable_sidebar', 1);
    return '';
}

/**
 * Disable Profile footer for a specific module page
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrProfile_disable_footer($params, $smarty)
{
    jrCore_set_flag('jrprofile_disable_option_enabled', 1);
    jrCore_set_flag('jrprofile_disable_footer', 1);
    return '';
}

/**
 * Disable Profile module tabs for a specific module page
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrProfile_disable_module_tabs($params, $smarty)
{
    jrCore_set_flag('jrprofile_disable_option_enabled', 1);
    jrCore_set_flag('jrprofile_disable_module_tabs', 1);
    return '';
}

/**
 * Profile Statistics for modules that have registered
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrProfile_stats($params, $smarty)
{
    if (!isset($params['profile_id']) || !jrCore_checktype($params['profile_id'], 'number_nz')) {
        return 'jrProfile_stats: invalid profile_id parameter';
    }
    if (empty($params['template'])) {
        return 'jrProfile_stats: template parameter required';
    }
    $_tmp = jrCore_get_registered_module_features('jrProfile', 'profile_stats');
    if (!$_tmp) {
        // No registered modules
        return '';
    }
    $_prf = jrCore_db_get_item('jrProfile', $params['profile_id']);
    if (!$_prf || !is_array($_prf)) {
        // Invalid profile id
        return '';
    }
    $_prf['_stats'] = array();

    $_lang = jrUser_load_lang_strings();
    foreach ($_tmp as $mod => $_stats) {
        foreach ($_stats as $key => $title) {
            // Make sure this module is allowed in quota
            if (!isset($_prf["quota_{$mod}_allowed"]) || $_prf["quota_{$mod}_allowed"] != 'on') {
                continue;
            }
            if (is_numeric($title) && isset($_lang[$mod][$title])) {
                $title = $_lang[$mod][$title];
            }
            // See if we have been given a function
            $count = false;
            if (function_exists($key)) {
                $count = $key($_prf);
            }
            elseif (isset($_prf[$key]) && $_prf[$key] > 0) {
                $count = $_prf[$key];
            }
            if ($count) {
                $_prf['_stats'][$title] = array(
                    'count'  => $count,
                    'module' => $mod
                );
            }
        }
    }
    $out = '';
    if (isset($_prf['_stats']) && is_array($_prf['_stats'])) {
        $out = jrCore_parse_template($params['template'], $_prf, 'jrProfile');
    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Creates a URL to a specific Item page in a Profile
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrProfile_item_url($params, $smarty)
{
    global $_conf;
    if (!isset($params['module'])) {
        return 'jrProfile_item_url: module parameter required';
    }
    if (!isset($params['profile_url'])) {
        return 'jrProfile_item_url: profile_url parameter required';
    }
    if (!isset($params['item_id']) || !jrCore_checktype($params['item_id'], 'number_nz')) {
        return 'jrProfile_item_url: item_id required';
    }
    if (!isset($params['title']) || strlen($params['title']) === 0) {
        return 'jrProfile_item_url: title required';
    }
    if (!isset($params['title_url']) || strlen($params['title_url']) === 0) {
        $nam = jrCore_url_string($params['title']);
    }
    else {
        $nam = $params['title_url'];
    }
    $url = jrCore_get_module_url($params['module']);
    $out = "{$_conf['jrCore_base_url']}/{$params['profile_url']}/{$url}/{$params['item_id']}/{$nam}";
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Creates a dynamic Profile Menu based on the modules and options
 * That are available to a profile within their Profile Quota
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrProfile_menu($params, $smarty)
{
    global $_conf, $_mods, $_post;
    if (empty($params['template'])) {
        return 'jrProfile_menu: template parameter required';
    }
    if (!isset($params['profile_quota_id']) || !jrCore_checktype($params['profile_quota_id'], 'number_nz')) {
        return 'jrProfile_menu: valid profile_quota_id parameter required';
    }
    if (!isset($params['profile_url'])) {
        return 'jrProfile_menu: valid profile_url parameter required';
    }

    // Get profile info
    $_rt = jrCore_db_get_item_by_key('jrProfile', 'profile_url', $params['profile_url']);

    // Params event
    $params = jrCore_trigger_event('jrProfile', 'profile_menu_params', $params, $_rt);

    $_rt['_items'] = array();
    // We need to go through EACH module that has been passed in
    // and see if the module is ACTIVE and ALLOWED
    if (!isset($params['modules']) || strlen($params['modules']) === 0) {
        $params['modules'] = array_keys($_mods);
        sort($params['modules']);
    }
    else {
        $params['modules'] = explode(',', $params['modules']);
    }
    // Check for excluded modules
    if (isset($params['exclude_modules']) && strlen($params['exclude_modules']) > 0) {
        $_exc = explode(',', $params['exclude_modules']);
        $_exc = array_flip($_exc);
    }
    if (!isset($params['modules']) || !is_array($params['modules']) || count($params['modules']) === 0) {
        return '';
    }
    // Check for alternate module targets
    if (!empty($params['targets']) && strpos($params['targets'], ':')) {
        $_tgt = explode(',', $params['targets']);
        if (is_array($_tgt)) {
            foreach ($_tgt as $k => $target) {
                list($mod, $tgt) = explode(':', $target, 2);
                $mod = trim($mod);
                $tgt = trim($tgt);
                if (jrCore_module_is_active($mod) && strlen($tgt) > 0) {
                    $_tgt[$mod] = $tgt;
                }
                unset($_tgt[$k]);
            }
        }
    }

    // See if we are ordering our output
    if (isset($params['order']) && strlen($params['order']) > 0) {
        $_ord = explode(',', $params['order']);
        if (is_array($_ord)) {
            $_new = array();
            foreach ($_ord as $mod) {
                $_new[$mod] = $mod;
            }
            foreach ($params['modules'] as $mod) {
                if (!isset($_new[$mod])) {
                    $_new[$mod] = $mod;
                }
            }
            $params['modules'] = $_new;
            unset($_new);
        }
    }

    // BY default a menu button will NOT show if there are no entries in the DS
    // for that specific module/profile_id. We can override that behavior by
    // setting which modules should always show.
    $_show = false;
    if (isset($params['always_show']) && strlen($params['always_show']) > 0) {
        $_show = array_flip(explode(',', $params['always_show']));
    }

    // See if we are requiring a login
    $_login = array();
    if (isset($params['require_login']) && strlen($params['require_login']) > 0) {
        $_login = array_flip(explode(',', $params['require_login']));
    }

    // Bring in language strings
    $_lang = jrUser_load_lang_strings();

    // Get loaded template vars...
    $_prf = $smarty->getTemplateVars();

    // Modules can register "exclude", "always_show", "owner_only" and "require_login"
    $_rm = jrCore_get_registered_module_features('jrProfile', 'profile_menu');

    $_ac = array();
    foreach ($params['modules'] as $module) {
        switch ($module) {

            case 'jrCore':
            case 'jrProfile':
            case 'jrUser':
                // We can ignore some modules we know don't ever have menu items
                continue 2;

            default:

                // Module is NOT active
                if (!jrCore_module_is_active($module)) {
                    continue 2;
                }

                $mod_url = jrCore_get_module_url($module);
                // Module has been purposely excluded (parameter override)
                if (isset($_exc[$module])) {
                    continue 2;
                }
                elseif (isset($_rm[$module]['exclude']) && $_rm[$module]['exclude'] !== false) {
                    continue 2;
                }
                elseif (isset($_rm[$module]['active']) && strlen($_rm[$module]['active']) > 0) {
                    // This module wants us to show a different module as active
                    $_ac[$mod_url] = $_rm[$module]['active'];
                    continue 2;
                }

                // Check that we are not requiring a login (parameter override)
                if (isset($_login[$module]) && !jrUser_is_logged_in()) {
                    continue 2;
                }
                // Function init
                elseif (isset($_rm[$module]['require_login']) && $_rm[$module]['require_login'] !== false && !jrUser_is_logged_in()) {
                    continue 2;
                }

                // See if this Profile's Quota allows access to the module
                if (isset($_rt["quota_{$module}_allowed"]) && $_rt["quota_{$module}_allowed"] != 'on') {
                    continue 2;
                }

                // Owner Only
                if (isset($_rm[$module]['owner_only']) && $_rm[$module]['owner_only'] !== false && !jrProfile_is_profile_owner($_prf['_profile_id'])) {
                    continue 2;
                }

                // If this module REQUIRES another module, and the other module is excluded, we don't show
                if (!empty($_mods[$module]['module_requires'])) {
                    $_req = explode(',', $_mods[$module]['module_requires']);
                    if (is_array($_req)) {
                        foreach ($_req as $rmod) {
                            if (strpos($rmod, ':')) {
                                list($rmod,) = explode(':', $rmod);
                            }
                            $rmod = trim($rmod);
                            if (isset($_exc[$rmod]) || !isset($_mods[$rmod]) || isset($_rt["quota_{$rmod}_allowed"]) && $_rt["quota_{$rmod}_allowed"] != 'on' || isset($_login[$rmod]) && !jrUser_is_logged_in()) {
                                continue 3;
                            }
                        }
                    }
                }

                // Our module must have an item_index.tpl file...
                if (!is_file(APP_DIR . "/modules/{$module}/templates/item_index.tpl")) {
                    continue 2;
                }

                if (isset($_rm[$module]['always_show']) && $_rm[$module]['always_show'] !== false) {
                    if (!is_array($_show)) {
                        $_show = array();
                    }
                    $_show[$module] = true;
                }

                // See if we have been given an alternate target
                $target = $mod_url;
                if (isset($_tgt[$module])) {
                    $target .= '/' . $_tgt[$module];
                }
                // If there are NO ITEMS of this type, we only show the menu option to profile owners so they can create a new one.
                // [profile_jrAction_item_count] => 49
                // [profile_jrAudio_item_count] => 18
                if (!isset($_show[$module]) && (!isset($_prf["profile_{$module}_item_count"]) || $_prf["profile_{$module}_item_count"] == '0')) {
                    if (!jrProfile_is_profile_owner($_prf['_profile_id'])) {
                        continue 2;
                    }
                }
                $_rt['_items'][$module] = array(
                    'active'     => 0,
                    'module_url' => $mod_url,
                    'label'      => (isset($_lang[$module]['menu'])) ? $_lang[$module]['menu'] : $mod_url,
                    'target'     => "{$_conf['jrCore_base_url']}/{$params['profile_url']}/{$target}"
                );
                if ((isset($_post['option']) && $_post['option'] == $mod_url || isset($_rm[$module]['active']) && $_rm[$module]['active'] == $module)) {
                    $_rt['_items'][$module]['active'] = 1;
                }
                break;
        }
    }
    if (count($_ac) > 0 && isset($_post['option']) && isset($_ac["{$_post['option']}"])) {
        $mod                           = $_ac["{$_post['option']}"];
        $_rt['_items'][$mod]['active'] = 1;
    }
    $out = '';
    if (isset($_rt['_items']) && is_array($_rt['_items']) && count($_rt['_items']) > 0) {
        // Process template
        $_rt = jrCore_trigger_event('jrProfile', 'profile_menu_items', $_rt);
        $out = jrCore_parse_template($params['template'], $_rt);
    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}
