<?php
 /**
 * Jamroom Profiles module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * Profile Quota Functions
 * @copyright 2003-2022 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Get value of a quota setting from profile info
 * @param array $_profile Profile info
 * @param string $module Module setting advanced config item
 * @param string $key Unique Advanced Key Name
 * @param mixed $default Default value for key not set
 * @return mixed
 */
function jrProfile_get_quota_value($_profile, $module, $key, $default)
{
    return (isset($_profile["quota_{$module}_{$key}"])) ? $_profile["quota_{$module}_{$key}"] : $default;
}

/**
 * Get quota settings by quota_id
 * @param $quota_id integer Quota ID
 * @param $cache bool Set to false to disable memory caching for quota info
 * @return mixed
 */
function jrProfile_get_quota($quota_id, $cache = true)
{
    $key = "jrprofile_get_quota_{$quota_id}";
    if ($cache) {
        if ($_rt = jrCore_is_cached('jrProfile', $key, false, false)) {
            return $_rt;
        }
    }
    $tb1 = jrCore_db_table_name('jrProfile', 'quota_setting');
    $tb2 = jrCore_db_table_name('jrProfile', 'quota_value');
    $req = "SELECT s.`module` AS m, s.`name` AS k, s.`default` AS d, v.`value` AS v
              FROM {$tb1} s
         LEFT JOIN {$tb2} v ON (v.`quota_id` = '{$quota_id}' AND v.`module` = s.`module` AND v.`name` = s.`name`)";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if (!is_array($_rt)) {
        return false;
    }
    $_qt = array();
    foreach ($_rt as $v) {
        $_qt["quota_{$v['m']}_{$v['k']}"] = (isset($v['v'])) ? $v['v'] : $v['d'];
    }
    unset($_rt);
    if ($cache) {
        jrCore_add_to_cache('jrProfile', $key, $_qt, 0, 0, false, false);
    }
    return $_qt;
}

/**
 * Create a new profile quota
 * @param $name string Name for new Quota
 * @return mixed integer on success, bool false on failure
 */
function jrProfile_create_quota($name)
{
    // Make sure this quota does not already exist
    $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
    $req = "SELECT quota_id FROM {$tbl} WHERE `name` = 'name' AND `value` = '" . jrCore_db_escape($name) . "'";
    $_rt = jrCore_db_query($req, 'SINGLE');
    if (isset($_rt) && is_array($_rt)) {
        return $_rt['quota_id'];
    }
    $tbl = jrCore_db_table_name('jrProfile', 'quota');
    $req = "INSERT INTO {$tbl} (quota_created,quota_updated) VALUES (UNIX_TIMESTAMP(),UNIX_TIMESTAMP())";
    $qid = jrCore_db_query($req, 'INSERT_ID');
    if (isset($qid) && jrCore_checktype($qid, 'number_nz')) {
        jrProfile_set_quota_value('jrProfile', $qid, 'name', $name);
        return $qid;
    }
    return false;
}

/**
 * Delete an Existing Profile Quota
 * @param $id integer Quota ID to delete
 * @return bool
 */
function jrProfile_delete_quota($id)
{
    // Delete Quota
    $tbl = jrCore_db_table_name('jrProfile', 'quota');
    $req = "DELETE FROM {$tbl} WHERE quota_id = '{$id}'";
    $cnt = jrCore_db_query($req, 'COUNT');
    if (isset($cnt) && $cnt > 0) {
        // Delete Values
        $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
        $req = "DELETE FROM {$tbl} WHERE quota_id = '{$id}'";
        $cnt = jrCore_db_query($req, 'COUNT');
        if (isset($cnt) && $cnt > 0) {
            return true;
        }
    }
    return false;
}

/**
 * Increment profile count for a given quota
 * @param $quota_id int Quota ID to increment
 * @param $amount int Amount to increment by
 * @return true
 */
function jrProfile_increment_quota_profile_count($quota_id, $amount = 1)
{
    $qid = (int) $quota_id;
    $amt = (int) $amount;
    $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
    $req = "UPDATE {$tbl} SET `value` = (`value` + {$amt}) WHERE `quota_id` = '{$qid}' AND `module` = 'jrProfile' AND `name` = 'profile_count' LIMIT 1";
    $cnt = jrCore_db_query($req, 'COUNT');
    if ($cnt !== 1) {
        // This field does not exist - init
        jrProfile_set_quota_value('jrProfile', $qid, 'profile_count', $amt);
    }
    return true;
}

/**
 * Decrement profile count for a given quota
 * @param $quota_id int Quota ID to increment
 * @param $amount int Amount to increment by
 * @return true
 */
function jrProfile_decrement_quota_profile_count($quota_id, $amount = 1)
{
    $qid = (int) $quota_id;
    $amt = (int) $amount;
    $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
    $req = "UPDATE {$tbl} SET `value` = (`value` - {$amt}) WHERE `quota_id` = '{$qid}' AND `module` = 'jrProfile' AND `name` = 'profile_count' AND `value` >= {$amt} LIMIT 1";
    $cnt = jrCore_db_query($req, 'COUNT');
    if ($cnt !== 1) {
        // This field does not exist - init
        jrProfile_set_quota_value('jrProfile', $qid, 'profile_count', 0);
    }
    return true;
}

/**
 * Updates the profile count for a given quota_id
 * @param $quota_id integer Quota ID to update profile count for
 * @return bool
 * @deprecated
 */
function jrProfile_update_profile_count($quota_id)
{
    $qid = intval($quota_id);
    $tbl = jrCore_db_table_name('jrProfile', 'item_key');
    $req = "SELECT COUNT(_item_id) AS pcount FROM {$tbl} WHERE `key` = 'profile_quota_id' AND `value` = '{$qid}'";
    $_rt = jrCore_db_query($req, 'SINGLE');
    $num = 0;
    if (isset($_rt['pcount']) && jrCore_checktype($_rt['pcount'], 'number_nz')) {
        $num = intval($_rt['pcount']);
    }
    // Update quota value
    jrProfile_set_quota_value('jrProfile', $qid, 'profile_count', $num);
    return true;
}

/**
 * Create a new entry in the Quota Settings
 * @param $module string Module to create Quota setting for
 * @param $_field array Array of field information for new setting
 * @return bool
 */
function jrProfile_register_quota_setting($module, $_field)
{
    if (!isset($_field['type'])) {
        jrCore_notice('CRI', "jrProfile_register_quota_setting() required field: type missing for setting");
    }

    // example $_field:
    // $_tmp = array(
    //     'name'     => 'from_email',
    //     'label'    => 'from email address',
    //     'default'  => '',
    //     'type'     => 'text',
    //     'validate' => 'email',
    //     'help'     => 'When the system sends an automated / system message, what email address should the email be sent from?',
    //     'section'  => 'general email settings'
    // );
    // Optional:
    //     'min'      => (int) (minimum allow numerical value - validated)
    //     'max'      => (int) (maximum allow numerical value - validated)
    //     'options'  => array() (array of key => value pairs for fields with "select" type

    // See if we have already been called for this module/key in this process
    $key = jrCore_db_escape($_field['name']);
    $_tm = jrCore_get_flag('jrprofile_register_quota_setting');
    if (isset($_tm["{$module}_{$key}"])) {
        return true;
    }
    if (!$_tm || !is_array($_tm)) {
        $_tm = array();
    }
    $_tm["{$module}_{$key}"] = 1;
    jrCore_set_flag('jrprofile_register_quota_setting', $_tm);
    if (!$_orig = jrCore_get_flag('jrProfile_register_quota_setting_fields')) {
        $_orig = array();
    }
    $_orig["{$_field['name']}"] = $_field;
    if (isset($_orig["{$_field['name']}"]['value'])) {
        unset($_orig["{$_field['name']}"]['value']);
    }
    jrCore_set_flag('jrProfile_register_quota_setting_fields', $_orig);

    // Some items are required for form fields
    $_ri = array_flip(array('name', 'default', 'validate', 'label', 'help'));
    switch ($_field['type']) {
        // we already internally validate hidden and select elements
        case 'hidden':
            unset($_ri['validate'], $_ri['label'], $_ri['help']);
            break;
        case 'checkbox':
            $_field['validate'] = 'onoff';
            break;
        case 'select':
        case 'select_and_text':
        case 'select_multiple':
        case 'optionlist':
        case 'radio':
            unset($_ri['validate']);
            // Handle field options for select statements if set
            if (isset($_field['options']) && is_array($_field['options'])) {
                $_field['options'] = json_encode($_field['options']);
            }
            elseif (isset($_field['options']) && !function_exists($_field['options'])) {
                // These select options are generated at display time by a function
                jrCore_logger('CRI', "jrProfile_register_quota_setting() option function defined for field: {$_field['name']} does not exist");
                return false;
            }
            break;
    }
    foreach ($_ri as $k => $v) {
        if (!isset($_field[$k])) {
            jrCore_logger('CRI', "jrProfile_register_quota_setting() required field: {$k} missing for setting: {$_field['name']}");
            return false;
        }
    }
    // Make sure setting is properly updated
    return jrProfile_update_quota_setting($module, $_field);
}

/**
 * Create a new entry in the Quota Settings for other modules
 * @param $module string Module to create Quota setting for
 * @param $_field array Array of field information for new setting
 * @return bool
 * @deprecated
 */
function jrProfile_register_global_quota_setting($module, $_field)
{
    $_tmp = jrCore_get_flag('jrprofile_register_global_quota_setting');
    if (!$_tmp) {
        $_tmp = array();
    }
    if (!isset($_tmp[$module])) {
        $_tmp[$module] = array();
    }
    $_tmp[$module][] = $_field;
    jrCore_set_flag('jrprofile_register_global_quota_setting', $_tmp);
    return true;
}

/**
 * Verifies a Quota Setting is configured correctly in the settings
 * table - creates or updates as necessary.
 * @param $module string Module to create quota setting for
 * @param $_field array Array of setting information
 * @return bool
 */
function jrProfile_update_quota_setting($module, $_field)
{
    $tbl = jrCore_db_table_name('jrProfile', 'quota_setting');
    $req = "SELECT * FROM {$tbl} WHERE `module` = '" . jrCore_db_escape($module) . "' AND `name` = '" . jrCore_db_escape($_field['name']) . "'";
    $_ex = jrCore_db_query($req, 'SINGLE');
    $_rt = jrCore_db_table_columns('jrProfile', 'quota_setting');

    // Create
    if (!$_ex || !is_array($_ex)) {
        $_cl = array();
        $_vl = array();

        // When creating a NEW entry in settings, our value is set to the default
        $_field['value'] = $_field['default'];

        foreach ($_rt as $k => $v) {
            if (isset($_field[$k])) {
                $_cl[] = "`{$k}`";
                $_vl[] = jrCore_db_escape($_field[$k]);
            }
        }
        if (!is_array($_cl) || count($_cl) < 1) {
            return false;
        }
        $req = "INSERT INTO {$tbl} (`module`,`created`," . implode(',', $_cl) . ") VALUES ('" . jrCore_db_escape($module) . "',UNIX_TIMESTAMP(),'" . implode("','", $_vl) . "')";
    }
    // Update
    else {
        $req = "UPDATE {$tbl} SET ";
        $upd = false;
        foreach ($_rt as $k => $v) {
            if (isset($_field[$k])) {
                if ($_field[$k] != $_ex[$k]) {
                    $req .= "`{$k}` = '" . jrCore_db_escape($_field[$k]) . "',";
                    $upd = true;
                }
            }
        }
        if (!$upd) {
            return false;
        }
        $req = substr($req, 0, strlen($req) - 1) . " WHERE module = '" . jrCore_db_escape($module) . "' AND `name` = '" . jrCore_db_escape($_field['name']) . "' LIMIT 1";
    }
    $cnt = jrCore_db_query($req, 'COUNT');
    if (isset($_cl) && $cnt === 1) {
        jrCore_logger('INF', "validated quota setting for {$module} module: {$_field['name']}");
        return true;
    }
    return false;
}

/**
 * Deletes an existing quota setting from the quota settings table
 * @param $module string Module Name
 * @param $name string Quota Setting Name
 * @return bool
 */
function jrProfile_delete_quota_setting($module, $name)
{
    $tbl = jrCore_db_table_name('jrProfile', 'quota_setting');
    $req = "DELETE FROM {$tbl} WHERE `module` = '" . jrCore_db_escape($module) . "' AND `name` = '" . jrCore_db_escape($name) . "' LIMIT 1";
    $cnt = jrCore_db_query($req, 'COUNT');
    if ($cnt === 1) {
        $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
        $req = "DELETE FROM {$tbl} WHERE `module` = '" . jrCore_db_escape($module) . "' AND `name` = '" . jrCore_db_escape($name) . "' LIMIT 1";
        $cnt = jrCore_db_query($req, 'COUNT');
        if ($cnt === 1) {
            jrCore_logger('INF', "quota setting {$module}_{$name} was successfully deleted");
            return true;
        }
    }
    return false;
}

/**
 * Update a Quota setting to a new value
 * @param $module string Module to set Quota value for
 * @param $quota_id integer Quota ID to set value for
 * @param $name string Quota setting to set value for
 * @param $value mixed Value to set for $name
 * @return bool
 */
function jrProfile_set_quota_value($module, $quota_id, $name, $value)
{
    global $_user;
    if (!jrCore_checktype($quota_id, 'number_nz')) {
        return false;
    }
    $uid = (isset($_user['user_name']) && strlen($_user['user_name']) > 0 && isset($_user['user_group']) && $_user['user_group'] == 'master') ? jrCore_db_escape($_user['user_name']) : 'system';
    $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
    $qid = intval($quota_id);
    $mod = jrCore_db_escape($module);
    $nam = jrCore_db_escape($name);
    $val = jrCore_db_escape($value);
    $req = "INSERT INTO {$tbl} (`quota_id`,`module`,`name`,`updated`,`value`,`user`)
            VALUES ('{$qid}','{$mod}','{$nam}',UNIX_TIMESTAMP(),'{$val}','{$uid}')
            ON DUPLICATE KEY UPDATE `updated` = UNIX_TIMESTAMP(), `value` = '{$val}', `user` = '{$uid}'";
    $cnt = jrCore_db_query($req, 'COUNT');
    if ($cnt && $cnt > 0) {
        // We successfully changed or updated a Quota Value - this means we are going to need
        // to re-sync active sessions with the new Quota values
        $_tm = jrCore_get_flag('session_sync_quota_ids');
        if (!$_tm) {
            $_tm = array();
        }
        $_tm[$quota_id] = $quota_id;
        jrCore_set_flag('session_sync_quota_ids', $_tm);
        return true;
    }
    return false;
}

/**
 * Returns an array of all system quotas
 * @return mixed
 */
function jrProfile_get_quotas()
{
    $_rt = jrCore_get_flag('jrprofile_get_quotas');
    if ($_rt) {
        return $_rt;
    }
    $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
    $req = "SELECT `quota_id`, `value` FROM {$tbl} WHERE `module` = 'jrProfile' AND `name` = 'name' GROUP BY `quota_id` ORDER BY `value` ASC";
    $_rt = jrCore_db_query($req, 'quota_id');
    if (!$_rt || !is_array($_rt)) {
        return false;
    }
    foreach ($_rt as $k => $_v) {
        $_rt[$k] = $_v['value'];
    }
    jrCore_set_flag('jrprofile_get_quotas', $_rt);
    return $_rt;
}

/**
 * Get the "default" signup quota - used when unable to determine quota_id
 * @return int
 */
function jrProfile_get_default_signup_quota()
{
    $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
    $req = "SELECT `quota_id` FROM {$tbl} WHERE `name` IN('allow_signups','name') ORDER BY `quota_id` ASC LIMIT 1";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if ($_rt && is_array($_rt)) {
        return (int) $_rt['quota_id'];
    }
    return 1;
}

/**
 * Get quotas that allow signups
 * @return mixed
 */
function jrProfile_get_signup_quotas()
{
    $_rt = jrCore_get_flag('jrprofile_get_signup_quotas');
    if ($_rt) {
        return $_rt;
    }
    // Get Sign Up setting and Quota Names
    $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
    $req = "SELECT `quota_id`,`name`,`value` FROM {$tbl} WHERE `name` IN('allow_signups','name')";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    $_qt = array();
    if ($_rt && is_array($_rt)) {
        foreach ($_rt as $_quota) {
            if ($_quota['name'] == 'allow_signups' && $_quota['value'] == 'on') {
                $_qt["{$_quota['quota_id']}"] = 1;
            }
        }
        foreach ($_rt as $_quota) {
            if ($_quota['name'] == 'name' && isset($_qt["{$_quota['quota_id']}"])) {
                $_qt["{$_quota['quota_id']}"] = $_quota['value'];
            }
        }
    }
    if (count($_qt) > 0) {
        jrCore_set_flag('jrprofile_get_signup_quotas', $_qt);
        return $_qt;
    }
    return false;
}

/**
 * A special function used in the profile "settings" screen
 * @return array
 */
function jrProfile_get_settings_quotas()
{
    $_qt = false;
    if (jrUser_is_admin()) {
        $_qt = jrProfile_get_quotas();
    }
    else {
        // We're a power user and may only have access to selected Quotas
        $key = jrUser_get_profile_home_key('quota_jrUser_power_user_quotas');
        if (strpos($key, ',')) {
            $_aq = jrProfile_get_quotas();
            foreach (explode(',', $key) as $qid) {
                if (isset($_aq[$qid])) {
                    if (!$_qt) {
                        $_qt = array();
                    }
                    $_qt[$qid] = $_aq[$qid];
                }
            }
        }
        elseif (jrCore_checktype($key, 'number_nz')) {
            $_qt = jrProfile_get_quota($key);
            if ($_qt) {
                $_qt = array($key => $_qt['quota_jrProfile_name']);
            }
            else {
                $_qt = false;
            }
        }
    }
    return $_qt;
}

/**
 * Display Quota settings for the master admin
 * @param $module string Module to show Quota settings for
 * @param $_post array global $_post
 * @param $_user array global $_user
 * @param $_conf array global $_conf
 * @return string
 */
function jrProfile_show_module_quota_settings($module, $_post, $_user, $_conf)
{
    global $_mods;
    $_quota = jrCore_get_registered_module_features('jrCore', 'quota_support');
    // Make sure we have a valid quota.php file for this module
    if (!isset($_quota[$module]) && !is_file(APP_DIR . "/modules/{$module}/quota.php")) {
        jrCore_notice('CRI', "unable to open required quota config file: {$module}/quota.php");
    }
    // If we do not get a quota_id, we use the oldest created quota
    if (!isset($_post['id']) || !jrCore_checktype($_post['id'], 'number_nz')) {
        $tbl = jrCore_db_table_name('jrProfile', 'quota');
        $req = "SELECT quota_id FROM {$tbl} ORDER BY quota_created ASC LIMIT 1";
        $_rt = jrCore_db_query($req, 'SINGLE');
        if (isset($_rt) && is_array($_rt)) {
            $_post['id'] = (int) $_rt['quota_id'];
        }
        else {
            jrCore_notice('CRI', 'unable to retrieve any quotas from database!');
        }
    }

    if (isset($_post['hl']) && strlen($_post['hl']) > 0) {
        jrCore_form_field_hilight($_post['hl']);
    }

    // Bring in quota config
    if (is_file(APP_DIR . "/modules/{$module}/quota.php")) {
        $func = "{$module}_quota_config";
        if (!function_exists($func)) {
            require_once APP_DIR . "/modules/{$module}/quota.php";
        }
        if (function_exists($func)) {
            $func();
        }
    }

    // Get this module's quota config fields
    $tb1 = jrCore_db_table_name('jrProfile', 'quota_setting');
    $tb2 = jrCore_db_table_name('jrProfile', 'quota_value');
    $mod = jrCore_db_escape($module);
    $req = "SELECT s.*, v.`updated`, v.`value`, v.`user`, IF(LENGTH(s.`section`) = 0,'general',s.`section`) AS `section`
              FROM {$tb1} s LEFT JOIN {$tb2} v ON (v.`quota_id` = '{$_post['id']}' AND v.`module` = '{$mod}' AND v.`name` = s.`name`)
             WHERE s.`module` = '{$mod}' AND s.`type` != 'hidden'
             ORDER BY FIELD(s.`section`,'access') DESC, s.`order` ASC, s.`section` ASC, s.`name` ASC";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if (!$_rt || !is_array($_rt)) {
        // Ok this module is setup for Quota options - make sure the
        // "allowed" setting is setup if it is supported
        if (isset($_quota[$module])) {
            // Access
            $dval = 'off';
            if (is_array($_quota[$module]) && isset($_quota[$module]['on']) && $_quota[$module]['on'] != '1' && is_array($_quota[$module]['on'])) {
                if (isset($_quota[$module]['on']['default']) && $_quota[$module]['on']['default'] == 'on') {
                    $dval = 'on';
                }
            }
            $_tmp = array(
                'name'     => 'allowed',
                'label'    => 'allowed on profile',
                'default'  => $dval,
                'type'     => 'checkbox',
                'required' => 'on',
                'help'     => 'Should this Quota be allowed access to this module?',
                'validate' => 'onoff',
                'section'  => 'permissions',
                'order'    => 1
            );
            jrProfile_register_quota_setting($module, $_tmp);
        }
    }
    if (isset($_quota[$module]) && is_array($_quota[$module])) {
        $_tm = array_values($_quota[$module]);
        if ($_tm && isset($_tm[0]) && is_array($_tm[0])) {
            foreach ($_tm[0] as $k => $v) {
                if (isset($_rt[0][$k])) {
                    $_rt[0][$k] = $v;
                }
            }
        }
    }

    // Generate our output
    jrCore_page_admin_tabs($module, 'quota');

    $_mds = array();
    foreach ($_mods as $mod_dir => $_info) {
        if (!jrCore_module_is_active($mod_dir)) {
            continue;
        }
        if (isset($_quota[$mod_dir]) || is_file(APP_DIR . "/modules/{$mod_dir}/quota.php")) {
            $_mds[] = $mod_dir;
        }
    }
    $subtitle = jrCore_get_module_jumper('module_jumper', $_post['module'], "jrCore_window_location('{$_conf['jrCore_base_url']}/'+ $(this).val() + '/admin/quota/id={$_post['id']}')", $_mds);

    jrCore_page_banner('Quota Config', $subtitle);
    // See if we are disabled
    if (!jrCore_module_is_active($module)) {
        jrCore_set_form_notice('notice', 'This module is currently disabled');
    }
    jrCore_get_form_notice();

    // Form init
    $_tmp = array(
        'submit_value' => 'save changes',
        'action'       => 'admin_save/quota/id=' . intval($_post['id'])
    );
    jrCore_form_create($_tmp);

    // Show our Quota Jumper
    $_tmp = array(
        'name'      => 'id',
        'label'     => 'selected quota',
        'help'      => 'Select the Quota you would like to adjust the settings for',
        'type'      => 'select',
        'options'   => 'jrProfile_get_quotas',
        'value'     => $_post['id'],
        'error_msg' => 'you have selected an invalid quota',
        'validate'  => 'number_nz',
        'onchange'  => "var m=this.options[this.selectedIndex].value;self.location='{$_conf['jrCore_base_url']}/{$_post['module_url']}/admin/quota/id='+ m;"
    );
    jrCore_form_field_create($_tmp);

    // Apply to all quotas
    if ($_post['module'] != 'jrProfile') {
        $_tmp = array(
            'name'     => 'apply_to_all_quotas',
            'label'    => 'apply to all quotas',
            'help'     => 'If this option is checked, the quota settings saved here will be applied to all quotas.',
            'type'     => 'checkbox',
            'default'  => 'off',
            'validate' => 'onoff'
        );
        jrCore_form_field_create($_tmp);
    }

    $_orig = jrCore_get_flag('jrProfile_register_quota_setting_fields');
    foreach ($_rt as $_field) {
        if (!isset($_field['value']) || strlen($_field['value']) === 0) {
            $_field['value'] = $_field['default'];
        }
        if (isset($_orig["{$_field['name']}"]) && is_array($_orig["{$_field['name']}"])) {
            $_field = array_merge($_field, $_orig["{$_field['name']}"]);
        }
        // See if this is the "allowed" field - if so, our module can change
        // the label and help that is shown to the user
        if (isset($_field['name']) && $_field['name'] == 'allowed' && isset($_quota[$module]['on']) && $_quota[$module]['on'] != '1') {
            if (is_array($_quota[$module]['on'])) {
                // Both label and help
                $_field = array_merge($_field, $_quota[$module]['on']);
            }
            else {
                // Just label
                $_field['label'] = $_quota[$module]['on'];
            }
        }
        jrCore_form_field_create($_field);
    }

    jrCore_page_set_no_header_or_footer();
    return jrCore_page_display(true);
}

/**
 * Return true if a module has been enabled for a Quota
 * @param $module string Module to check for
 * @param $smarty object Current Smarty object
 * @param $value string on|off If provided will be used instead of smarty template value
 * @return bool
 */
function jrProfile_is_allowed_by_quota($module, $smarty, $value = null)
{
    if ($value == 'on') {
        return true;
    }
    // Not provided - find it
    elseif (isset($smarty->tpl_vars['item']->value["quota_{$module}_allowed"]) && $smarty->tpl_vars['item']->value["quota_{$module}_allowed"] == 'on') {
        return true;
    }
    elseif (isset($smarty->tpl_vars["quota_{$module}_allowed"]->value) && $smarty->tpl_vars["quota_{$module}_allowed"]->value == 'on') {
        return true;
    }
    return false;
}
