<?php
 /**
 * Jamroom Profiles module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * Profile Event Listeners
 * @copyright 2003-2022 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Watch for media file changes and update profile disk usage
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_save_media_file_listener($_data, $_user, $_conf, $_args, $event)
{
    if (!$_pr = jrCore_get_flag('jrprofile_media_changes')) {
        $_pr = array();
    }
    $pid       = (int) $_args['profile_id'];
    $_pr[$pid] = $pid;
    jrCore_set_flag('jrprofile_media_changes', $_pr);
    return $_data;
}

/**
 * Update disk usage for profiles that have changed
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_hourly_maintenance_listener($_data, $_user, $_conf, $_args, $event)
{
    // Have we had any media changes?
    if (jrCore_get_config_value('jrProfile', 'disable_profile_disk_usage', 'off') == 'off') {
        $_pr = jrCore_db_get_items_missing_key('jrProfile', 'profile_disk_usage', 1000);
        if ($_pr && is_array($_pr)) {
            foreach (array_chunk($_pr, 50) as $k => $_ch) {
                jrCore_queue_create('jrProfile', 'update_profile_disk_usage', array('profile_ids' => $_ch), $k);
            }
        }
    }
    return $_data;
}

/**
 * Watch for allowed number of items on a profile list view
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_item_index_view_listener($_data, $_user, $_conf, $_args, $event)
{
    if (!jrProfile_is_profile_owner($_data['_profile_id']) && isset($_args['module'])) {
        $cap_type  = jrProfile_get_quota_value($_data, 'jrProfile', 'cap_type', 'hard');
        $max_items = jrProfile_get_quota_value($_data, $_args['module'], 'max_items', 0);
        if ($cap_type == 'hard' && $max_items > 0) {
            jrCore_set_flag("profile_limit_item_count_{$_args['module']}", $max_items);
        }
    }
    return $_data;
}

/**
 * Get private profile id's
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_private_item_ids_listener($_data, $_user, $_conf, $_args, $event)
{
    // Get private profiles
    $_pp = jrCore_db_get_private_profiles();
    if ($_pp && is_array($_pp)) {
        $_data['jrProfile'] = array_keys($_pp);
    }
    return $_data;
}

/**
 * Re-sync users and process Pulse keys
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_process_exit_listener($_data, $_user, $_conf, $_args, $event)
{
    // Check for pulse key updates
    if ($_pk = jrCore_get_flag('jrprofile_process_pulse_keys')) {
        // We have pulse keys to increment
        $_in = array();
        foreach ($_pk as $id => $_tm) {
            $_in[] = "({$id}, '{$_tm[0]}', '{$_tm[1]}', UNIX_TIMESTAMP(), {$_tm[2]})";
            jrProfile_reset_cache($id, 'jrProfile');
        }
        $tbl = jrCore_db_table_name('jrProfile', 'pulse');
        $req = "INSERT INTO {$tbl} (pulse_profile_id, pulse_module, pulse_key, pulse_updated, pulse_count)
                VALUES " . implode(',', $_in) . " ON DUPLICATE KEY UPDATE pulse_count = (pulse_count + VALUES(pulse_count))";
        jrCore_db_query($req);
        jrCore_delete_flag('process_pulse_keys');
    }

    // See if we have had an updated Quota config and need to re-sync sessions
    if ($_tm = jrCore_get_flag('session_sync_quota_ids')) {
        // We need to set the session_sync flag for all active sessions that match the quotas that have changed
        jrUser_set_session_sync_for_quota($_tm, 'on');
    }

    // Watch for disk space updates
    if (jrCore_get_config_value('jrProfile', 'disable_profile_disk_usage', 'off') == 'off') {
        if ($_pr = jrCore_get_flag('jrprofile_media_changes')) {
            foreach (array_chunk($_pr, 25) as $k => $_ch) {
                jrCore_queue_create('jrProfile', 'update_profile_disk_usage', array('profile_ids' => $_ch), ($k * 3));
            }
            jrCore_delete_flag('jrprofile_media_changes');
        }
    }

    return $_data;
}

/**
 *
 * @param array $_data incoming data array from jrCore_save_media_file()
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrProfile_process_init_listener($_data, $_user, $_conf, $_args, $event)
{
    // disable pulse developer option
    if (jrCore_get_config_value('jrProfile', 'disable_pulse', 'off') == 'on') {
        $_js = array("var jrProfile_disable_pulse = 'on'; ");
        jrCore_create_page_element('javascript_embed', $_js);
    }
    return $_data;
}

/**
 * Remove unused form designer fields
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_verify_module_listener($_data, $_user, $_conf, $_args, $event)
{
    // Make sure some fields are removed from the designer form table
    jrCore_delete_designer_form_field('jrProfile', 'create', 'profile_quota_id');
    jrCore_delete_designer_form_field('jrProfile', 'create', 'profile_user_id');
    return $_data;
}

/**
 * Cleanup database stuff
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_repair_module_listener($_data, $_user, $_conf, $_args, $event)
{
    // Make sure quota profile counts are correct
    $_qt = jrProfile_get_quotas();
    if ($_qt && is_array($_qt)) {
        foreach ($_qt as $qid => $qname) {
            $cnt = jrCore_db_run_key_function('jrProfile', 'profile_quota_id', $qid, 'COUNT');
            if (!$cnt) {
                $cnt = 0;
            }
            jrProfile_set_quota_value('jrProfile', $qid, 'profile_count', $cnt);
        }
    }

    // Make sure all users have an entry in the profile_link table
    $_pr = jrCore_db_get_all_key_values('jrProfile', '_user_id');
    if ($_pr && is_array($_pr)) {
        $cnt = 0;
        $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
        $_pr = array_chunk($_pr, 250, true);
        foreach ($_pr as $_ch) {
            $_in = array();
            foreach ($_ch as $pid => $uid) {
                $uid = (int) $uid;
                $pid = (int) $pid;
                if ($uid > 0 && $pid > 0) {
                    $_in[] = "({$uid},{$pid})";
                }
            }
            if (count($_in) > 0) {
                $req = "INSERT IGNORE INTO {$tbl} (user_id,profile_id) VALUES " . implode(',', $_in);
                $cnt += jrCore_db_query($req, 'COUNT');
            }
        }
        if ($cnt > 0) {
            jrCore_logger('INF', "corrected " . jrCore_number_format($cnt) . " profile_link entries");
        }
    }

    return $_data;
}

/**
 * Track pulse key counters
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_db_increment_key_listener($_data, $_user, $_conf, $_args, $event)
{
    // Is this an item counter - i.e profile_jrFollower_item_count
    if (strpos($_data['key'], 'profile_') === 0 && strpos($_data['key'], '_item_count')) {
        // Is this a registered Pulse Key?
        // [key] => profile_jrAction_mention_item_count
        $_pk = jrCore_get_registered_module_features('jrProfile', 'pulse_key');
        if ($_pk && is_array($_pk)) {
            $mod = jrCore_string_field($_data['key'], 2, '_');
            if (isset($_pk[$mod]["{$_data['key']}"])) {
                // We have an item counter - increment pulse key
                $_tm = jrCore_get_flag('jrprofile_process_pulse_keys');
                if (!$_tm) {
                    $_tm = array();
                }
                foreach ($_data['id'] as $id) {
                    if (isset($_tm[$id])) {
                        $_tm[$id] = array($mod, $_pk[$mod]["{$_data['key']}"], ($_tm[$id][2] + $_data['value']));
                    }
                    else {
                        $_tm[$id] = array($mod, $_pk[$mod]["{$_data['key']}"], $_data['value']);
                    }
                }
                jrCore_set_flag('jrprofile_process_pulse_keys', $_tm);
            }
        }
    }
    return $_data;
}

/**
 * Add Quota and support for "profile_id" and "quota_id" parameters to jrCore_list
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_db_search_params_listener($_data, $_user, $_conf, $_args, $event)
{
    $admin = jrUser_is_admin();

    // Make sure we know all our inactive profiles
    $_ip = array();
    if (!$admin && jrCore_get_config_value('jrProfile', 'disable_active_check', 'off') != 'on') {
        if (!isset($_data['jrProfile_active_check']) || jrCore_checktype($_data['jrProfile_active_check'], 'is_true')) {
            $key = "jrprofile_inactive_profiles";
            if (!$_ip = jrCore_is_cached('jrProfile', $key, false, false, true, false)) {
                if ($_ip = jrCore_db_get_multiple_items_by_key('jrProfile', 'profile_active', 0, true)) {
                    $_ip = array_flip($_ip);
                }
                else {
                    $_ip = 'no_results';
                }
                jrCore_add_to_cache('jrProfile', $key, $_ip, 0, 0, false, false, false);
            }
        }
        if (!is_array($_ip)) {
            $_ip = array();
        }
    }

    // profile_id=(id)[,id][,id][,..]
    // We've been given a specific Profile ID or group of Profile IDs
    if (isset($_data['profile_id'])) {
        if (jrCore_checktype($_data['profile_id'], 'number_nz')) {
            if (!isset($_data['search']) || !is_array($_data['search'])) {
                $_data['search'] = array();
            }
            $_data['search'][] = "_profile_id = " . intval($_data['profile_id']);
        }
        elseif (strpos($_data['profile_id'], ',')) {
            if ($_tmp = explode(',', $_data['profile_id'])) {
                $_pi = array();
                foreach ($_tmp as $pid) {
                    if (is_numeric($pid)) {
                        $pid = intval($pid);
                        if ($pid > 0) {
                            $_pi[$pid] = $pid;
                        }
                    }
                }
                if (count($_pi) > 0) {
                    if (!isset($_data['search']) || !is_array($_data['search'])) {
                        $_data['search'] = array();
                    }
                    $_data['search'][] = "_profile_id in " . implode(',', $_pi);
                }
                unset($_pi);
            }
        }
    }

    // We have not been given specific profile_id's - we just need
    // to make sure that no inactive profiles are in our listings
    elseif (!$admin && count($_ip) > 0) {
        $key = '_profile_id';
        if ($_args['module'] == 'jrProfile') {
            $key                     = '_item_id';
            $_data['ignore_missing'] = true;  // Tell the DS it can ignore_missing since we KNOW all items in the DS have this key
        }
        $_data['search'][] = "{$key} not_in " . implode(',', array_keys($_ip));
    }

    // quota_id=(id)[,id][,id][,..]
    if (!empty($_data['quota_id'])) {
        $qid = false;
        if (jrCore_checktype($_data['quota_id'], 'number_nz')) {
            $qid = (int) $_data['quota_id'];
        }
        elseif (strpos($_data['quota_id'], ',')) {
            $_tmp = explode(',', $_data['quota_id']);
            if ($_tmp && is_array($_tmp)) {
                $_id = array();
                foreach ($_tmp as $id) {
                    $id = (int) $id;
                    if (jrCore_checktype($id, 'number_nz')) {
                        $_id[$id] = $id;
                    }
                }
                $qid = implode(',', $_id);
                unset($_id);
            }
        }
        if ($qid) {
            if (!isset($_data['search']) || !is_array($_data['search'])) {
                $_data['search'] = array();
            }
            if ($_args['module'] == 'jrProfile') {
                $_data['search'][] = "profile_quota_id in {$qid}";
            }
            else {
                // Get profile_id's for other modules
                if (jrCore_db_key_has_index_table('jrProfile', 'profile_quota_id')) {
                    $tbl               = jrCore_db_get_index_table_name('jrProfile', 'profile_quota_id');
                    $_data['search'][] = "_profile_id IN (SELECT `_item_id` FROM {$tbl} WHERE `value` IN({$qid}))";
                }
                else {
                    $tbl               = jrCore_db_table_name('jrProfile', 'item_key');
                    $_data['search'][] = "_profile_id IN (SELECT `_item_id` FROM {$tbl} WHERE `key` = 'profile_quota_id' AND `value` IN({$qid}))";
                }
            }
        }
    }

    // Check for quota feature search - by default we always look for the quota
    // "access" flag - i.e. "quota_jrAudio_allowed" - it must be set to "on"
    // in order for profiles to be returned
    if (!isset($_data['quota_check']) || $_data['quota_check'] === true) {
        $_sup = jrCore_get_registered_module_features('jrCore', 'quota_support');
        if (isset($_sup["{$_args['module']}"])) {
            if (!$_tm = jrCore_get_flag('jrprofile_check_quota_support')) {

                // Any info returned must be from a profile that is allowed to use this module
                $key = 'jrprofile_quota_check';
                if (!$_qt = jrCore_is_cached('jrProfile', $key, false, false, true, false)) {
                    $tbq = jrCore_db_table_name('jrProfile', 'quota_value');
                    $req = "SELECT `module`, `quota_id`, `value` FROM {$tbq} WHERE `name` = 'allowed'";
                    $_qt = jrCore_db_query($req, 'NUMERIC');
                    if (!$_qt || !is_array($_qt)) {
                        $_qt = 'no_results';
                    }
                    jrCore_add_to_cache('jrProfile', $key, $_qt, 0, 0, false, false, false);
                }

                if ($_qt && is_array($_qt)) {
                    $_tm = array(
                        'subquery_required' => array()
                    );
                    foreach ($_qt as $_entry) {
                        if (isset($_entry['value']) && $_entry['value'] == 'off') {
                            // If this item is turned off for any quota, we must run our
                            // sub query to get just those quota_id's that are allowed
                            $_tm['subquery_required']["{$_entry['module']}"] = 1;
                        }
                        else {
                            if (!isset($_tm["{$_entry['module']}"])) {
                                $_tm["{$_entry['module']}"] = array();
                            }
                            $_tm["{$_entry['module']}"]["{$_entry['quota_id']}"] = 1;
                        }
                    }
                    jrCore_set_flag('jrprofile_check_quota_support', $_tm);
                }
            }
            // Check for sub query
            if (isset($_tm['subquery_required']["{$_args['module']}"])) {
                // We have some quotas turned off - make sure we only get profiles
                // in quota_ids where this feature is actually enabled
                if (isset($_tm["{$_args['module']}"]) && is_array($_tm["{$_args['module']}"])) {
                    if (jrCore_db_key_has_index_table('jrProfile', 'profile_quota_id')) {
                        $tbl               = jrCore_db_get_index_table_name('jrProfile', 'profile_quota_id');
                        $_data['search'][] = "_profile_id in (SELECT `_item_id` FROM {$tbl} WHERE `value` IN(" . implode(',', array_keys($_tm["{$_args['module']}"])) . "))";
                    }
                    else {
                        $tbl               = jrCore_db_table_name('jrProfile', 'item_key');
                        $_data['search'][] = "_profile_id in (SELECT `_item_id` FROM {$tbl} WHERE `key` = 'profile_quota_id' AND `value` IN(" . implode(',', array_keys($_tm["{$_args['module']}"])) . "))";
                    }
                }
                else {
                    // Turned off for all quotas - set no match
                    $_data['search'] = array('_item_id = 0');
                }
            }

        }
    }
    if ($cnt = jrCore_get_flag("profile_limit_item_count_{$_args['module']}")) {
        $_data['limit'] = $cnt;
    }
    return $_data;
}

/**
 * Watch for empty jrCore_list calls on module profile indexes
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_db_search_count_query_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_urls, $_post;
    if (isset($_conf['jrProfile_show_help']) && $_conf['jrProfile_show_help'] === 'on' && !jrCore_get_flag('jrprofile_item_index_help_html')) {
        if ($_profile = jrCore_get_flag('jrprofile_item_index_view_data')) {
            if (jrProfile_is_profile_owner($_profile['_profile_id'])) {
                // does the jrCore_list have a profile_id=$profile_id call.
                $is_main = false;
                if (isset($_data['profile_id']) && $_data['profile_id'] == $_profile['_profile_id']) {
                    $is_main = true;
                }
                if (isset($_data['search']) && is_array($_data['search'])) {
                    foreach ($_data['search'] as $query) {
                        list($k, $o, $v) = explode(' ', $query, 3);
                        if ($k == '_profile_id' && $o == '=' && $v == $_profile['_profile_id']) {
                            $is_main = true;
                        }
                    }
                }
                if ($is_main) {
                    // We're in a profile module index view - do we have results?
                    if ($_args['count'] === 0) {

                        $mod                      = $_urls["{$_post['option']}"];
                        $_ln                      = jrUser_load_lang_strings();
                        $_profile['module_title'] = (isset($_ln[$mod]['menu'])) ? $_ln[$mod]['menu'] : $_urls[$mod];
                        $_profile['create_icon']  = jrCore_get_icon_html('plus', jrCore_get_skin_icon_size());

                        // No results - show help message if we have one
                        if (is_file(APP_DIR . "/modules/{$mod}/templates/item_index_help.tpl")) {
                            $html = jrCore_parse_template('item_index_help.tpl', $_profile, $mod);
                        }
                        else {
                            $html = jrCore_parse_template('item_index_default_help.tpl', $_profile, 'jrProfile');
                        }
                        jrCore_set_flag('jrprofile_item_index_help_html', $html);
                    }
                }
            }
        }
    }
    return $_data;
}

/**
 * Add Profile and Quota keys to items
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_db_search_items_listener($_data, $_user, $_conf, $_args, $event)
{
    if (($_args['module'] != 'jrUser' || isset($_args['include_jrProfile_keys']) && jrCore_checktype($_args['include_jrProfile_keys'], 'is_true')) && $_args['module'] != 'jrProfile' && isset($_data['_items'][0]) && isset($_data['_items'][0]['_profile_id'])) {

        // See if we do NOT include Profile keys with the results
        if (isset($_args['exclude_jrProfile_keys']) && $_args['exclude_jrProfile_keys'] === true) {
            return $_data;
        }

        // See if only specific keys are being requested - if none of them are profile keys
        // then we do not need to go back to the DB to get any profile/quota info
        if (isset($_args['return_keys']) && is_array($_args['return_keys']) && count($_args['return_keys']) > 0) {
            $found = false;
            foreach ($_args['return_keys'] as $key) {
                if (strpos($key, 'profile_') === 0) {
                    $found = true;
                    break;
                }
            }
            if (!$found) {
                return $_data;
            }
            unset($found);
        }

        // Add profile keys to data
        $_us = array();
        foreach ($_data['_items'] as $v) {
            if (isset($v['_profile_id']) && jrCore_checktype($v['_profile_id'], 'number_nz') && !isset($v['profile_url'])) {
                $pid       = (int) $v['_profile_id'];
                $_us[$pid] = $pid;
            }
        }
        if ($_us && count($_us) > 0) {
            $_rt = jrCore_db_get_multiple_items('jrProfile', $_us);
            if ($_rt && is_array($_rt)) {
                // We've found profile info - go though and setup by _profile_id
                $_pr = array();
                $_up = array();
                foreach ($_rt as $v) {
                    if (isset($v['_profile_id'])) {
                        $_pr["{$v['_profile_id']}"] = $v;
                        unset($_pr["{$v['_profile_id']}"]['_created']);
                        unset($_pr["{$v['_profile_id']}"]['_updated']);
                        unset($_pr["{$v['_profile_id']}"]['_item_id']);
                        $_up["{$v['_profile_id']}"] = array($v['_created'], $v['_updated']);
                    }
                }
                // Add to results
                $_qi = array();
                foreach ($_data['_items'] as $k => $v) {
                    if (isset($_pr["{$v['_profile_id']}"]) && is_array($_pr["{$v['_profile_id']}"])) {
                        $_data['_items'][$k]                    = $v + $_pr["{$v['_profile_id']}"];
                        $_data['_items'][$k]['profile_created'] = $_up["{$v['_profile_id']}"][0];
                        $_data['_items'][$k]['profile_updated'] = $_up["{$v['_profile_id']}"][1];
                        if (isset($_pr["{$v['_profile_id']}"]['profile_url'])) {
                            $_data['_items'][$k]['profile_url'] = strip_tags(rawurldecode($_pr["{$v['_profile_id']}"]['profile_url']));
                        }
                        // Bring in Quota info
                        if ((!isset($_args['exclude_jrProfile_quota_keys']) || $_args['exclude_jrProfile_quota_keys'] !== true) && isset($_pr["{$v['_profile_id']}"]['profile_quota_id'])) {
                            $qid = $_pr["{$v['_profile_id']}"]['profile_quota_id'];
                            if (!isset($_qi[$qid])) {
                                $_qi[$qid] = jrProfile_get_quota($qid);
                            }
                            if (is_array($_qi[$qid])) {
                                $_data['_items'][$k] = $_data['_items'][$k] + $_qi[$qid];
                            }
                        }
                    }
                }
                unset($_qi);
            }
        }
    }

    // See if this is a jrProfile list call where we just need to add in Quota info
    elseif ($_args['module'] == 'jrProfile' && (!isset($_args['exclude_jrProfile_quota_keys']) || $_args['exclude_jrProfile_quota_keys'] !== true) && !isset($_data['_items'][0]['quota_jrProfile_name']) && isset($_data['_items'][0]['profile_quota_id'])) {
        // Add Quota info
        $_temp = array();
        foreach ($_data['_items'] as $k => $v) {
            $qid = (int) $v['profile_quota_id'];
            if (!isset($_temp[$qid])) {
                $_temp[$qid] = jrProfile_get_quota($qid);
            }
            if (is_array($_temp[$qid])) {
                $_data['_items'][$k] = $v + $_temp[$qid];
            }
        }
        unset($_temp);
    }
    return $_data;
}

/**
 * Delete datastore entries when a profile is deleted
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_db_delete_item_listener($_data, $_user, $_conf, $_args, $event)
{
    // If a profile has been deleted, cleanup
    if (isset($_args['module']) && $_args['module'] == 'jrProfile' && isset($_data['_profile_id']) && jrCore_checktype($_data['_profile_id'], 'number_nz')) {

        // We are deleting a profile - cleanup this profiles items from our data stores
        $_mds = jrCore_get_datastore_modules();
        if ($_mds && is_array($_mds)) {

            // Get all items for profile in each datastore
            $_sc = array(
                'search'              => array(
                    "_profile_id = {$_data['_profile_id']}"
                ),
                'return_item_id_only' => true,
                'skip_triggers'       => true,
                'ignore_pending'      => true,
                'privacy_check'       => false,
                'limit'               => 100000
            );
            foreach ($_mds as $module => $prefix) {
                switch ($module) {
                    case 'jrCore':
                    case 'jrUser':
                    case 'jrProfile':
                        continue 2;
                }
                // Get all items for this profile
                $_rt = jrCore_db_search_items($module, $_sc);
                if ($_rt && is_array($_rt)) {
                    jrCore_db_delete_multiple_items($module, $_rt);
                    $tmp = (bool) jrCore_get_flag('jrProfile_delete_log_message');
                    if ($tmp === true) {
                        jrCore_logger('INF', "deleted " . count($_rt) . " {$module} entries for profile {$_data['profile_name']}");
                    }
                }
            }
        }

        // Get users associated with this profile - we're going to delete them
        // too if this profile is the only profile they are associated with
        $tmp = (bool) jrCore_get_flag('jrProfile_delete_user_check');
        if ($tmp === true) {

            // Get user accounts associated with this profile
            $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
            $req = "SELECT user_id FROM {$tbl} WHERE profile_id = '{$_data['_profile_id']}'";
            $_us = jrCore_db_query($req, 'NUMERIC');
            if ($_us && is_array($_us)) {
                $_tm = array();
                foreach ($_us as $_usr) {
                    $_tm["{$_usr['user_id']}"] = 1;
                }
                if ($_tm && count($_tm) > 0) {
                    // See how many profiles each user is linked to...
                    $req = "SELECT * FROM {$tbl} WHERE user_id IN(" . implode(',', array_keys($_tm)) . ")";
                    $_fn = jrCore_db_query($req, 'NUMERIC');
                    if ($_fn && is_array($_fn)) {
                        $_tm = array();
                        foreach ($_fn as $_usr) {
                            if (!isset($_tm["{$_usr['user_id']}"])) {
                                $_tm["{$_usr['user_id']}"] = 0;
                            }
                            $_tm["{$_usr['user_id']}"]++;
                        }
                        if ($_tm && count($_tm) > 0) {
                            foreach ($_tm as $user_id => $num_profiles) {
                                if ($num_profiles <= 1) {

                                    // This user id is only associated with THIS profile - remove
                                    if ($_ui = jrCore_db_get_item('jrUser', $user_id, true)) {

                                        // trigger delete user event
                                        jrCore_trigger_event('jrUser', 'delete_user', $_ui);

                                        // Delete User Account
                                        jrCore_db_delete_item('jrUser', $user_id);

                                        // Delete active session (if any)
                                        jrUser_session_remove($user_id);

                                        // Delete Caches
                                        jrCore_delete_all_cache_entries(null, $user_id);
                                    }

                                }
                            }
                        }
                    }
                }
            }
        }

    }
    else {
        // When we delete an item that has media we need to flag that our disk space has changed
        if (isset($_data['rb_item_media']) || strpos(json_encode($_data), '_extension')) {
            if (!$_pr = jrCore_get_flag('jrprofile_media_changes')) {
                $_pr = array();
            }
            $pid       = (int) $_data['_profile_id'];
            $_pr[$pid] = $pid;
            jrCore_set_flag('jrprofile_media_changes', $_pr);
        }
    }
    return $_data;
}

/**
 * Cleanup profile => user links when recycle bin is emptied
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_expire_recycle_bin_listener($_data, $_user, $_conf, $_args, $event)
{
    if (isset($_data['_items']) && is_array($_data['_items'])) {
        // The items array has all the items that are being purged from the recycle bin
        $_dl = array();
        foreach ($_data['_items'] as $v) {
            if ($v['module'] == 'jrProfile') {
                $_dl[] = (int) $v['item_id'];
            }
        }
        if (count($_dl) > 0) {

            // Cleanup Profile Link
            $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
            $req = "DELETE FROM {$tbl} WHERE profile_id IN(" . implode(',', $_dl) . ')';
            jrCore_db_query($req);

            // Cleanup any other Recycle Bin items that belong to these profiles
            $tbl = jrCore_db_table_name('jrCore', 'recycle');
            $req = "DELETE FROM {$tbl} WHERE r_profile_id IN(" . implode(',', $_dl) . ')';
            jrCore_db_query($req);

            // Make sure all media files for these profiles are removed
            foreach ($_dl as $pid) {
                $dir = jrCore_get_media_directory($pid);
                jrCore_delete_dir_contents($dir, false);
                rmdir($dir);
            }

        }
        unset($_dl);
    }
    return $_data;
}

/**
 * Listens for when new users sign up so it can create their profile
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_signup_created_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    // User account is created - create profile
    if (jrUser_is_admin() && isset($_post['create_profile']) && $_post['create_profile'] == 'off') {
        // We're not creating one on purpose
        return $_data;
    }

    // We need to check and see if a quota_id came in via $_post ($_data).  If
    // it did, then we are going to check to be sure that quota_id allows sign ups -
    // if it does, then we use that quota_id - else we use the default.
    $qid = (isset($_conf['jrProfile_default_quota_id']) && jrCore_checktype($_conf['jrProfile_default_quota_id'], 'number_nz')) ? intval($_conf['jrProfile_default_quota_id']) : jrProfile_get_default_signup_quota();
    if (isset($_data['quota_id']) && jrCore_checktype($_data['quota_id'], 'number_nz')) {
        // Looks like a different quota_id is being requested - validate
        $_qt = jrProfile_get_quota($_data['quota_id']);
        if ($_qt && is_array($_qt)) {
            // Make sure sign ups are allowed
            if (!isset($_qt['quota_jrUser_allow_signups']) || $_qt['quota_jrUser_allow_signups'] != 'on') {
                jrCore_db_delete_item('jrUser', $_args['_user_id']);
                jrCore_logger('MAJ', "attempted signup to a Quota that does not allow signups (quota_id: {$_data['quota_id']}");
                jrCore_set_form_notice('error', 'An error was encountered creating your profile - please try again');
                jrCore_form_result();
            }
            $qid = intval($_data['quota_id']);
        }
    }
    else {
        $_qt = jrProfile_get_quota($qid);
        if (!$_qt || !is_array($_qt)) {
            jrCore_logger('CRI', "profile default quota_id has been configured with an invalid quota!");
            // Do we have any quota that allows sign ups?
            $_sq = jrProfile_get_signup_quotas();
            if (!$_sq) {
                jrCore_db_delete_item('jrUser', $_args['_user_id']);
                jrCore_set_form_notice('error', 'An error was encountered creating your profile - please try again (2)');
                jrCore_form_result();
            }
            $_sq = array_keys($_sq);
            $qid = (int) reset($_sq);
            jrCore_logger('INF', "using signup quota_id {$qid} for new profile");
        }

    }

    // data to save to datastore
    $_prof = array(
        'profile_name'     => $_args['user_name'],
        'profile_url'      => jrCore_url_string($_args['user_name']),
        'profile_quota_id' => $qid,
        'profile_active'   => (isset($_data['user_active']) && intval($_data['user_active']) === 1) ? 1 : 0,
        'profile_private'  => (isset($_qt['quota_jrProfile_default_privacy']) && jrCore_checktype($_qt['quota_jrProfile_default_privacy'], 'number_nn')) ? $_qt['quota_jrProfile_default_privacy'] : 1
    );
    $pid   = jrCore_db_create_item('jrProfile', $_prof);
    if (!$pid || !jrCore_checktype($pid, 'number_nz')) {
        jrCore_db_delete_item('jrUser', $_args['_user_id']);
        jrCore_set_form_notice('error', 'An error was encountered creating your profile - please try again');
        jrCore_form_result();
    }

    // Update with new profile id
    $_temp = array();
    $_core = array(
        '_user_id'    => $_args['_user_id'],
        '_profile_id' => $pid
    );
    jrCore_db_update_item('jrProfile', $pid, $_temp, $_core);

    // Update User info with the proper _profile_id
    unset($_core['_user_id']);
    jrCore_db_update_item('jrUser', $_args['_user_id'], $_temp, $_core);

    // Profile is created - add user_id -> profile_id into link table
    jrProfile_create_user_link($_args['_user_id'], $pid);

    // Make sure profile media directory is created
    jrCore_create_media_directory($pid);

    // Update the profile_count for the quota this profile just signed up
    jrProfile_increment_quota_profile_count($qid);

    // Save off Quota Info - this is used in the User Module to determine the Signup Method.
    if (!isset($_qt)) {
        $_qt = jrProfile_get_quota($qid);
    }
    $_data['signup_method'] = (isset($_qt['quota_jrUser_signup_method'])) ? $_qt['quota_jrUser_signup_method'] : 'email';

    return $_data;
}

/**
 * Listens for new user activation so it can activate the associated user profile
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrProfile_signup_activated_listener($_data, $_user, $_conf, $_args, $event)
{
    // Update Profile so it is active
    if (isset($_data['_profile_id']) && jrCore_checktype($_data['_profile_id'], 'number_nz')) {

        // Make sure profile is active
        jrCore_db_update_item('jrProfile', $_data['_profile_id'], array('profile_active' => '1'));
        jrCore_delete_cache('jrProfile', 'jrprofile_inactive_profiles', false, false);
        $_data['profile_active'] = 1;

    }
    return $_data;
}

