<?php
 /**
 * Jamroom Products module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrProduct_meta()
{
    return array(
        'name'        => 'Products',
        'url'         => 'product',
        'version'     => '1.2.2',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Supports sales of physical merchandise from a profile',
        'category'    => 'profiles',
        'license'     => 'jcl'
    );
}

/**
 * init
 */
function jrProduct_init()
{
    // Tools
    global $_mods;
    if (isset($_mods['jrStore'])) {
        jrCore_register_module_feature('jrCore', 'tool_view', 'jrProduct', 'import', array('Import Store Items', 'Import existing store items from the Merchandise module'));
    }

    // Core Features
    jrCore_register_module_feature('jrCore', 'quota_support', 'jrProduct', 'off');
    jrCore_register_module_feature('jrCore', 'pending_support', 'jrProduct', 'on');
    jrCore_register_module_feature('jrCore', 'max_item_support', 'jrProduct', true);
    jrCore_register_module_feature('jrCore', 'item_order_support', 'jrProduct', 'on');
    jrCore_register_module_feature('jrCore', 'action_support', 'jrProduct', 'create', 'item_action.tpl');
    jrCore_register_module_feature('jrCore', 'action_support', 'jrProduct', 'update', 'item_action.tpl');

    // Profile tabs
    jrCore_register_module_feature('jrProfile', 'profile_tab', 'jrProduct', 'default', 19); // 19 = 'products'
    jrCore_register_module_feature('jrProfile', 'profile_tab', 'jrProduct', 'categories', 26); // 26 = 'product categories'
    jrCore_register_module_feature('jrProfile', 'profile_tab', 'jrProduct', 'shipping_tracker', 48); // 48 = 'shipping tracker'

    // We have some JS and small custom CSS for our update page
    jrCore_register_module_feature('jrCore', 'css', 'jrProduct', 'jrProduct.css');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrProduct', 'jrProduct.js');

    // Let Payment module know we support shipping and item quantities
    jrCore_register_module_feature('jrPayment', 'shipping_support', 'jrProduct', true);
    jrCore_register_module_feature('jrPayment', 'quantity_support', 'jrProduct', true);

    // Allow admin to customize our forms
    jrCore_register_module_feature('jrCore', 'designer_form', 'jrProduct', 'create');
    jrCore_register_module_feature('jrCore', 'designer_form', 'jrProduct', 'update');

    // Recycle Bin support
    jrCore_register_module_feature('jrCore', 'recycle_bin_profile_id_table', 'jrProduct', 'category', 'cat_profile_id');

    // Listeners
    jrCore_register_event_listener('jrCore', 'db_get_item', 'jrProduct_db_get_item_listener');

    // Add item price field
    jrCore_register_event_listener('jrPayment', 'get_item_price_field', 'jrProduct_get_item_price_field_listener');
    jrCore_register_event_listener('jrPayment', 'cart_shipping', 'jrProduct_cart_shipping_listener');
    jrCore_register_event_listener('jrPayment', 'cart_update_quantity', 'jrProduct_cart_update_quantity_listener');
    jrCore_register_event_listener('jrPayment', 'purchase_item', 'jrProduct_purchase_item_listener');
    jrCore_register_event_listener('jrPayment', 'purchase_entry', 'jrProduct_purchase_entry_listener');

    // Add item price field - PayPal Buy It Now
    jrCore_register_event_listener('jrPayPal', 'add_paypal_price_field', 'jrProduct_add_price_field_listener');

    // We have fields that can be searched
    jrCore_register_module_feature('jrSearch', 'search_fields', 'jrProduct', 'product_title,product_category,product_description,product_cat_field_1,product_cat_field_2,product_cat_field_3,product_cat_field_4,product_cat_field_5', 19);

    return true;
}

//---------------------
// EVENT LISTENERS
//---------------------

/**
 * Return purchase fields for price
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrProduct_add_price_field_listener($_data, $_user, $_conf, $_args, $event)
{
    // Module/View => File Field
    $_data['jrProduct/create'] = 'product';
    $_data['jrProduct/update'] = 'product';
    return $_data;
}

/**
 * Check that item is still available
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrProduct_cart_update_quantity_listener($_data, $_user, $_conf, $_args, $event)
{
    if (isset($_data['cart_new_quantity']) && $_data['cart_new_quantity'] > 0) {
        // Requesting a new quantity - are there enough left?
        if ($qty = jrCore_db_get_item_key('jrProduct', $_data['cart_item_id'], 'product_qty')) {
            if (jrCore_checktype($qty, 'number_nn') && $qty < $_data['cart_new_quantity']) {
                $_data['cart_new_quantity'] = $qty;
            }
        }
    }
    return $_data;
}

/**
 * Decrement number of available items
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrProduct_purchase_item_listener($_data, $_user, $_conf, $_args, $event)
{
    if (!empty($_data['cart_module']) && $_data['cart_module'] == 'jrProduct') {
        jrCore_db_decrement_key('jrProduct', $_data['cart_item_id'], 'product_qty', $_data['cart_quantity']);
        jrProfile_reset_cache($_data['_profile_id'], 'jrProduct');
    }
    return $_data;
}

/**
 * Add contact button for buyer to contact seller
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrProduct_purchase_entry_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_conf;
    $_data[5]['title'] = '&nbsp;';
    if (jrCore_module_is_active('jrPrivateNote') && isset($_args['r_item_data']['_user_id'])) {
        $_ln               = jrUser_load_lang_strings();
        $url               = jrCore_get_module_url('jrPrivateNote');
        $uid               = (int) $_args['r_item_data']['_user_id'];
        $_data[5]['title'] = jrCore_page_button('contact', $_ln['jrProduct'][45], "jrCore_window_location('{$_conf['jrCore_base_url']}/{$url}/new/user_id={$uid}');");
    }
    return $_data;
}

/**
 * Get shipping for a product
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return int
 */
function jrProduct_cart_shipping_listener($_data, $_user, $_conf, $_args, $event)
{
    $shipping = 0;
    if (isset($_args['product_item_shipping']) && $_args['product_item_shipping'] > 0) {
        $shipping = jrPayment_price_to_cents($_args['product_item_shipping']);
    }
    return $shipping;
}

/**
 * Get item price field for product form
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return string
 */
function jrProduct_get_item_price_field_listener($_data, $_user, $_conf, $_args, $event)
{
    switch ($_args['option']) {
        case 'jrProduct/create':
        case 'jrProduct/update':
            return 'product_item_price';
    }
    return false;
}

/**
 * Expand item images and category into useful array for templates
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrProduct_db_get_item_listener($_data, $_user, $_conf, $_args, $event)
{
    if (jrCore_is_view_request() && $_args['module'] == 'jrProduct') {
        foreach ($_data as $k => $v) {
            if (strpos(' ' . $k, 'product_image') && strpos($k, '_time')) {
                if (!isset($_data['_product_images'])) {
                    $_data['_product_images'] = array();
                }
                $_data['_product_images'][] = substr($k, 0, strpos($k, '_time'));
            }
            elseif ($k == 'product_category_id' && jrCore_checktype($v, 'number_nz')) {
                $tbl = jrCore_db_table_name('jrProduct', 'category');
                $req = "SELECT * FROM {$tbl} WHERE cat_id = '{$v}' LIMIT 1";
                $_rt = jrCore_db_query($req, 'SINGLE');
                if ($_rt && is_array($_rt)) {
                    $_fields = json_decode(base64_decode($_rt['cat_field']), true);
                    $murl    = jrCore_db_get_prefix('jrProduct');
                    for ($i = 1; $i <= 5; $i++) {
                        if (isset($_fields["cat_field_type_{$i}"]) && $_fields["cat_field_type_{$i}"] != 'none') {
                            $_data['_product_cat_fields'][$i]['label'] = $_fields["cat_field_label_{$i}"];
                            $_data['_product_cat_fields'][$i]['type']  = $_fields["cat_field_type_{$i}"];
                            $_data['_product_cat_fields'][$i]['value'] = $_data["{$murl}_cat_field_{$i}"];
                        }
                    }
                }
            }
        }
        $_data['product_image_count'] = (isset($_data['_product_images'])) ? count($_data['_product_images']) : 0;
    }
    return $_data;
}

//---------------------
// FUNCTIONS
//---------------------

/**
 * Get profile product categories
 * @return array|false
 */
function jrProduct_get_profile_categories()
{
    global $_user;
    $tbl = jrCore_db_table_name('jrProduct', 'category');
    $req = "SELECT * FROM {$tbl} WHERE cat_profile_id = '{$_user['user_active_profile_id']}'";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if ($_rt && is_array($_rt) && count($_rt) > 0) {
        $_out = array();
        foreach ($_rt as $rt) {
            $_out["{$rt['cat_id']}"] = $rt['cat_title'];
        }
        return $_out;
    }
    return false;
}

/**
 * Get a category by profile_id and category URL
 * @param int $profile_id
 * @param string $url
 * @return array|false
 */
function jrProduct_get_category_by_url($profile_id, $url)
{
    $pid = (int) $profile_id;
    $url = jrCore_db_escape($url);
    $tbl = jrCore_db_table_name('jrProduct', 'category');
    $req = "SELECT * FROM {$tbl} WHERE cat_title_url = '{$url}' AND cat_profile_id = {$pid} LIMIT 1";
    return jrCore_db_query($req, 'SINGLE');
}

/**
 * Get user name from purchase info
 * @param array $_data
 * @return mixed
 */
function jrProduct_get_user_name($_data)
{
    if (isset($_data['_payment']['txn_customer_name'])) {
        return $_data['_payment']['txn_customer_name'];
    }
    return $_data['_user']['user_name'];
}

/**
 * Construct a user address based on values in user account
 * @param array $_usr user info
 * @return string
 */
function jrProduct_get_user_address($_usr)
{
    $addr = '';

    // TXN keys
    $_kys = array(
        'txn_address_line1'   => '<br>',
        'txn_address_line2'   => '<br>',
        'txn_address_city'    => ', ',
        'txn_address_state'   => ' ',
        'txn_address_zip'     => '<br>',
        'txn_address_country' => ''
    );
    foreach ($_kys as $k => $s) {
        if (!empty($_usr[$k])) {
            $addr .= $_usr[$k] . $s;
        }
    }
    if (strlen($addr) > 0) {
        return $addr;
    }

    // Foxycart keys
    $_kys = array(
        'user_foxycart_address1'    => '<br>',
        'user_foxycart_address2'    => '<br>',
        'user_foxycart_city'        => ', ',
        'user_foxycart_state'       => ' ',
        'user_foxycart_postal_code' => '<br>',
        'user_foxycart_country'     => ''
    );
    foreach ($_kys as $k => $s) {
        if (!empty($_usr[$k])) {
            $addr .= $_usr[$k] . $s;
        }
    }

    return $addr;
}

/**
 * Get the Carrier name from the carrier tracking URL
 * @param string $url
 * @return bool|string
 */
function jrProduct_get_carrier_from_tracking_url($url)
{
    if (strpos($url, 'ups.com/')) {
        return 'UPS';
    }
    if (strpos($url, 'usps.com/')) {
        return 'USPS';
    }
    if (strpos($url, 'fedex.com/')) {
        return 'FedEx';
    }
    if (strpos($url, 'ontrac.com/')) {
        return 'OnTrac';
    }
    if (strpos($url, 'dhl.com/')) {
        return 'DHL';
    }
    return false;
}

// Copyright (c) 2012, 2017, Darkain Multimedia
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this
//   list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright notice,
//   this list of conditions and the following disclaimer in the documentation
//   and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
// ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

/**
 * Get tracking URL from a tracking ID number
 * @see https://github.com/darkain/php-tracking-urls
 * @param $tracking_number
 * @return bool|string
 */
function jrProduct_get_tracking_url($tracking_number)
{
    if (empty($tracking_number)) {
        return false;
    }
    if (!is_string($tracking_number) && !is_int($tracking_number)) {
        return false;
    }

    $tracking_urls = array(

        // UPS
        array(
            'url' => 'http://wwwapps.ups.com/WebTracking/processInputRequest?TypeOfInquiryNumber=T&InquiryNumber1=',
            'reg' => '/\b(1Z ?[0-9A-Z]{3} ?[0-9A-Z]{3} ?[0-9A-Z]{2} ?[0-9A-Z]{4} ?[0-9A-Z]{3} ?[0-9A-Z]|T\d{3} ?\d{4} ?\d{3})\b/i'
        ),

        // USPS
        array(
            'url' => 'https://tools.usps.com/go/TrackConfirmAction?qtc_tLabels1=',
            'reg' => '/\b((420 ?\d{5} ?)?(91|92|93|94|01|03|04|70|23|13)\d{2} ?\d{4} ?\d{4} ?\d{4} ?\d{4}( ?\d{2,6})?)\b/i'
        ),

        // USPS
        array(
            'url' => 'https://tools.usps.com/go/TrackConfirmAction?qtc_tLabels1=',
            'reg' => '/\b((M|P[A-Z]?|D[C-Z]|LK|E[A-C]|V[A-Z]|R[A-Z]|CP|CJ|LC|LJ) ?\d{3} ?\d{3} ?\d{3} ?[A-Z]?[A-Z]?)\b/i'
        ),

        // USPS
        array(
            'url' => 'https://tools.usps.com/go/TrackConfirmAction?qtc_tLabels1=',
            'reg' => '/\b(82 ?\d{3} ?\d{3} ?\d{2})\b/i'
        ),

        // USPS
        array(
            'url' => 'http://wwwapps.ups.com/WebTracking/processInputRequest?TypeOfInquiryNumber=T&InquiryNumber1=',
            'reg' => '/^[0-9]{18}$/'
        ),

        // FEDEX
        array(
            'url' => 'http://www.fedex.com/Tracking?language=english&cntry_code=us&tracknumbers=',
            'reg' => '/\b(((96\d\d|6\d)\d{3} ?\d{4}|96\d{2}|\d{4}) ?\d{4} ?\d{4}( ?\d{3})?)\b/i'
        ),

        // FEDEX
        array(
            'url' => 'http://www.fedex.com/Tracking?language=english&cntry_code=us&tracknumbers=',
            'reg' => '/^[0-9]{20}$/'
        ),

        // ONTRAC
        array(
            'url' => 'http://www.ontrac.com/trackres.asp?tracking_number=',
            'reg' => '/\b(C\d{14})\b/i'
        ),

        // ONTRAC 2
        array(
            'url' => 'http://www.ontrac.com/trackres.asp?tracking_number=',
            'reg' => '/\b^D[0-9]{14}\b/i'
        ),

        // DHL
        array(
            'url' => 'http://www.dhl.com/content/g0/en/express/tracking.shtml?brand=DHL&AWB=',
            'reg' => '/\b(\d{4}[- ]?\d{4}[- ]?\d{2}|\d{3}[- ]?\d{8}|[A-Z]{3}\d{7})\b/i'
        )
    );

    //TEST EACH POSSIBLE COMBINATION
    foreach ($tracking_urls as $item) {
        $match = array();
        preg_match($item['reg'], $tracking_number, $match);
        if (count($match)) {
            return $item['url'] . preg_replace('/\s/', '', strtoupper($match[0]));
        }
    }

    // TRIM LEADING ZEROES AND TRY AGAIN
    // https://github.com/darkain/php-tracking-urls/issues/4
    if (substr($tracking_number, 0, 1) === '0') {
        return jrProduct_get_tracking_url(ltrim($tracking_number, '0'));
    }

    //NO MATCH FOUND, RETURN FALSE
    return false;
}
