<?php
 /**
 * Jamroom Private Notes module
 *
 * copyright 2021 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrPrivateNote_meta()
{
    return array(
        'name'        => 'Private Notes',
        'url'         => 'note',
        'version'     => '1.6.2',
        'developer'   => 'The Jamroom Network, &copy;' . strftime('%Y'),
        'description' => 'Users can send and receive private notes',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/286/private-notes',
        'category'    => 'communication',
        'requires'    => 'jrCore:6.4.0b1,jrUser:2.9.0b1',
        'license'     => 'jcl'
    );
}

/**
 * init
 */
function jrPrivateNote_init()
{
    // Register event triggers
    jrCore_register_event_trigger('jrPrivateNote', 'inbox_notes', 'Fired with message content when viewing the Notes inbox');

    // Our custom JS
    jrCore_register_module_feature('jrCore', 'javascript', 'jrPrivateNote', 'jrPrivateNote.js');

    // Core support
    $_tmp = array(
        'label' => 'Allowed Access',
        'help'  => 'If checked, User Accounts associated with Profiles in this quota will be allowed to send and receive Private Notes.'
    );
    jrCore_register_module_feature('jrCore', 'quota_support', 'jrPrivateNote', 'on', $_tmp);

    // We have some small custom CSS for our notes
    jrCore_register_module_feature('jrCore', 'css', 'jrPrivateNote', 'jrPrivateNote.css');

    // Skin menu link to Private notes
    $_tmp = array(
        'group'    => 'user',
        'label'    => 32, // 'private notes'
        'url'      => 'notes',
        'function' => 'jrPrivateNote_unread_count'
    );
    jrCore_register_module_feature('jrCore', 'skin_menu_item', 'jrPrivateNote', 'private_note_link', $_tmp);

    // We can notify the user when they receive a new private note
    $_tmp = array(
        'label'      => 23, // 'private note received'
        'help'       => 35, // 'Do you want to be notified by email when you receive a new private note?',
        'email_only' => true
    );
    jrCore_register_module_feature('jrUser', 'notification', 'jrPrivateNote', 'note_received', $_tmp);

    // Recycle Bin support
    jrCore_register_module_feature('jrCore', 'recycle_bin_user_id_table', 'jrPrivateNote', 'thread', 'thread_from_user_id');
    jrCore_register_module_feature('jrCore', 'recycle_bin_user_id_table', 'jrPrivateNote', 'note', 'note_from_user_id');

    // We add a new "Private Note" notification plugin
    jrCore_register_module_feature('jrUser', 'notification_plugin', 'jrPrivateNote', 'note', 'jrPrivateNote_send_private_note');
    jrCore_register_event_listener('jrUser', 'notification_plugins', 'jrPrivateNote_notification_plugins_listener');
    jrCore_register_event_listener('jrUser', 'notification_event', 'jrPrivateNote_notification_event_listener');
    return true;
}

//--------------------
// Event Listeners
//--------------------

/**
 * Add "note" notification plugin
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrPrivateNote_notification_plugins_listener($_data, $_user, $_conf, $_args, $event)
{
    // @note: $_args is the active user info - do not use global $_user!
    if (isset($_args['quota_jrPrivateNote_allowed']) && $_args['quota_jrPrivateNote_allowed'] == 'on') {
        $_lang         = jrUser_load_lang_strings();
        $_data['note'] = $_lang['jrPrivateNote'][53];
    }
    return $_data;
}

/**
 * Disable "note" as an option for "Private Note Received"
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrPrivateNote_notification_event_listener($_data, $_user, $_conf, $_args, $event)
{
    if ($_args['name'] == 'note_received') {
        unset($_data['note']);
    }
    return $_data;
}

//--------------------
// Functions
//--------------------

/**
 * Get list of users a user is allowed to send to (for live search)
 * @param string $search Search String for matching users
 * @param int $limit limit returned results to top X entries (ordered by user_name asc)
 * @return array|false
 */
function jrPrivateNote_search_users($search, $limit = 0)
{
    global $_user;
    $id = (int) $_user['_user_id'];
    $ss = jrCore_db_escape(jrCore_str_to_lower($search)) . '%';

    if (jrUser_is_admin() || !jrCore_module_is_active('jrFollower') || jrProfile_get_quota_value($_user, 'jrPrivateNote', 'followers_only', 'on') == 'off') {
        // We can send to anyone:
        // 1) User is an admin user
        // 2) Followers module is not active
        // 3) Followers Only quota setting is OFF
        $tbl = jrCore_db_table_name('jrUser', 'item_key');
        $req = "SELECT `_item_id` AS i, `value` AS v FROM {$tbl} WHERE `key` = 'user_name' AND `value` LIKE '{$ss}'";
        if (jrCore_checktype($limit, 'number_nz')) {
            $req .= " ORDER BY `value` ASC LIMIT {$limit}";
        }
        $_su = jrCore_db_query($req, 'i', false, 'v');
        if ($_su && is_array($_su)) {
            if (isset($_su[$id])) {
                unset($_su[$id]);
            }
            return $_su;
        }
    }

    // Fall through / We can only send a note to a follower -
    // but only if Private Notes are enabled in that user's quota
    // $_fl = array( user_id => user_name )
    $_fl = jrFollower_get_users_following($_user['user_active_profile_id']);

    $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
    $req = "SELECT `quota_id`, `value` FROM {$tbl} WHERE `module` = 'jrPrivateNote' AND `name` = 'allowed' AND `value` = 'off'";
    $_qa = jrCore_db_query($req, 'quota_id', false, 'value');
    if (!$_qa || !is_array($_qa) || count($_qa) === 0) {

        // All Quotas have access to Private Notes - no need to restrict by quota
        if ($_fl && is_array($_fl) && count($_fl) > 0) {
            $tbl = jrCore_db_table_name('jrUser', 'item_key');
            $req = "SELECT `_item_id` AS i, `value` AS v FROM {$tbl}
                     WHERE `key` = 'user_name'
                       AND (`_item_id` IN(" . implode(',', array_keys($_fl)) . ") OR `_item_id` IN(
                          SELECT `_item_id` FROM {$tbl} WHERE `key` = 'user_group' AND `value` IN('master','admin')
                       ))
                       AND `value` LIKE '{$ss}'";
        }
        // No followers - admins only
        else {
            $tbl = jrCore_db_table_name('jrUser', 'item_key');
            $req = "SELECT `_item_id` AS i, `value` AS v FROM {$tbl}
                     WHERE `key` = 'user_name'
                       AND `_item_id` IN( SELECT `_item_id` FROM {$tbl} WHERE `key` = 'user_group' AND `value` IN('master','admin') )
                       AND `value` LIKE '{$ss}'";
        }
        if (jrCore_checktype($limit, 'number_nz')) {
            $req .= " ORDER BY `value` ASC LIMIT {$limit}";
        }
        $_su = jrCore_db_query($req, 'i', false, 'v');
        if ($_su && is_array($_su)) {
            if (isset($_su[$id])) {
                unset($_su[$id]);
            }
            return $_su;
        }
        // No users match
        return false;
    }

    // Fall through - We have quotas that do not have PrivateNotes enabled - exclude those users

    // Step 1:
    // Get user_name's that actually match.  This allows us to bypass the quota checking
    // if we don't actually get any matching users
    $tbl = jrCore_db_table_name('jrUser', 'item_key');
    $req = "SELECT `_item_id` AS i, `value` AS v FROM {$tbl} WHERE `key` = 'user_name' AND `value` LIKE '{$ss}'";
    $_su = jrCore_db_query($req, 'i', false, 'v');
    if ($_su && is_array($_su)) {
        if (isset($_su[$id])) {
            unset($_su[$id]);
        }
        if (count($_su) > 0) {
            // Step 2:
            // Now we know what user_id's matched our search - prune out user_id's that
            // are in quotas that do NOT allow Private Notes
            $tb1 = jrCore_db_table_name('jrUser', 'item_key');
            $tb2 = jrCore_db_table_name('jrProfile', 'item_key');
            $req = "SELECT u.`_item_id` AS i, u.`value` AS v FROM {$tb1} u
                     WHERE u.`key` = 'user_name'
                       AND u.`_item_id` IN(" . implode(',', array_keys($_su)) . ")
                       AND u.`_profile_id` IN( SELECT p.`_item_id` FROM {$tb2} p WHERE p.`key` = 'profile_quota_id' AND p.`value` NOT IN(" . implode(',', array_keys($_qa)) . ") )";
            if (jrCore_checktype($limit, 'number_nz')) {
                $req .= " ORDER BY u.`value` ASC LIMIT {$limit}";
            }
            $_su = jrCore_db_query($req, 'i', false, 'v');
            if ($_su && is_array($_su)) {
                return $_su;
            }
        }
    }
    // No matches
    return false;
}

/**
 * Get number of unread Private Notes for a user
 * @param array $_conf Global Config
 * @param array $_user User Information
 * @return int Number of unread Private Notes
 */
function jrPrivateNote_unread_count($_conf, $_user)
{
    $cnt = jrCore_db_get_item_key('jrUser', $_user['_user_id'], 'user_jrPrivateNote_unread_count');
    if ($cnt && $cnt > 0) {
        return (int) $cnt;
    }
    return true;
}

/**
 * Decrement a user's unread note count
 * @param int $user_id User ID
 * @param int $count Number to decrement by
 * @return bool
 */
function jrPrivateNote_decrement_unread_count($user_id, $count = 1)
{
    // Update the recipients unread note count
    $cnt = (isset($count) && is_numeric($count)) ? (int) $count : 1;
    return jrCore_db_decrement_key('jrUser', $user_id, 'user_jrPrivateNote_unread_count', $cnt);
}

/**
 * Increment a user's unread note count by one
 * @param int $user_id User ID
 * @param int $count Number to increment by
 * @return bool
 */
function jrPrivateNote_increment_unread_count($user_id, $count = 1)
{
    // Update the recipients unread note count
    $cnt = (isset($count) && is_numeric($count)) ? (int) $count : 1;
    return jrCore_db_increment_key('jrUser', $user_id, 'user_jrPrivateNote_unread_count', $cnt);
}

/**
 * User Notifications plugin wrapper
 * @param array $_data notification data
 * @return bool|int
 */
function jrPrivateNote_send_private_note($_data)
{
    return jrPrivateNote_send_note($_data['_user']['_user_id'], $_data['from_user_id'], $_data['subject'], $_data['message']);
}

/**
 * Send a Private Note to a User ID
 * @param int $to_user_id User_ID to send private note to
 * @param int $from_user_id User_ID private note is from
 * @param string $subject Subject of Private Note
 * @param string $message Message Body of Private Note
 * @param int $thread_id Thread ID this note belongs to
 * @return int|false
 */
function jrPrivateNote_send_note($to_user_id, $from_user_id, $subject, $message, $thread_id = 0)
{
    global $_conf, $_user;
    // Make sure receiving user is valid
    $_us = jrCore_db_get_item('jrUser', $to_user_id, true);
    if (!$_us || !is_array($_us)) {
        return false;
    }
    // See if this is a system note (from user id = 0)
    $name = false;
    if ($from_user_id == '0') {
        // See if we have a SYSTEM USER ID
        if (isset($_conf['jrPrivateNote_system_user_id']) && jrCore_checktype($_conf['jrPrivateNote_system_user_id'], 'number_nz')) {
            $name = jrCore_db_get_item_key('jrUser', $_conf['jrPrivateNote_system_user_id'], 'user_name');
        }
        if (!$name || strlen($name) === 0) {
            $name = $_conf['jrCore_system_name'];
        }
        $msg = $message;
    }
    else {
        $name = $_user['user_name'];
        $msg  = $message;
    }
    // If this is the first note in a thread...
    $sub = jrCore_strip_emoji(jrCore_strip_html($subject));
    if (strlen($sub) > 254) {
        $sub = substr($sub, 0, 251) . '...';
    }
    $sub = jrCore_db_escape($sub);
    $tbl = jrCore_db_table_name('jrPrivateNote', 'thread');
    if ($thread_id === 0) {
        $req = "INSERT INTO {$tbl} (thread_created,thread_updated,thread_from_user_id,thread_to_user_id,thread_from_deleted,thread_to_deleted,thread_updated_user_id,thread_replies,thread_subject)
                VALUES (UNIX_TIMESTAMP(),UNIX_TIMESTAMP(),'{$from_user_id}','{$to_user_id}','0','0','{$from_user_id}','0','{$sub}')";
        $tid = jrCore_db_query($req, 'INSERT_ID');
        if (!$tid || !jrCore_checktype($tid, 'number_nz')) {
            return false;
        }
        $thread_id = $tid;
    }
    else {
        $req = "UPDATE {$tbl} SET thread_from_deleted = '0', thread_to_deleted = '0', thread_updated = UNIX_TIMESTAMP(), thread_updated_user_id = '{$from_user_id}', thread_user_seen = '{$from_user_id}', thread_replies = (thread_replies + 1)
                 WHERE thread_id = '{$thread_id}' LIMIT 1";
        $cnt = jrCore_db_query($req, 'COUNT');
        if ($cnt !== 1) {
            return false;
        }
    }

    // Okay thread is updated - insert new note
    $tbl = jrCore_db_table_name('jrPrivateNote', 'note');
    $req = "INSERT INTO {$tbl} (note_thread_id,note_created,note_from_user_id,note_to_seen,note_subject,note_message)
            VALUES ('{$thread_id}',UNIX_TIMESTAMP(),'{$from_user_id}','0','{$sub}','" . jrCore_db_escape(jrCore_strip_emoji($msg)) . "')";
    $nid = jrCore_db_query($req, 'INSERT_ID');
    if (!$nid || !jrCore_checktype($nid, 'number_nz')) {
        return false;
    }
    jrCore_set_flag('jrprivatenote_created_note_id', $nid);

    // Update the recipients unread note count
    jrPrivateNote_increment_unread_count($to_user_id);

    // Notify User
    if (isset($_us['user_email']) && jrCore_checktype($_us['user_email'], 'email') && (!isset($_us["user_jrPrivateNote_note_received_notifications"]) || $_us["user_jrPrivateNote_note_received_notifications"] == 'email')) {
        $url = jrCore_get_module_url('jrPrivateNote');
        $_rp = array(
            'system_name'  => $_conf['jrCore_system_name'],
            'system_url'   => $_conf['jrCore_base_url'],
            'note_subject' => $sub,
            'note_text'    => $msg,
            'note_from'    => $name,
            'note_url'     => "{$_conf['jrCore_base_url']}/{$url}/show/{$thread_id}#last"
        );
        list($sub, $msg) = jrCore_parse_email_templates('jrPrivateNote', 'note', $_rp);
        // DO NOT user jrUser_notify() here
        jrCore_send_email($_us['user_email'], $sub, $msg);
    }
    return $thread_id;
}
