<?php
 /**
 * Jamroom Newsletters module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Jamroom file is LICENSED SOFTWARE, and cannot be redistributed.
 *
 * This Source Code is subject to the terms of the Jamroom Network
 * Commercial License -  please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrNewsLetter_meta()
{
    return array(
        'name'        => 'Newsletters',
        'url'         => 'letter',
        'version'     => '2.5.4',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Send a targetted email newsletter to your users',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/283/email-newsletters',
        'category'    => 'communication',
        'requires'    => 'jrCore:6.5.12,jrMailer:2.3.0',
        'license'     => 'jcl'
    );
}

/**
 * init
 */
function jrNewsLetter_init()
{
    // Register our custom CSS and JS
    jrCore_register_module_feature('jrCore', 'css', 'jrNewsLetter', 'jrNewsLetter.css');
    jrCore_register_module_feature('jrCore', 'javascript', 'jrNewsLetter', 'jrNewsLetter.js', 'admin');

    // Register our newsletter tools
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrNewsLetter', 'compose', array('Create Newsletter', 'Create a new Email Newsletter'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrNewsLetter', 'browse', array('Newsletter Browser', 'Browse previously created newsletters'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrNewsLetter', 'template_browser', array('Newsletter Templates', 'Browse previously created newsletter templates'));

    // Our Browser Tab
    jrCore_register_module_feature('jrCore', 'admin_tab', 'jrNewsLetter', 'browse', 'Newsletter Browser');

    // Our default view for admins
    jrCore_register_module_feature('jrCore', 'default_admin_view', 'jrNewsLetter', 'admin/tools');

    // Cleanup old settings
    jrCore_register_event_listener('jrCore', 'repair_module', 'jrNewsLetter_repair_module_listener');
    jrCore_register_event_listener('jrCore', 'parsed_template', 'jrNewsLetter_parsed_template_listener');
    jrCore_register_event_listener('jrMailer', 'campaign_result_header', 'jrNewsLetter_campaign_result_header_listener');
    jrCore_register_event_listener('jrMailer', 'campaign_result_row', 'jrNewsLetter_campaign_result_row_listener');

    // System reset listener
    jrCore_register_event_listener('jrDeveloper', 'reset_system', 'jrNewsLetter_reset_system_listener');

    // Daily Maintenance listener
    jrCore_register_event_listener('jrCore', 'daily_maintenance', 'jrNewsLetter_daily_maintenance_listener');

    // our newsletter_filters event
    jrCore_register_event_trigger('jrNewsLetter', 'get_matching_recipients', 'Fired to gather email addresses when sending');
    jrCore_register_event_trigger('jrNewsLetter', 'newsletter_filters', 'Fired in Compose to get available Newsletter Filters');

    // Let users unsubscribe in their notifications
    $_tmp = array(
        'label'      => 1,  // 'Newsletter'
        'help'       => 4,  // 'Would you like to receive the system newsletter?'
        'email_only' => true,
        'html_email' => true
    );
    jrCore_register_module_feature('jrUser', 'notification', 'jrNewsLetter', 'newsletter', $_tmp);

    // We provide a dashboard panel for subscribers
    jrCore_register_module_feature('jrCore', 'dashboard_panel', 'jrNewsLetter', 'newsletter subscribers', 'jrNewsLetter_dashboard_panels');

    // Our newsletter Prep worker
    jrCore_register_queue_worker('jrNewsLetter', 'prep_newsletter', 'jrNewsLetter_prep_newsletter_worker', 0, 1, 82800);

    // Our newsletter send worker
    jrCore_register_queue_worker('jrNewsLetter', 'send_newsletter', 'jrNewsLetter_send_newsletter_worker', 0, 1, 82800, LOW_PRIORITY_QUEUE);

    // Skin menu link to 'Archive'
    $_tmp = array(
        'group' => 'user',
        'label' => 8, // 'Newsletter Archive'
        'url'   => 'archive'
    );
    jrCore_register_module_feature('jrCore', 'skin_menu_item', 'jrNewsLetter', 'archive', $_tmp);

    return true;
}

//---------------------------------------
// QUEUE WORKER
//---------------------------------------

/**
 * Prep a newsletter for sending
 * @param $_queue array Queue entry
 * @return bool
 */
function jrNewsLetter_prep_newsletter_worker($_queue)
{
    @ini_set('memory_limit', '1024M');

    // Get our chunk size
    $chunk_size = jrCore_get_config_value('jrNewsLetter', 'send_rate', 30);

    // Our prep newsletter worker will gather all email addresses
    // for all selected options and create the necessary send_newsletter
    // queue workers ensuring each user email only receives one copy
    $_groups = explode(',', $_queue['letter_quota']);
    $_filter = explode(',', $_queue['letter_filter']);
    $_custom = (!empty($_queue['letter_custom'])) ? json_decode($_queue['letter_custom'], true) : null;
    $_emails = jrNewsLetter_get_matching_recipients($_groups, $_filter, $_custom);
    $total   = count($_emails);
    if ($total > 0) {
        $_emails = array_chunk($_emails, $chunk_size, true);
        $chunks  = count($_emails);
        foreach ($_emails as $k => $_chunk) {

            // Is this our LAST chunk?
            $_last = false;
            if (($k + 1) == $chunks) {
                // This is our last chunk - parse email template for notification
                $_queue['total_sent'] = $total;
                list($sub, $msg) = jrCore_parse_email_templates('jrNewsLetter', 'send_complete', $_queue);
                $_last = array(
                    'subject' => $sub,
                    'message' => $msg,
                    'email'   => $_queue['notification_email']
                );
            }

            // See if any recipients need to be delayed
            foreach ($_chunk as $l => $email) {
                $delay = jrNewsLetter_get_email_delay($email);
                if ($delay > 0) {
                    $_queue['emails'] = array($l => $email);
                    unset($_queue['queue_id']);
                    jrCore_queue_create('jrNewsLetter', 'send_newsletter', $_queue, $delay);
                    unset($_chunk[$l]);
                }
            }

            // Create our send queue
            $_queue['emails'] = $_chunk;
            if ($_last) {
                $_queue['notification'] = $_last;
            }
            unset($_queue['queue_id']);
            jrCore_queue_create('jrNewsLetter', 'send_newsletter', $_queue, ($k * 10));

        }
        $_tmp = array(
            'letter_recipients' => $total,
            'letter_quota'      => json_encode($_queue['letter_quota']),
            'letter_filter'     => json_encode($_queue['letter_filter']),
            'letter_custom'     => $_queue['letter_custom']
        );
        jrCore_db_update_item('jrNewsLetter', $_queue['newsletter_id'], $_tmp);
        jrCore_logger('INF', "successfully queued {$total} newsletter messages for delivery");
    }
    return true;
}

/**
 * Send a Newsletter
 * @param $_queue array Queue entry
 * @return bool
 */
function jrNewsLetter_send_newsletter_worker($_queue)
{
    $_options = array(
        'campaign_id'  => (int) $_queue['campaign_id'],
        'low_priority' => true
    );
    if (isset($_queue['emails']) && is_array($_queue['emails'])) {
        $_em = array();
        $_sp = array();
        $_us = array(
            'search'                       => array(
                '_item_id in ' . implode(',', array_keys($_queue['emails']))
            ),
            'order_by'                     => false,
            'include_jrProfile_keys'       => true,
            'exclude_jrProfile_quota_keys' => true,
            'privacy_check'                => false,
            'ignore_pending'               => true,
            'limit'                        => count($_queue['emails'])
        );
        $_us = jrCore_db_search_items('jrUser', $_us);
        if ($_us && is_array($_us) && isset($_us['_items']) && is_array($_us['_items'])) {
            $murl = jrCore_get_module_url('jrNewsLetter');
            $murl = jrCore_get_base_url() . "/{$murl}/unsubscribe";
            foreach ($_us['_items'] as $_u) {
                $uid = (int) $_u['_user_id'];
                // Check for unsubscribed user
                // @note: This is already checked before we get in to our newsletter worker queue
                // but we check it here again since it could have changed during the process
                if (isset($_queue['letter_ignore_unsub']) && $_queue['letter_ignore_unsub'] != 'on') {
                    if (isset($_u['user_unsubscribed']) && $_u['user_unsubscribed'] == 'on') {
                        $_sp[$uid] = $_u['user_email'];
                        continue;
                    }
                    // Make sure they have not disabled newsletters in notifications
                    if (isset($_u['user_jrNewsLetter_newsletter_notifications']) && $_u['user_jrNewsLetter_newsletter_notifications'] == 'off') {
                        $_sp[$uid] = $_u['user_email'];
                        continue;
                    }
                }
                // Include list-unsubscribe header
                if (!empty($_u['user_validate'])) {
                    $_options['headers'] = array('List-Unsubscribe' => "<{$murl}/{$_queue['campaign_id']}/{$_u['user_validate']}>");
                }
                else {
                    unset($_options['headers']);
                }
                // Replacement variables on all user keys
                $_rep = array();
                foreach ($_u as $k => $v) {
                    $_rep['{$' . $k . '}'] = $v;
                }
                $msg = jrCore_replace_emoji(str_replace(array_keys($_rep), $_rep, $_queue['letter_message']));
                $ttl = jrCore_replace_emoji(str_replace(array_keys($_rep), $_rep, $_queue['letter_title']));
                jrUser_notify($uid, 0, 'jrNewsLetter', 'newsletter', $ttl, $msg, $_options);
                $_em[$uid] = $_u['user_email'];
            }
            if (count($_sp) > 0) {
                // We had suppressed - update for newsletter
                jrCore_db_increment_key('jrNewsLetter', $_queue['newsletter_id'], 'letter_suppressed', count($_sp));
            }
        }

        // Archive emails we sent to...
        $_dt = array('emails' => $_em);
        if (count($_sp) > 0) {
            $_dt['suppressed'] = $_sp;
        }
        $lid = (int) $_queue['newsletter_id'];
        $tbl = jrCore_db_table_name('jrNewsLetter', 'emails');
        $req = "INSERT INTO {$tbl} (e_lid, e_time, e_emails) VALUES ({$lid}, UNIX_TIMESTAMP(), '" . jrCore_db_escape(json_encode($_dt)) . "')";
        jrCore_db_query($req);
    }

    // Notify that we are complete
    if (isset($_queue['notification'])) {
        jrCore_send_email($_queue['notification']['email'], $_queue['notification']['subject'], $_queue['notification']['message']);
    }

    return true;
}

//---------------------------------------
// DASHBOARD
//---------------------------------------

/**
 * Dashboard Panels
 * @param $panel
 * @return int|array
 */
function jrNewsLetter_dashboard_panels($panel)
{
    // The panel being asked for will come in as $panel
    $out = 0;
    switch ($panel) {

        case 'newsletter subscribers':
            $_sc = array(
                'search'         => array(
                    'user_notifications_disabled = on || user_jrNewsLetter_newsletter_notifications = off || user_active = 0'
                ),
                'return_count'   => true,
                'skip_triggers'  => true,
                'privacy_check'  => false,
                'ignore_missing' => true,
                'ignore_pending' => true,
                'order_by'       => false,
                'limit'          => 1000000
            );
            $all = jrCore_db_get_datastore_item_count('jrUser');
            $exc = jrCore_db_search_items('jrUser', $_sc);
            $out = array(
                'title' => jrCore_number_format($all - intval($exc))
            );
            break;

    }
    return $out;
}

//---------------------------------------
// EVENT LISTENERS
//---------------------------------------

/**
 * Daily Maintenance listener - Send scheduled newsletters
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrNewsLetter_daily_maintenance_listener($_data, $_user, $_conf, $_args, $event)
{
    // Send any scheduled newsletters
    $sch = jrCore_format_time(time(), false, "%Y/%m/%d");
    $_s  = array(
        "search"        => array(
            "letter_scheduled = {$sch}",
            'letter_draft = 1'
        ),
        "skip_triggers" => true,
        "limit"         => 1000
    );
    $_rt = jrCore_db_search_items('jrNewsLetter', $_s);
    if ($_rt && is_array($_rt['_items']) && count($_rt['_items']) > 0) {
        // We have newsletter(s) to send
        foreach ($_rt['_items'] as $rt) {
            // Our newsletter
            $rt['newsletter_id'] = $rt['_item_id'];
            // Setup campaign
            $campaign          = jrMailer_get_campaign_id('jrNewsLetter', $rt['_item_id'], $rt['letter_title'], $rt['letter_message']);
            $rt['campaign_id'] = $campaign;
            // Update newsletter with campaign ID
            jrCore_db_update_item('jrNewsLetter', $rt['_item_id'], array('letter_campaign_id' => $campaign));
            // It's no longer a draft - remove draft key
            jrCore_db_delete_item_key('jrNewsLetter', $rt['_item_id'], 'letter_draft');
            // Submit to prep queue
            jrCore_queue_create('jrNewsLetter', 'prep_newsletter', $rt);
            jrCore_logger('INF', "Scheduled NewsLetter sent: '{$rt['letter_title']}'");
        }
    }
    return $_data;
}

/**
 * System Reset listener
 * @param $_data array incoming data array
 * @param $_user array current user info
 * @param $_conf array Global config
 * @param $_args array additional info about the module
 * @param $event string Event Trigger name
 * @return array
 */
function jrNewsLetter_reset_system_listener($_data, $_user, $_conf, $_args, $event)
{
    $_tb = array(
        'track',
        'template',
        'emails'
    );
    foreach ($_tb as $table) {
        $tbl = jrCore_db_table_name('jrNewsLetter', $table);
        jrCore_db_query("TRUNCATE TABLE {$tbl}");
        jrCore_db_query("OPTIMIZE TABLE {$tbl}");
    }
    return $_data;
}

/**
 * Insert Suppressed Column to campaign results
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrNewsLetter_campaign_result_header_listener($_data, $_user, $_conf, $_args, $event)
{
    $_data[1]['width'] = '14.3%';
    $_data[2]['width'] = '14.3%';
    $_data[3]['width'] = '14.3%';
    $_data[4]['width'] = '14.3%';
    $_data[5]['width'] = '14.3%';
    $_data[6]['width'] = '14.3%';
    $_data[7]          = array(
        'title' => 'suppressed',
        'width' => '14.3%'
    );
    return $_data;
}

/**
 * Insert Suppressed Column to campaign results
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrNewsLetter_campaign_result_row_listener($_data, $_user, $_conf, $_args, $event)
{
    $suppress = jrCore_db_get_item_key('jrNewsLetter', $_args['campaign']['c_unique'], 'letter_suppressed');
    if ($suppress > 0) {
        $_data[7] = array(
            'title' => $suppress,
            'class' => 'bignum bignum1 nocursor'
        );
    }
    else {
        $_data[7] = array(
            'title' => 0,
            'class' => 'bignum bignum3 nocursor'
        );
    }
    return $_data;
}

/**
 * Add in FULL PAGE TinyMCE plugin for newsletter
 * @param $_data mixed information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrNewsLetter_parsed_template_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    if (isset($_post['module']) && $_post['module'] == 'jrNewsLetter' && isset($_args['jr_template']) && $_args['jr_template'] == 'form_editor.tpl' && isset($_args['jr_template_directory']) && $_args['jr_template_directory'] == 'jrCore') {
        // $_data contains our parsed editor config file - add in full page plugin
        if (!strpos($_data, 'fullpage,')) {
            $_data = str_replace('plugins: "', 'plugins: "fullpage,', $_data);
            $_data = str_replace('toolbar1: "', 'toolbar1: "fullpage | ', $_data);
        }
    }
    return $_data;
}

/**
 * Cleanup old settings
 * @param $_data array Array of information from trigger
 * @param $_user array Current user
 * @param $_conf array Global Config
 * @param $_args array additional parameters passed in by trigger caller
 * @param $event string Triggered Event name
 * @return array
 */
function jrNewsLetter_repair_module_listener($_data, $_user, $_conf, $_args, $event)
{
    // Move any users that have un subscribed to the proper key name
    $_rt = jrCore_db_get_multiple_items_by_key('jrUser', 'user_unsubscribed', 'on', true);
    if ($_rt && is_array($_rt)) {
        $_up = array();
        foreach ($_rt as $id) {
            $_up[$id] = array('user_jrNewsLetter_newsletter_notifications' => 'off');
        }
        jrCore_db_update_multiple_items('jrUser', $_up);
        jrCore_db_delete_key_from_multiple_items('jrUser', $_rt, 'user_unsubscribed');
    }

    // Fix up Newsletter titles
    $_rt = jrCore_db_get_all_key_values('jrNewsLetter', 'letter_subject');
    if ($_rt && is_array($_rt)) {
        $_up = array();
        foreach ($_rt as $id => $sub) {
            $_up[$id] = array('letter_title' => $sub, 'letter_title_url' => jrCore_url_string($sub));
        }
        if (count($_up) > 0) {
            if (jrCore_db_update_multiple_items('jrNewsLetter', $_up)) {
                jrCore_db_delete_key_from_all_items('jrNewsLetter', 'letter_subject');
            }
        }
    }

    return $_data;
}

//---------------------------------------
// FUNCTIONS
//---------------------------------------

/**
 * Replace user variables on subject and message
 * @param string $subject
 * @param string $message
 * @param array $_user
 * @return array
 */
function jrNewsLetter_replace_user_variables($subject, $message, $_user)
{
    // Replacement variables on all user keys
    $_rep = array();
    foreach ($_user as $k => $v) {
        // jrcore_form_notices and jrcore_memory_urls are array.  Only convert strings
        if (jrCore_checktype($v, 'string')) {
            $_rep['{$' . $k . '}'] = $v;
        }
    }
    return array(
        jrCore_replace_emoji(str_replace(array_keys($_rep), $_rep, $subject)),
        jrCore_replace_emoji(str_replace(array_keys($_rep), $_rep, $message))
    );
}

/**
 * Get email delivery delay for sending to specific providers
 * @param string $email
 * @return int
 * @see https://support.google.com/mail/answer/81126
 * @see http://www.yetesoft.com/free-email-marketing-resources/email-sending-limit/
 */
function jrNewsLetter_get_email_delay($email)
{
    if (jrCore_get_config_value('jrNewsLetter', 'delivery_delay', 'on') == 'on') {
        $key = 'jrnewsletter_email_delay';
        if (!$_dl = jrCore_get_flag($key)) {
            $_dl = array(
                // array(Delay Value, Delay Increment in Seconds)
                'hotmail'     => array(0, 5),
                'outlook.com' => array(0, 5),
                'msn.com'     => array(0, 5),
                'live.com'    => array(0, 5),
                'yahoo'       => array(0, 10),    // Yahoo: max 8640 in 24 hours
                'aol'         => array(0, 30)     // AOL: Max 2880 in 24 hours
            );
        }
        // Sending to specific destinations - i.e. Yahoo and AOL
        // require that we "trickle in" the newsletters - if they
        // see too many emails at once they will shut things down
        foreach ($_dl as $name => $_p) {
            if (strpos(jrCore_str_to_lower($email), $name)) {
                // We are delaying this email
                $delay         = $_p[0];
                $_dl[$name][0] += $_p[1];
                jrCore_set_flag($key, $_dl);
                return $delay;
            }
        }
    }
    return 0;
}

/**
 * Get all emails that match newsletter criteria
 * @param array $_groups Quotas that are being sent to
 * @param array $_filters Module provided filters
 * @param mixed $_custom User provided custom filters
 * @param bool $count_only set to TRUE to get count of matching recipients
 * @param int $size Loop Size (set lower for queue)
 * @param int $loop_sleep MS to sleep between each $size chunk
 * @return array|bool
 */
function jrNewsLetter_get_matching_recipients($_groups, $_filters = null, $_custom = null, $count_only = false, $size = 500, $loop_sleep = 0)
{
    global $_post;
    $_em = array();
    $_em = jrCore_trigger_event('jrNewsLetter', 'get_matching_recipients', $_em, array('_groups' => $_groups, '_filters' => $_filters, '_custom' => $_custom, 'size' => $size));
    if (count($_em) === 0 && is_array($_groups)) {

        if ($count_only) {

            // Used in listeners
            jrCore_set_flag('jrNewsLetter_count_only', 1);

            // 1) Get users in selected quotas
            if (!isset($_groups[0]) || intval($_groups[0]) !== 0) {
                foreach ($_groups as $gk => $group) {
                    $_groups[$gk] = intval($group);
                }
                $tb1 = jrCore_db_table_name('jrUser', 'item_key');
                $tb2 = jrCore_db_table_name('jrProfile', 'item_key');
                $req = "SELECT u.`_item_id` AS i FROM {$tb1} u
                     LEFT JOIN {$tb2} p ON (p.`_item_id` = u.`_profile_id` AND p.`key` = 'profile_quota_id')
                         WHERE u.`key` = '_created' AND p.`value` IN(" . implode(',', $_groups) . ')';
                $_em = jrCore_db_query($req, 'i', false, 'i');
            }
            else {
                // We're starting with ALL user_ids
                $_em = jrCore_db_get_all_key_values('jrUser', '_user_id');
            }
            if (empty($_em)) {
                // No matches in quotas
                return 0;
            }

            // 2) Get all users with disabled notifications
            // 3) Get all users with newsletters disabled
            // 4) Get all inactive users
            $_ky = array(
                'user_notifications_disabled'                => 'on',
                'user_jrNewsLetter_newsletter_notifications' => 'off',
                'user_active'                                => '0'
            );
            if ($_tm = jrCore_db_get_multiple_items_by_key('jrUser', 'ignored', $_ky, true)) {
                foreach ($_tm as $uid) {
                    if (isset($_em[$uid])) {
                        unset($_em[$uid]);
                    }
                }
            }
            if (empty($_em)) {
                return 0;
            }
        }
        else {

            // Get all users in selected Quotas
            // @note: If quota_id 0 is set we send to ALL USERS
            $_ids = false;
            if (!empty($_groups)) {
                foreach ($_groups as $qid) {
                    $qid = (int) $qid;
                    if ($qid === 0) {
                        $_ids = false;
                        break;
                    }
                    else {
                        if (!is_array($_ids)) {
                            $_ids = array();
                        }
                        $_ids[$qid] = $qid;
                    }
                }
                if (is_array($_ids)) {
                    $_ids = implode(',', $_ids);
                }
            }

            $uid = 0;
            while (true) {

                // Get all users in this quota
                $_us = array(
                    'search'        => array("_item_id > {$uid}"),
                    'order_by'      => array('_item_id' => 'asc'),
                    'return_keys'   => array('_user_id', 'user_active', 'user_email', 'user_notifications_disabled', 'user_jrNewsLetter_newsletter_notifications'),
                    'skip_triggers' => true,
                    'quota_check'   => false,
                    'privacy_check' => false,
                    'order_by'      => false,
                    'limit'         => $size
                );
                if ($_ids) {
                    $_us['search'][] = "profile_quota_id in {$_ids}";
                }
                $_us = jrCore_db_search_items('jrUser', $_us);
                if ($_us && is_array($_us) && isset($_us['_items']) && count($_us['_items']) > 0) {
                    foreach ($_us['_items'] as $_u) {
                        $uid = (int) $_u['_user_id'];
                        if (jrCore_checktype($_u['user_email'], 'email')) {
                            if (!isset($_u['user_active']) || intval($_u['user_active']) !== 1) {
                                // This user is NOT active
                                continue;
                            }
                            if (!isset($_post['letter_ignore_unsub']) || $_post['letter_ignore_unsub'] == 'off') {
                                if (isset($_u['user_notifications_disabled']) && $_u['user_notifications_disabled'] == 'on') {
                                    // All notifications are disabled for this user
                                    continue;
                                }
                                if (isset($_u['user_jrNewsLetter_newsletter_notifications']) && $_u['user_jrNewsLetter_newsletter_notifications'] == 'off') {
                                    // Newsletters have been disabled for user
                                    continue;
                                }
                            }
                            $_em[$uid] = $_u['user_email'];
                        }
                    }
                    if ($loop_sleep > 0) {
                        usleep($loop_sleep);
                    }
                }
                else {
                    break;
                }
            }
        }

        // Add module filters...
        if (count($_em) > 0 && is_array($_filters)) {
            $_gd = array();
            $add = false;
            foreach ($_filters as $id) {
                if (strpos($id, ':')) {
                    list($mod, $name) = explode(':', $id);
                    if (jrCore_module_is_active($mod)) {
                        $func = "{$mod}_newsletter_filter";
                        if (function_exists($func)) {
                            $_tm = $func($name);
                            if ($_tm && is_array($_tm)) {
                                if ($count_only) {
                                    foreach ($_em as $euid => $ignore) {
                                        if (!isset($_tm[$euid])) {
                                            unset($_em[$euid]);
                                        }
                                    }
                                }
                                else {
                                    // $_em contains the list of ALL email addresses that matched our GROUP selections.
                                    // $_tm contains the uid => email of users that match THIS filter
                                    foreach ($_tm as $uid => $email) {
                                        if (isset($_em[$uid])) {
                                            $_gd[$uid] = $email;
                                        }
                                    }
                                    $add = true;
                                }
                            }
                            else {
                                // We came out empty
                                return ($count_only) ? 0 : false;
                            }
                        }
                        else {
                            jrCore_logger('CRI', "registered newsletter recipient function for {$mod} does not exist: {$func}");
                        }
                    }
                }
            }
            if ($add) {
                if (count($_gd) > 0) {
                    $_em = $_gd;
                }
                else {
                    // We came out empty
                    return ($count_only) ? 0 : false;
                }
            }
        }

        // Add custom filters...
        if (count($_em) > 0 && !empty($_custom) && is_array($_custom) && count($_custom) > 0) {

            foreach ($_custom as $match) {

                if (strpos($match, 'user_') === 0) {
                    $mod = 'jrUser';
                }
                elseif (strpos($match, 'profile_') === 0) {
                    $mod = 'jrProfile';
                }
                elseif (strpos($match, '_created') === 0) {
                    $mod = 'jrUser';
                }
                elseif (strpos($match, '_updated') === 0) {
                    $mod = 'jrUser';
                }
                else {
                    continue;
                }

                if ($mod == 'jrUser') {
                    // We can do this one directly
                    $_us = array(
                        'search'              => array(trim($match)),
                        'skip_triggers'       => true,
                        'quota_check'         => false,
                        'privacy_check'       => false,
                        'return_item_id_only' => true,
                        'limit'               => 10000000
                    );
                    $_us = jrCore_db_search_items('jrUser', $_us);
                    if ($_us && is_array($_us)) {
                        $_us = array_flip($_us);
                        foreach ($_em as $uid => $email) {
                            if (!isset($_us[$uid])) {
                                unset($_em[$uid]);
                            }
                        }
                    }
                    else {
                        // No matches
                        return ($count_only) ? 0 : false;
                    }
                }
                else {
                    // For profiles we have to do a little bit more work - we have to
                    // find matching PROFILES, then get users assigned to those profiles
                    $_us = array(
                        'search'              => array(
                            trim($match),
                            'profile_active = 1'
                        ),
                        'skip_triggers'       => true,
                        'quota_check'         => false,
                        'privacy_check'       => false,
                        'return_item_id_only' => true,
                        'limit'               => 10000000
                    );
                    $_us = jrCore_db_search_items('jrProfile', $_us);
                    if ($_us && is_array($_us)) {
                        // We have found matching profiles - get user's associated with these profile id's
                        $tbl = jrCore_db_table_name('jrProfile', 'profile_link');
                        $req = "SELECT user_id FROM {$tbl} WHERE profile_id IN(" . implode(',', $_us) . ')';
                        $_us = jrCore_db_query($req, 'user_id', false, 'user_id');
                        if ($_us && is_array($_us)) {
                            foreach ($_em as $uid => $email) {
                                if (!isset($_us[$uid])) {
                                    unset($_em[$uid]);
                                }
                            }
                        }
                        else {
                            // No matches
                            return ($count_only) ? 0 : false;
                        }
                    }
                    else {
                        // No matches
                        return ($count_only) ? 0 : false;
                    }
                }
            }
        }
    }
    if (count($_em) > 0) {
        return ($count_only) ? count($_em) : $_em;
    }
    return ($count_only) ? 0 : false;
}

/**
 * Get any saved templates
 * @return array|false
 */
function jrNewsLetter_get_templates()
{
    $tbl = jrCore_db_table_name('jrNewsLetter', 'template');
    $req = "SELECT t_id, t_title FROM {$tbl} ORDER BY t_time DESC";
    return jrCore_db_query($req, 't_id', false, 't_title');
}

/**
 * Get an individual template
 * @param $template_id int Template ID
 * @return array|false
 */
function jrNewsLetter_get_template($template_id)
{
    $tid = (int) $template_id;
    $tbl = jrCore_db_table_name('jrNewsLetter', 'template');
    $req = "SELECT * FROM {$tbl} WHERE t_id = '{$tid}'";
    return jrCore_db_query($req, 'SINGLE');
}

/**
 * Check if an existing newsletter template exists
 * @param $title string Title
 * @return bool
 */
function jrNewsletter_template_exists($title)
{
    $ttl = jrCore_db_escape($title);
    $tbl = jrCore_db_table_name('jrNewsLetter', 'template');
    $req = "SELECT * FROM {$tbl} WHERE t_title = '{$ttl}'";
    $_ex = jrCore_db_query($req, 'SINGLE');
    if ($_ex && is_array($_ex)) {
        return true;
    }
    return false;
}

/**
 * Save a template
 * @param $title string Template title
 * @param $message string Message (body) of newsletter
 * @return int|false
 */
function jrNewsLetter_save_template($title, $message)
{
    $ttl = jrCore_db_escape($title);
    $msg = jrCore_db_escape($message);
    $tbl = jrCore_db_table_name('jrNewsLetter', 'template');
    $req = "INSERT IGNORE INTO {$tbl} (t_time, t_title, t_template) VALUES (UNIX_TIMESTAMP(), '{$ttl}', '{$msg}')";
    $tid = jrCore_db_query($req, 'INSERT_ID');
    if ($tid && $tid > 0) {
        return $tid;
    }
    return false;
}

/**
 * Delete an existing template
 * @param $template_id int Template ID
 * @return bool
 */
function jrNewsLetter_delete_template($template_id)
{
    $tid = (int) $template_id;
    $tbl = jrCore_db_table_name('jrNewsLetter', 'template');
    $req = "DELETE FROM {$tbl} WHERE t_id = '{$tid}'";
    $cnt = jrCore_db_query($req, 'COUNT');
    if ($cnt === 1) {
        return true;
    }
    return false;
}
