<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package Utilities
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 * @noinspection PhpElementIsNotAvailableInCurrentPhpVersionInspection
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

//-----------------------------------------
// FEATURE registration
//-----------------------------------------

/**
 * Register a feature for a module
 * @param string $module The Module that provides the feature
 * @param string $feature The Feature name
 * @param string $r_module The module that wants to use the feature
 * @param string $r_feature The unique name/key for the setting
 * @param mixed $_options optional parameters for the feature
 * @return bool
 */
function jrCore_register_module_feature($module, $feature, $r_module, $r_feature, $_options = true)
{
    if (!isset($GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module])) {
        $GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module] = array(
            $feature => array(
                $r_module => array()
            )
        );
    }
    elseif (!isset($GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature])) {
        $GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature] = array(
            $r_module => array()
        );
    }
    elseif (!isset($GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature][$r_module])) {
        $GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature][$r_module] = array();
    }
    // See if we have a registered module feature function
    if (isset($GLOBALS['__JR_FLAGS']['jrcore_register_module_feature_function'][$module][$feature])) {
        $GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature][$r_module][$r_feature] = $GLOBALS['__JR_FLAGS']['jrcore_register_module_feature_function'][$module][$feature]($r_module, $r_feature, $_options);
    }
    else {
        $GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature][$r_module][$r_feature] = $_options;
    }
    return true;
}

/**
 * Returns an array of modules registered for a given feature
 * @param string $module The Module that provides the feature
 * @param string $feature The unique Feature name from the providing module
 * @return mixed
 */
function jrCore_get_registered_module_features($module, $feature)
{
    return (!empty($GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature])) ? $GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature] : false;
}

/**
 * Delete a registered module feature for a module
 * @param string $module Module that provides the feature
 * @param string $feature Feature name
 * @param string $r_module Module that has registered for feature
 * @param string $r_feature Feature name
 * @return bool
 */
function jrCore_delete_registered_module_feature($module, $feature, $r_module, $r_feature = null)
{
    if (!is_null($r_feature) && isset($GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature][$r_module][$r_feature])) {
        unset($GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature][$r_module][$r_feature]);
    }
    elseif (isset($GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature][$r_module])) {
        unset($GLOBALS['__JR_FLAGS']['jrcore_register_module_feature'][$module][$feature][$r_module]);
    }
    return true;
}

/**
 * Run a function when a module calls the jrCore_register_module_feature function
 * @param string $module The Module that provides the feature
 * @param string $feature The unique Feature name from the providing module
 * @param string $function Function to execute when jrCore_register_module_feature is called for this feature
 * @return bool
 */
function jrCore_register_module_feature_function($module, $feature, $function)
{
    if (!function_exists($function)) {
        return false;
    }
    if (!$_tmp = jrCore_get_flag('jrcore_register_module_feature_function')) {
        $_tmp = array();
    }
    if (!isset($_tmp[$module])) {
        $_tmp[$module] = array();
    }
    $_tmp[$module][$feature] = $function;
    jrCore_set_flag('jrcore_register_module_feature_function', $_tmp);
    return true;
}

//-----------------------------------------
// COOKIE functions
//-----------------------------------------

/**
 * Set a persistent cookie
 * @param string $name Cookie Name
 * @param mixed $content Content (max ~4k)
 * @param int $expires epoch Time OR Days to expire (default 10)
 * @param bool $httponly Set to TRUE to force HTTP ONLY cookie (will not show in JS)
 * @return bool
 */
function jrCore_set_cookie($name, $content, $expires = 10, $httponly = false)
{
    if (headers_sent()) {
        return false;
    }
    if (!jrCore_checktype($name, 'core_string')) {
        jrCore_logger('CRI', "core: invalid cookie name recieved in jrCore_set_cookie: {$name}");
        return false;
    }
    $content = json_encode($content);
    $expires = intval($expires);
    if ($expires < 1000) {
        // We've been given a value in days
        $expires = (time() + ($expires * 86400));
    }
    else {
        $expires = (time() + $expires);
    }
    $name = jrCore_get_cookie_name($name);
    $ssl  = strpos(jrCore_get_base_url(), 'https') === 0;
    $same = jrCore_get_config_value('jrCore', 'same_site', 'Lax');

    // If we are on PHP >= 7.3 we have options
    if (version_compare(phpversion(), '7.3.0', '>=')) {
        $_op = array(
            'expires'  => $expires,
            'path'     => '/',
            'domain'   => '',
            'secure'   => $ssl,
            'httponly' => $httponly,
            'samesite' => $same
        );
        $tmp = setcookie($name, $content, $_op);
    }
    else {
        $tmp = setcookie($name, $content, $expires, "/; samesite={$same}", '', $ssl, $httponly);
    }
    if ($tmp) {
        $_COOKIE[$name] = $content;
        return true;
    }
    return false;
}

/**
 * Get value for persistent cookie if it exists
 * @param string $name Name of cookie to retrieve
 * @return bool|mixed
 */
function jrCore_get_cookie($name)
{
    $name = jrCore_get_cookie_name($name);
    if (!empty($_COOKIE[$name])) {
        if ($val = json_decode($_COOKIE[$name], true)) {
            return $val;
        }
        return $_COOKIE[$name];
    }
    return false;
}

/**
 * Delete a persistent cookie
 * @param string $name Name of cookie to delete
 * @return bool
 */
function jrCore_delete_cookie($name)
{
    $name = jrCore_get_cookie_name($name);
    setcookie($name, '', (time() - (365 * 86400)), '/');
    if (isset($_COOKIE[$name])) {
        unset($_COOKIE[$name]);
    }
    return true;
}

/**
 * Get the correct cookie name for the domain
 * @param string $name
 * @return string
 */
function jrCore_get_cookie_name($name)
{
    $uniq = jrUser_unique_install_id();
    return str_replace($uniq, '', $name) . $uniq;
}

//-----------------------------------------
// functions
//-----------------------------------------

/**
 * Check for a PHP FATAL error
 * @param string $error Error text (overrides error_get_last());
 */
function jrCore_check_for_fatal_error($error = false)
{
    $_err = error_get_last();
    if (!empty($error) || (is_array($_err) && $_err['type'] === E_ERROR)) {
        $_err['trace_id'] = md5(microtime() . mt_rand(0, 999999));
        $_err             = jrCore_trigger_event('jrCore', 'fatal_error', $_err);
        ob_start();
        echo jrCore_parse_template('fatal_error.tpl', $_err, 'jrCore');
        ob_end_flush();
        // @note: The system is an unknown state at this point - skip process_exit trigger
        jrCore_exit(false);
    }
}

/**
 * Return an "option" image HTML for pass/fail
 * @param string $state pass|fail
 * @param string $title alt|title of img tag
 * @param int $size pixel height/width
 * @return string
 */
function jrCore_get_option_image($state, $title = null, $size = 12)
{
    $key = "oi-{$state}-{$title}-{$size}";
    if ($image = jrCore_get_flag($key)) {
        return $image;
    }
    if (is_null($title)) {
        $title = $state;
    }
    switch ($state) {
        case 'pass':
        case 'success':
            $class = 'success';
            break;
        case 'fail':
            $class = 'fail';
            break;
        case 'warning':
        case 'error':
            $class = 'error';
            break;
        default:
            $class = 'notice';
            break;
    }
    $title = jrCore_entity_string($title);
    $size  = intval($size);

    $_rep = array(
        'title' => jrCore_escape_js_string($title),
        'state' => $state,
        'class' => $class,
        'size'  => $size
    );
    $html = jrCore_parse_template('option_image.tpl', $_rep, 'jrCore');
    jrCore_set_flag($key, $html);
    return $html;
}

/**
 * Creates a configurable "module jumper" for use in ACP views
 * @param $name string Select field name
 * @param $selected string Value to be pre-selected
 * @param $onchange string JS code to execute when changed
 * @param null $_modules array limit display to only these modules
 * @return string
 */
function jrCore_get_module_jumper($name, $selected, $onchange, $_modules = null)
{
    global $_mods;
    $_options = array();
    foreach ($_mods as $mod_dir => $_inf) {
        if ($_modules == null || is_array($_modules) && (isset($_modules[$mod_dir]) || in_array($mod_dir, $_modules))) {
            $cat = (isset($_inf['module_category'])) ? $_inf['module_category'] : 'custom';
            if (!isset($_options[$cat])) {
                $_options[$cat] = array();
            }
            $_options[$cat]["{$_inf['module_url']}"] = $_inf['module_name'];
        }
    }
    if (!empty($_options)) {
        foreach ($_options as $k => $v) {
            natcasesort($_options[$k]);
        }
    }
    ksort($_options, SORT_STRING);
    return jrCore_page_jumper($name, $_options, $_mods[$selected]['module_url'], $onchange);
}

/**
 * Return the detected URL to the system installation
 * @return string Returns full install URL
 */
function jrCore_get_detected_url()
{
    $protocol = jrCore_get_server_protocol() . '://';
    $uri      = (!empty($_SERVER['REQUEST_URI'])) ? rtrim($_SERVER['REQUEST_URI'], '/') : '';
    if (!empty($_SERVER['HTTP_HOST'])) {
        // Are we in a subdirectory?
        $subdir = rtrim(str_replace($_SERVER['DOCUMENT_ROOT'], '', APP_DIR), '/');
        if (!empty($subdir)) {
            // Yes - we have been installed in a sub directory
            return $protocol . rtrim($_SERVER['HTTP_HOST'], '/') . "/{$subdir}";
        }
        return $protocol . rtrim($_SERVER['HTTP_HOST'], '/') . $uri;
    }
    if (!empty($_SERVER['SERVER_NAME'])) {
        return $protocol . @rtrim($_SERVER['SERVER_NAME'], '/') . $uri;
    }
    return $uri;
}

/**
 * Returns server request protocol (http or https)
 */
function jrCore_get_server_protocol()
{
    // @NOTE: DO not use jrCore_get_config_value() here
    global $_conf;
    if (!empty($_conf['jrCore_base_url']) && strpos($_conf['jrCore_base_url'], 'https:') === 0) {
        return 'https';
    }
    return (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] != 'off') ? 'https' : 'http';
}

/**
 * Encode a string to be sent over a URL
 * @param string $string to Encode
 * @return string
 */
function jrCore_url_encode_string($string)
{
    return rawurlencode(htmlentities(rawurlencode(urlencode(base64_encode($string))), ENT_QUOTES));
}

/**
 * Decode a string encoded with jrCore_url_encode_string
 * @param string $string to Decode
 * @return string
 */
function jrCore_url_decode_string($string)
{
    return base64_decode(urldecode(rawurldecode(html_entity_decode(rawurldecode($string), ENT_QUOTES))));
}

/**
 * Convert a string to all lowercase
 * @param string $str String to lowercase
 * @return string
 */
function jrCore_str_to_lower($str)
{
    return (function_exists('mb_strtolower')) ? mb_strtolower($str, 'UTF-8') : strtolower($str);
}

/**
 * Convert a string to all uppercase
 * @param string $str String to uppercase
 * @return string
 */
function jrCore_str_to_upper($str)
{
    return (function_exists('mb_strtoupper')) ? mb_strtoupper($str, 'UTF-8') : strtoupper($str);
}

/**
 * Get the current URL
 * @return string Returns current URL (including mapped domains)
 */
function jrCore_get_current_url()
{
    global $_conf;
    $pfx = jrCore_get_server_protocol() . '://';
    $hst = (!empty($_SERVER['HTTP_HOST'])) ? $_SERVER['HTTP_HOST'] : $_conf['jrCore_base_url'];
    $dir = (!empty($_SERVER['PHP_SELF']) && strpos($_SERVER['PHP_SELF'], 'router.php') && strpos($_SERVER['PHP_SELF'], '/modules/') !== 0) ? substr($_SERVER['PHP_SELF'], 0, strpos($_SERVER['PHP_SELF'], '/modules/')) : '';
    $uri = (isset($_REQUEST['_uri'])) ? trim($_REQUEST['_uri']) : '';
    if (strpos($uri, '/') !== 0) {
        $uri = "/{$uri}";
    }
    return jrCore_trigger_event('jrCore', 'get_current_url', $pfx . rtrim($hst, '/') . $dir . $uri);
}

/**
 * Returns HTTP_REFERER if the URL is from the local site
 * @return string Returns URL to forward to on success
 */
function jrCore_get_local_referrer()
{
    $ref = jrCore_get_base_url(false);
    if (isset($_SERVER['HTTP_REFERER']) && basename($_SERVER['SCRIPT_FILENAME']) == 'router.php') {
        if (strpos($_SERVER['HTTP_REFERER'], $ref) === 0) {
            // this is a local request
            $ref = $_SERVER['HTTP_REFERER'];
        }
    }
    return jrCore_trigger_event('jrCore', 'get_local_referrer', $ref);
}

/**
 * @return bool
 * @see jrCore_is_local_url()
 * returns a URL if the referring URL is from the local site
 * @deprecated
 */
function jrCore_is_local_referrer()
{
    if ($url = jrCore_get_local_referrer()) {
        if (jrCore_is_local_url($url)) {
            return true;
        }
    }
    return false;
}

/**
 * Return TRUE if given URL is either the master URL or a valid URL as defined by a module
 * @param string $url
 * @return bool
 */
function jrCore_is_local_url($url)
{
    global $_conf;
    if (strpos($url, $_conf['jrCore_base_url']) === 0) {
        return true;
    }
    // If we don't match, see if it is an SSL/non-SSL issue
    $_rep = array('https://', 'http://');
    $tmp1 = str_replace($_rep, '', $url);
    $tmp2 = str_replace($_rep, '', $_conf['jrCore_base_url']);
    if (strpos($tmp1, $tmp2) === 0) {
        return true;
    }
    $_dat = array(
        'url'           => $url,
        'url_no_scheme' => $tmp1,
        'is_valid'      => 0
    );
    $_dat = jrCore_trigger_event('jrCore', 'is_local_url', $_dat);
    if ($_dat && is_array($_dat) && isset($_dat['is_valid']) && $_dat['is_valid'] == 1) {
        return true;
    }
    return false;
}

/**
 * Strip parameters from a URL
 * @param string $url URL to strip parameters from
 * @param array $_strip Array of parameter keys to strip
 * @return string
 */
function jrCore_strip_url_params($url, $_strip)
{
    if (is_array($_strip)) {
        foreach ($_strip as $strip) {
            if (stripos(' ' . $url, $strip)) {
                $url = preg_replace("%[?/]{$strip}=[^/]*%i", '', $url);
            }
        }
    }
    return $url;
}

/**
 * Return TRUE if running in developer mode
 * @return bool
 */
function jrCore_is_developer_mode()
{
    return jrCore_get_config_value('jrDeveloper', 'developer_mode', 'off') == 'on' && jrCore_module_is_active('jrDeveloper');
}

/**
 * Check if a given log priority level is enabled
 * @param string $priority cri|maj|min|inf|dbg
 * @return bool
 */
function jrCore_is_log_level_enabled($priority)
{
    $pri = strtolower($priority);
    $cfg = jrCore_get_config_value('jrCore', 'enabled', 'cri,maj,min,inf');
    if (!empty($cfg) && strpos(' ' . $cfg, $pri)) {
        return true;
    }
    if ($pri == 'dbg' && jrCore_is_developer_mode()) {
        return true;
    }
    return false;
}

/**
 * Return true if client has detached from process
 * @return mixed
 */
function jrCore_client_is_detached()
{
    return jrCore_get_flag('jrcore_client_is_detached');
}

/**
 * Set a new temp global flag
 * @param string $flag Unique flag string to set value for
 * @param mixed $value Value to store
 * @return bool
 */
function jrCore_set_flag($flag, $value)
{
    $GLOBALS['__JR_FLAGS'][$flag] = $value;
    return true;
}

/**
 * Retrieve a previously set temp global flag
 * @param mixed $flag String or Array to save to flag
 * @return mixed
 */
function jrCore_get_flag($flag)
{
    return (isset($GLOBALS['__JR_FLAGS'][$flag])) ? $GLOBALS['__JR_FLAGS'][$flag] : false;
}

/**
 * delete a previously set temp global flag
 * @param mixed $flag String or Array to delete
 * @return bool
 */
function jrCore_delete_flag($flag)
{
    if (isset($GLOBALS['__JR_FLAGS'][$flag])) {
        unset($GLOBALS['__JR_FLAGS'][$flag]);
        return true;
    }
    return false;
}

/**
 * 'Define' an Advanced Setting
 * @param $module string Module setting advanced config item
 * @param $key string Unique Advanced Key Name
 * @param $default mixed Default value for key not set
 * @param $help string Text that will show in help section
 * @return bool
 */
function jrCore_register_advanced_config_key($module, $key, $default, $help)
{
    if (!jrCore_module_is_active($module) || empty($key) || empty($help)) {
        return false;
    }
    if (!$_advkeys = jrCore_get_flag('jrCore_defined_advanced_keys')) {
        $_advkeys = array();
    }
    if (!isset($_advkeys[$module])) {
        $_advkeys[$module] = array();
    }
    $_advkeys[$module][$key] = array(
        'default' => $default,
        'help'    => $help
    );
    jrCore_set_flag('jrCore_defined_advanced_keys', $_advkeys);
    return true;
}

/**
 * Get value for an Advanced Setting (if set)
 * @param $module string Module setting advanced config item
 * @param $key string Unique Advanced Key Name
 * @param $default mixed Default value for key not set
 * @return mixed
 * @deprecated use jrCore_get_config_value
 */
function jrCore_get_advanced_setting($module, $key, $default)
{
    return jrCore_get_config_value($module, $key, $default);
}

/**
 * Get value for Global Config setting
 * @param string $module Module that contains the config item
 * @param string $key key name
 * @param mixed $default Default value for key not set
 * @param bool $cached Set to FALSE to retrieve value directly from database instead of $_conf
 * @return mixed
 */
function jrCore_get_config_value($module, $key, $default, $cached = true)
{
    global $_conf;
    if ($cached) {
        return (isset($_conf["{$module}_{$key}"])) ? $_conf["{$module}_{$key}"] : $default;
    }
    $tbl = jrCore_db_table_name('jrCore', 'setting');
    $req = "SELECT `value` FROM {$tbl} WHERE `module` = '" . jrCore_db_escape($module) . "' AND `name` = '" . jrCore_db_escape($key) . "'";
    $_rt = jrCore_db_query($req, 'SINGLE');
    if ($_rt && is_array($_rt) && isset($_rt['value'])) {
        return $_rt['value'];
    }
    return $default;
}

/**
 * Get a $_POST / $_REQUEST value
 * @param string $name $_POST / $_REQUEST key
 * @param mixed $default value to return if $name is empty
 * @return bool|mixed
 */
function jrCore_get_post_value($name, $default = false)
{
    global $_post;
    return (!empty($_post[$name])) ? $_post[$name] : $default;
}

/**
 * Parse REQUEST_URI into it's components
 * @param $uri string URI to parse - if empty uses $_SERVER['REQUEST_URI']
 * @param $cache bool set to FALSE to disable flag cache
 * @param bool $overwrite_request set to FALSE to not change $_REQUEST
 * @param bool $skip_triggers Set to TRUE to skip parse_url triggers
 * @return array
 */
function jrCore_parse_url($uri = null, $cache = true, $overwrite_request = true, $skip_triggers = false)
{
    global $_urls;
    // Check for process cache
    $ckey = "{$uri}_jr_parse_url";
    if ($cache) {
        if ($tmp = jrCore_get_flag($ckey)) {
            return $tmp;
        }
    }
    $_out = array();
    if (is_null($uri)) {
        $uri  = (!empty($_REQUEST['_uri'])) ? $_REQUEST['_uri'] : '/';
        $curl = urldecode(str_replace(array('%2B', '%26'), array('+', '___AMP'), $_SERVER['REQUEST_URI']));
    }
    else {
        $curl = urldecode(str_replace(array('%2B', '%26'), array('+', '___AMP'), $uri));
    }

    // Get everything cleaned up and into $_post
    $_id = array();
    if ($uri != '/' && strlen($uri) > 0) {

        $ruri = substr($curl, strpos($curl, '/' . $uri));
        // If we are NOT in an upload, make sure we strip out some URL params
        if (!strpos($ruri, '/upload_file')) {
            if (strpos(' ' . $ruri, ";")) {
                $ruri = substr($ruri, 0, strpos($ruri, ";"));
            }
            if (strpos(' ' . $ruri, '(')) {
                $ruri = substr($ruri, 0, strpos($ruri, "("));
            }
        }

        // Break up our URL
        $_tmp = explode('/', str_replace(array('?', '&', '//', '///', '////', '/////'), '/', trim(urldecode($ruri), '/')));
        if (is_array($_tmp)) {
            $idx = 0;
            // Page
            if (isset($_tmp[0]) && !strpos($_tmp[0], '=') && (!isset($_tmp[1]) || strpos($_tmp[1], '='))) {
                $_out['module_url'] = rawurlencode($_tmp[0]);
                $_out['module']     = (isset($_urls["{$_out['module_url']}"])) ? $_urls["{$_out['module_url']}"] : '';
                $idx                = 1;
            }
            // Module/View
            elseif (isset($_tmp[1]) && !strpos($_tmp[1], '=')) {
                $_out['module_url'] = rawurlencode($_tmp[0]);
                $_out['module']     = (isset($_urls["{$_out['module_url']}"])) ? $_urls["{$_out['module_url']}"] : '';
                $_out['option']     = rawurlencode($_tmp[1]);
                $idx                = 2;
            }

            // Handle any additional parameters
            if (isset($_tmp[$idx]) && strlen($_tmp[$idx]) > 0) {
                $vc   = 1;
                $ucnt = count($_tmp);
                for ($i = $idx; $i <= $ucnt; $i++) {
                    if (isset($_tmp[$i])) {
                        if (strpos($_tmp[$i], '=')) {
                            list($key, $val) = explode('=', $_tmp[$i]);
                            if ($key == 'module' || $key == 'option') {
                                continue;
                            }
                            if (jrCore_is_encoded_item_id($val)) {
                                $_id[$val] = jrCore_get_decoded_item_id($val);
                                $val       = $_id[$val];
                            }
                            // Check for URL encoded array []'s
                            if (strpos($key, '%5B%5D')) {
                                $key          = substr($key, 0, strpos($key, '%5B%5D'));
                                $_out[$key][] = str_replace('___AMP', '&', trim($val));
                            }
                            elseif (strpos($key, '[]')) {
                                $key          = substr($key, 0, strpos($key, '[]'));
                                $_out[$key][] = str_replace('___AMP', '&', trim($val));
                            }
                            elseif ($key == 'search_string') {
                                $_out[$key] = str_replace('___FS', '/', trim($val));
                            }
                            else {
                                $_out[$key] = str_replace('___AMP', '&', trim($val));
                            }
                        }
                        else {
                            if (jrCore_is_encoded_item_id($_tmp[$i])) {
                                $_id["{$_tmp[$i]}"] = jrCore_get_decoded_item_id($_tmp[$i]);
                                $_out["_{$vc}"]     = $_id["{$_tmp[$i]}"];
                            }
                            else {
                                // these are our "bare" parameters
                                $_out["_{$vc}"] = str_replace('___AMP', '&', trim($_tmp[$i]));
                            }
                            $vc++;
                        }
                    }
                    else {
                        // Done parsing - at the end
                        break;
                    }
                }
            }
        }
        if ($overwrite_request) {
            $_REQUEST['_uri'] = $ruri;
        }
    }
    elseif ($overwrite_request) {
        $_REQUEST['_uri'] = '/';
    }

    // Lastly, check for an AJAX request
    $_SERVER['jr_is_ajax_request'] = 0;
    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] == 'XMLHttpRequest') {
        $_SERVER['jr_is_ajax_request'] = 1;
    }
    elseif (isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], '__ajax')) {
        $_SERVER['jr_is_ajax_request'] = 1;
    }

    // Combine
    if ($overwrite_request) {
        $_new = $_REQUEST + $_out;
    }
    else {
        $_new = $_out;
    }
    if (!empty($_id) && !empty($_new['_uri'])) {
        $_new['_uri'] = str_replace(array_keys($_id), $_id, $_new['_uri']);
    }

    if (!$skip_triggers) {
        // let other modules checkout $_post...
        $_new = jrCore_trigger_event('jrCore', 'parse_url', $_new);
    }
    jrCore_set_flag($ckey, $_new);

    if ($overwrite_request) {
        $_REQUEST = $_new;
    }
    return $_new;
}

/**
 * Redirect a browser to a URL
 * @param string $url URL to forward to
 */
function jrCore_location($url)
{
    if ($url == 'referrer') {
        $url = jrCore_get_local_referrer();
    }
    $_tmp = jrCore_get_flag('jrcore_set_custom_header');
    if ($_tmp && is_array($_tmp)) {
        foreach ($_tmp as $header) {
            if (strpos($header, 'Location') !== 0) {
                header($header);
            }
        }
    }
    if (jrCore_is_ajax_request()) {
        // AJAX redirect
        return jrCore_json_response(array('redirect' => $url), true, false);
    }
    header('Location: ' . trim($url));
    jrCore_send_response_and_detach($url);
}

/**
 * Exit and run process_exit and process_done events
 * @param bool $process_exit Set to FALSE to skip running 'process_exit' event
 * @note: function exits
 */
function jrCore_exit($process_exit = true)
{
    global $_post;
    if (isset($_SESSION)) {
        if (!session_write_close()) {
            jrCore_record_event('session_close_error');
        }
    }
    if (!is_array($_post)) {
        $_post = array();
    }
    if ($process_exit && !jrCore_get_flag('jrcore_in_process_exit')) {
        jrCore_set_flag('jrcore_in_process_exit', 1);
        jrCore_trigger_event('jrCore', 'process_exit', $_post);
        jrCore_delete_flag('jrcore_in_process_exit');
    }
    jrCore_trigger_event('jrCore', 'process_done', $_post);
    exit; // OK
}

/**
 * Log an entry to the Activity Log
 * @param string $pri Priority - one of INF, MIN, MAJ, CRI
 * @param string $txt Text string to log
 * @param mixed $debug Additional debug information associated with the log entry
 * @param bool $include_user Include logging User Name in text
 * @param string $ip_address use IP Address instead of detected IP address
 * @param bool $throttle set to TRUE to enable log throttling
 * @return bool
 */
function jrCore_logger($pri, $txt, $debug = null, $include_user = true, $ip_address = null, $created_time = 0, $throttle = false)
{
    global $_user;
    if (jrCore_get_flag('jrCore_suppress_activity_log')) {
        // Activity Logging is suppressed in this process
        return true;
    }

    if (!jrCore_is_log_level_enabled($pri)) {
        // We are not enabled for this priority
        return true;
    }

    // Check for throttling - no more than 1 message with same $txt every X seconds
    if ($throttle && jrCore_local_cache_is_enabled()) {
        $ttl = jrCore_get_config_value('jrCore', 'log_throttle_ttl', 10);
        if ($ttl > 0) {
            $pfx = jrCore_local_cache_get_key_prefix(false);
            $md5 = md5($txt);
            $key = "{$pfx}:g:{$md5}";
            $cnt = apcu_inc($key, 1, $tmp, $ttl);
            if ($cnt > 1) {
                // We already exist - do NOT log - Increment throttled counter
                apcu_inc("{$key}c", 1, $tmp, 600);
                return true;
            }
            // We ARE going to log - get existing throttle count and reset
            if ($cnt = apcu_fetch("{$key}c")) {
                apcu_delete("{$key}c");
                if ($cnt > 1) {
                    $txt .= " (throttled: " . jrCore_number_format($cnt) . ")";
                }
            }
        }
    }

    $pri = strtolower($pri);
    $usr = '';
    if ($include_user) {
        if (jrCore_get_flag('jrcore_logger_system_user_active')) {
            $usr = '[system] ';
            $uip = $_SERVER['SERVER_ADDR'];
        }
        elseif (strpos($txt, '[') !== 0) {
            $usr = (!empty($_user['user_name'])) ? "[{$_user['user_name']}] " : ((isset($_user['user_email'])) ? "[{$_user['user_email']}] " : '');
            $uip = jrCore_get_ip();
        }
        else {
            $usr = "[{$include_user}]";
            $uip = jrCore_get_ip();
        }
    }
    else {
        $uip = jrCore_get_ip();
    }
    if (!is_null($ip_address)) {
        $uip = $ip_address;
    }

    // Create completed message
    $txt = trim($txt);
    if (strpos($txt, '%') !== 0) {
        $message = '%' . getmypid() . '%:' . $usr . $txt;
    }
    else {
        // PID is already there
        $message = $usr . $txt;
    }

    // trigger
    // We must set this flag here, since if a LISTENER calls a jrCore_logger
    // call, we don't want to fall down a trigger/event spiral
    if (!jrCore_get_flag('jrcore_logger_trigger_active')) {
        jrCore_set_flag('jrcore_logger_trigger_active', 1);
        $_data = array(
            'priority' => $pri,
            'message'  => $message,
            'user_ip'  => $uip,
            'debug'    => $debug
        );
        $_data = jrCore_trigger_event('jrCore', 'log_message', $_data);
        jrCore_delete_flag('jrcore_logger_trigger_active');
        if (!is_array($_data)) {
            // We have been handled by a listener - return
            return true;
        }
    }

    $time = 'UNIX_TIMESTAMP()';
    if ($created_time > 0) {
        $time = (int) $created_time;
    }
    if (is_null($debug)) {
        $tbl = jrCore_db_table_name('jrCore', 'log');
        $req = "INSERT INTO {$tbl} (log_created, log_priority, log_ip, log_text) VALUES ({$time}, '{$pri}', '" . jrCore_db_escape($uip) . "', '" . jrCore_db_escape($message) . "')";
        jrCore_db_query($req, null, false, null, false, null, false);
    }
    else {
        if (is_array($debug) || is_object($debug)) {
            $debug = print_r($debug, true);
        }
        $sav = jrCore_strip_emoji(jrCore_strip_non_utf8($debug), false);
        if (strlen($sav) > 65535) {
            $sav = substr($sav, 0, 65535);
        }
        $sav = jrCore_db_escape($sav);
        $mem = memory_get_usage(true);
        $uri = (isset($_REQUEST['_uri'])) ? jrCore_db_escape(substr($_REQUEST['_uri'], 0, 255)) : '/';
        $tb1 = jrCore_db_table_name('jrCore', 'log');
        $tb2 = jrCore_db_table_name('jrCore', 'log_debug_data');
        $_rq = array(
            "INSERT INTO {$tb1} (log_created, log_priority, log_ip, log_text) VALUES ({$time}, '{$pri}', '" . jrCore_db_escape($uip) . "', '" . jrCore_db_escape($message) . "')",
            "INSERT IGNORE INTO {$tb2} (log_log_id, log_created, log_url, log_memory, log_data) VALUES (LAST_INSERT_ID(), {$time}, '{$uri}', '{$mem}', COMPRESS('{$sav}'))"
        );
        jrCore_db_multi_query($_rq, false, false);
    }
    return true;
}

/**
 * Purge the Activity log according to Purge Activity Logs global config
 * @return int
 */
function jrCore_purge_activity_logs()
{
    $num = 0;
    if ($dys = jrCore_get_config_value('jrCore', 'purge_log_days', false)) {
        $old = (int) ($dys * 86400);
        $tb1 = jrCore_db_table_name('jrCore', 'log');
        $tb2 = jrCore_db_table_name('jrCore', 'log_debug');
        $tb3 = jrCore_db_table_name('jrCore', 'log_debug_data');
        while (true) {
            $req = "SELECT log_id FROM {$tb1} WHERE log_created < (UNIX_TIMESTAMP() - {$old}) LIMIT 500";
            $_rt = jrCore_db_query($req, 'log_id', false, 'log_id', false);
            if ($_rt && is_array($_rt)) {
                $ids = implode(',', $_rt);
                $_rq = array(
                    "DELETE FROM {$tb1} WHERE log_id IN({$ids})",
                    "DELETE FROM {$tb2} WHERE log_log_id IN({$ids})",
                    "DELETE FROM {$tb3} WHERE log_log_id IN({$ids})"
                );
                jrCore_db_multi_query($_rq, false);
                $num += count($_rt);
                if ($num >= 50000) {
                    break;
                }
                usleep(100000);
            }
            else {
                break;
            }
        }
        if ($num > 1000) {
            jrCore_logger('DBG', "core: successfully deleted " . jrCore_number_format($num) . " expired activity log entries");
        }
    }
    return $num;
}

/**
 * Replace emoji unicode characters with placeholders in a string
 * @param $string string
 * @param $replace bool set to TRUE to store emoji replacements
 * @return string
 */
function jrCore_strip_emoji($string, $replace = true)
{
    if (is_string($string)) {
        $pattern = '/([0-9|#][\x{20E3}])|[\x{00ae}|\x{00a9}\x{203C}\x{2047}\x{2048}\x{2049}\x{3030}\x{303D}\x{2139}\x{2122}\x{3297}\x{3299}][\x{FE00}-\x{FEFF}]?|[\x{2190}-\x{21FF}][\x{FE00}-\x{FEFF}]?|[\x{2300}-\x{23FF}][\x{FE00}-\x{FEFF}]?|[\x{2460}-\x{24FF}][\x{FE00}-\x{FEFF}]?|[\x{25A0}-\x{25FF}][\x{FE00}-\x{FEFF}]?|[\x{2600}-\x{27BF}][\x{FE00}-\x{FEFF}]?|[\x{2900}-\x{297F}][\x{FE00}-\x{FEFF}]?|[\x{2B00}-\x{2BF0}][\x{FE00}-\x{FEFF}]?|[\x{1F000}-\x{1FFFF}][\x{FE00}-\x{FEFF}]?/u';
        if (preg_match_all($pattern, $string, $_match)) {
            $_rp = array();
            foreach ($_match[0] as $e) {
                if (strlen($e) > 1) {
                    $_rp[$e] = $e;
                }
            }
            if (count($_rp) > 0) {
                if ($replace) {
                    if (!$_rt = jrCore_is_cached('jrCore', 'emoji_list', false, false, true, false)) {
                        $tbl = jrCore_db_table_name('jrCore', 'emoji');
                        $req = "SELECT * FROM {$tbl}";
                        $_rt = jrCore_db_query($req, 'emoji_value', false, 'emoji_id', false, null, false);
                        if (!is_array($_rt)) {
                            $_rt = 'no_items';
                        }
                        jrCore_add_to_cache('jrCore', 'emoji_list', $_rt, 0, 0, false, false, false);
                    }
                    if (!is_array($_rt)) {
                        $_rt = array();
                    }
                    $reset = false;
                    $tbl   = jrCore_db_table_name('jrCore', 'emoji');
                    foreach ($_rp as $k => $e) {
                        if (!isset($_rt[$k])) {
                            $req = "INSERT INTO {$tbl} (emoji_value) VALUES ('{$e}')";
                            if ($eid = jrCore_db_query($req, 'INSERT_ID', false, null, false, null, false)) {
                                $_rt[$k] = $eid;
                                $reset   = true;
                            }
                        }
                        else {
                            $eid = $_rt[$k];
                        }
                        if (!empty($eid)) {
                            // Replace with placeholder in our string
                            $string = str_replace($e, "!!emoji!!{$eid}!!emoji!!", $string);
                        }
                    }
                    if ($reset) {
                        jrCore_delete_cache('jrCore', 'emoji_list', false, false, false);
                    }
                }
                else {
                    foreach ($_rp as $e) {
                        $string = str_replace($e, '', $string);
                    }
                }
            }
        }
        return jrCore_strip_non_utf8($string);
    }
    return $string;
}

/**
 * Replace emoji placeholders in a string with actual emoji
 * @param $string string
 * @return mixed
 */
function jrCore_replace_emoji($string)
{
    if (is_string($string) && strpos(' ' . $string, '!!emoji!!')) {
        if (preg_match_all('/!!emoji!!([0-9]*)!!emoji!!/i', $string, $_match)) {
            $_id = array();
            foreach ($_match[0] as $e) {
                $_id[] = (int) str_ireplace('!!emoji!!', '', $e);
            }
            if (count($_id) > 0) {
                $key = 'replace_emoji';
                if (!$_rt = jrCore_is_cached('jrCore', $key, false, false, true, false)) {
                    $tbl = jrCore_db_table_name('jrCore', 'emoji');
                    $req = "SELECT CONCAT('!!emoji!!', emoji_id, '!!emoji!!') AS emoji_id, emoji_value FROM {$tbl}";
                    $_rt = jrCore_db_query($req, 'emoji_id', false, 'emoji_value', false, null, false);
                    if (!is_array($_rt)) {
                        $_rt = 'no_items';
                    }
                    jrCore_add_to_cache('jrCore', $key, $_rt, 0, 0, false, false, false);
                }
                if (is_array($_rt)) {
                    $string = str_ireplace(array_keys($_rt), $_rt, $string);
                }
            }
        }
    }
    return $string;
}

/**
 * Find if a request is for a VIEW (template, view, etc) and NOT an Ajax, IMG or CSS/JS request
 * @note use of $_REQUEST since $_post may not be built at time of function call
 * @return bool
 */
function jrCore_is_view_request()
{
    if (!empty($_REQUEST['_uri'])) {
        if (jrCore_is_ajax_request()) {
            return false;
        }
        if (jrCore_is_image_request()) {
            return false;
        }
        if (jrCore_is_favicon_request()) {
            return false;
        }
        if (jrCore_is_css_request()) {
            return false;
        }
        if (jrCore_is_javascript_request()) {
            return false;
        }
        if (strpos($_REQUEST['_uri'], '_v=')) {
            return false;
        }
    }
    return true;
}

/**
 * Return TRUE if a request is an XHR request
 * @return bool
 */
function jrCore_is_ajax_request()
{
    //[HTTP_X_REQUESTED_WITH] => XMLHttpRequest
    return (isset($_SERVER['jr_is_ajax_request']) && $_SERVER['jr_is_ajax_request'] === 1) || (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] == 'XMLHttpRequest');
}

/**
 * Return TRUE if this is a request for the site favicon
 * @return bool
 */
function jrCore_is_favicon_request()
{
    return (!empty($_REQUEST['_uri']) && strpos(' ' . $_REQUEST['_uri'], 'favicon'));
}

/**
 * Return TRUE if current request is an image request
 * @note use of $_REQUEST since $_post may not be built at time of function call
 * @return bool
 */
function jrCore_is_image_request()
{
    return (!empty($_REQUEST['_uri']) && ((strpos($_REQUEST['_uri'], 'image/') && !strpos($_REQUEST['_uri'], '/admin')) || strpos($_REQUEST['_uri'], 'gravatar/') || strpos($_REQUEST['_uri'], 'img/') || strpos($_REQUEST['_uri'], 'icon_sprite/') || strpos($_REQUEST['_uri'], 'icon/')));
}

/**
 * Return TRUE if the current URL request is for a CSS file
 * @return bool
 */
function jrCore_is_css_request()
{
    return (!empty($_REQUEST['_uri']) && (strpos($_REQUEST['_uri'], '.css') || strpos($_REQUEST['_uri'], 'icon_css/')));
}

/**
 * Return TRUE if the current URL request is for a Javascript file
 * @return bool
 */
function jrCore_is_javascript_request()
{
    return (!empty($_REQUEST['_uri']) && (strpos($_REQUEST['_uri'], '.js') || strpos($_REQUEST['_uri'], 'js/')));
}

/**
 * Convert URLs in a string into click URLs
 * @param string $string Input string to parse
 * @return string
 */
function jrCore_string_to_url($string)
{
    // replace links with click able version
    if (strpos(' ' . $string, 'http')) {
        $string = preg_replace('`([^"\'])(http[s]?://[^.]+\.[@\p{L}\p{N}\-,:./_?%#&=;~!+\]\[]+)`iu', '\1<a href="\2" target="_blank" rel="nofollow noopener">\2</a>', ' ' . $string);
        return substr($string, 1);
    }
    return $string;
}

/**
 * Download a file from a remote site by URL
 * @param string $remote_url Remote File URL
 * @param string $local_file Local file to save data to
 * @param int $max_download_time How many seconds to allow for file download before failing
 * @param int $port Remote Port to create socket connection to.
 * @param string $username HTTP Basic Authentication User Name
 * @param string $password HTTP Basic Authentication Password
 * @param string $agent Browser Agent to appear as
 * @param bool $log_error Set to FALSE to prevent Activity Log error if an error is encountered
 * @param bool $ssl_verify set to TRUE to force SSL verification
 * @return bool Returns true if file is downloaded, false on error
 */
function jrCore_download_file($remote_url, $local_file, $max_download_time = 120, $port = 80, $username = null, $password = null, $agent = null, $log_error = true, $ssl_verify = false)
{
    if (!jrCore_checktype($remote_url, 'url')) {
        jrCore_logger('CRI', "core: invalid URL received: {$remote_url}", array('url' => jrCore_get_current_url(), 'args' => func_get_args()));
    }
    set_time_limit(0);
    $_temp = jrCore_module_meta_data('jrCore');
    if (is_null($agent)) {
        if (isset($_SERVER['HTTP_USER_AGENT'])) {
            $agent = $_SERVER['HTTP_USER_AGENT'];
        }
        else {
            $agent = 'Jamroom v' . $_temp['version'];
        }
    }
    if ($port === 80 && strpos($remote_url, 'https:') === 0) {
        $port = 443;
    }
    if ($local = fopen($local_file, 'wb')) {
        $_opts = array(
            CURLOPT_USERAGENT      => $agent,
            CURLOPT_URL            => $remote_url, // File we are downloading
            CURLOPT_PORT           => (int) $port,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => intval($max_download_time),
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_FILE           => $local,
            CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
            CURLOPT_HEADERFUNCTION => 'jrCore_save_load_url_response_headers'
        );
        if (!$ssl_verify) {
            $_opts[CURLOPT_SSL_VERIFYHOST] = false;
            $_opts[CURLOPT_SSL_VERIFYPEER] = false;
        }
        if ($_headers = jrCore_get_flag('jrcore_custom_request_headers')) {
            $_opts[CURLOPT_HTTPHEADER] = $_headers;
            jrCore_delete_flag('jrcore_custom_request_headers');
        }
        // Check for HTTP Basic Authentication
        if (!is_null($username) && !is_null($password)) {
            $_opts[CURLOPT_USERPWD] = $username . ':' . $password;
        }
        $ch = curl_init();
        if (curl_setopt_array($ch, $_opts)) {
            jrCore_start_timer('download_file');
            curl_exec($ch);
            jrCore_stop_timer('download_file');
            $err = curl_errno($ch);
            fclose($local);
            if (!isset($err) || $err === 0) {
                curl_close($ch);
                return true;
            }
            $errmsg = curl_error($ch);
            curl_close($ch);
            if ($log_error) {
                jrCore_logger('CRI', "core: {$remote_url} returned error #{$err} ({$errmsg})");
            }
            return false;
        }
        fclose($local);
        curl_close($ch);
    }
    return false;
}

/**
 * Store response headers from a jrCore_load_url() call
 * @param $curl resource
 * @param $line string
 * @return int
 */
function jrCore_save_load_url_response_headers($curl, $line)
{
    // Used to hold response headers
    $_tm = jrCore_get_flag('jrcore_load_url_response_headers');
    if (!$_tm) {
        $_tm = array();
    }
    if (strlen(trim($line)) > 0) {
        $_tm[] = trim($line);
        jrCore_set_flag('jrcore_load_url_response_headers', $_tm);
    }
    return strlen($line);
}

/**
 * Get last response headers from a jrCore_load_url() call
 * @return bool|mixed
 */
function jrCore_get_load_url_response_headers()
{
    if ($_tm = jrCore_get_flag('jrcore_load_url_response_headers')) {
        return $_tm;
    }
    return false;
}

/**
 * Set custom request headers
 * @param array $_headers
 * @return bool
 */
function jrCore_set_custom_request_headers($_headers)
{
    if (is_array($_headers)) {
        return jrCore_set_flag('jrcore_custom_request_headers', $_headers);
    }
    return false;
}

/**
 * Get contents of a remote URL
 * @param string $url Url to load
 * @param mixed $_vars URI variables for URL
 * @param string $method URL method (POST or GET)
 * @param int $port Remote Port to create socket connection to.
 * @param string $username HTTP Basic Authentication User Name
 * @param string $password HTTP Basic Authentication Password
 * @param bool $log_error Set to false to prevent error logging on failed URL load
 * @param int $max_transfer_time Time in seconds to allow for data transfer
 * @param string $agent Browser Agent to appear as
 * @param bool $uploads set to TRUE to allow @file_uploads
 * @param bool $ssl_verify set to TRUE to verify SSL certificate
 * @return string Returns value of loaded URL, or false on failure
 */
function jrCore_load_url($url, $_vars = null, $method = 'GET', $port = 80, $username = null, $password = null, $log_error = true, $max_transfer_time = 30, $agent = null, $uploads = false, $ssl_verify = false)
{
    if (!jrCore_checktype($url, 'url')) {
        $_rs = array(
            '_vars'    => $_vars,
            'referrer' => jrCore_get_local_referrer()
        );
        jrCore_logger('CRI', "core: invalid URL received: {$url}", $_rs);
        return false;
    }

    // Save URLs
    jrCore_delete_flag('jrcore_load_url_response_headers');
    // Our user agent
    if (is_null($agent)) {
        $_temp = jrCore_module_meta_data('jrCore');
        $agent = 'Jamroom v' . $_temp['version'];
    }
    if (intval($port) === 80 && strpos($url, 'https:') === 0) {
        $port = 443;
    }
    $_opts = array(
        CURLOPT_POST           => false,
        CURLOPT_HEADER         => false,
        CURLOPT_USERAGENT      => $agent,
        CURLOPT_URL            => $url,
        CURLOPT_PORT           => (int) $port,
        CURLOPT_FRESH_CONNECT  => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FORBID_REUSE   => true,
        CURLOPT_TIMEOUT        => intval($max_transfer_time),
        CURLOPT_CONNECTTIMEOUT => ($max_transfer_time < 10) ? $max_transfer_time : 10,
        CURLOPT_VERBOSE        => false,
        CURLOPT_FAILONERROR    => false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS      => 3,
        CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
        CURLOPT_HEADERFUNCTION => 'jrCore_save_load_url_response_headers'
    );
    if (!$ssl_verify) {
        $_opts[CURLOPT_SSL_VERIFYHOST] = false;
        $_opts[CURLOPT_SSL_VERIFYPEER] = false;
    }
    else {
        $_opts[CURLOPT_SSL_VERIFYHOST] = 2;
        $_opts[CURLOPT_SSL_VERIFYPEER] = true;
    }
    if ($_headers = jrCore_get_flag('jrcore_custom_request_headers')) {
        $_opts[CURLOPT_HTTPHEADER] = $_headers;
        jrCore_delete_flag('jrcore_custom_request_headers');
    }

    // Curl File Handling has changed in PHP 5.5.0
    if ($uploads) {
        // Are we uploading any files?
        if (function_exists('curl_file_create')) {
            foreach ($_vars as $k => $v) {
                if (strpos($v, '@') === 0) {
                    $file = substr($v, 1);
                    if (file_exists($file) && strpos(realpath($file), APP_DIR) === 0) {
                        $mime = jrCore_mime_type($file);
                        /** @noinspection PhpElementIsNotAvailableInCurrentPhpVersionInspection */
                        $_vars[$k] = curl_file_create($file, $mime, basename($file));
                    }
                }
            }
        }
    }

    // Check for HTTP Basic Authentication
    if (!is_null($username) && !is_null($password)) {
        $_opts[CURLOPT_USERPWD] = $username . ':' . $password;
    }
    switch (strtoupper("{$method}")) {
        case 'POST':
            $_opts[CURLOPT_POST] = true;
            if (is_array($_vars) && count($_vars) > 0) {
                $_opts[CURLOPT_POSTFIELDS] = $_vars;
            }
            elseif (!is_null($_vars) && strlen($_vars) > 0) {
                $_opts[CURLOPT_POSTFIELDS] = trim($_vars);
            }
            break;
        case 'GET':
            $_opts[CURLOPT_HTTPGET] = true;
            if (is_array($_vars) && count($_vars) > 0) {
                if (strpos($url, '?')) {
                    $_opts[CURLOPT_URL] = $url . '&' . http_build_query($_vars);
                }
                else {
                    $_opts[CURLOPT_URL] = $url . '?' . http_build_query($_vars);
                }
            }
            elseif (!is_null($_vars) && strlen($_vars) > 0) {
                if (strpos($url, '?')) {
                    $_opts[CURLOPT_URL] = $url . '&' . trim($_vars);
                }
                else {
                    $_opts[CURLOPT_URL] = $url . '?' . trim($_vars);
                }
            }
            break;
        case 'PUT':
            $_opts[CURLOPT_CUSTOMREQUEST] = 'PUT';
            if (is_array($_vars) && count($_vars) > 0) {
                $_opts[CURLOPT_POSTFIELDS] = $_vars;
            }
            elseif (!is_null($_vars) && strlen($_vars) > 0) {
                $_opts[CURLOPT_POSTFIELDS] = trim($_vars);
            }
            break;
        case 'DELETE':
            $_opts[CURLOPT_CUSTOMREQUEST] = 'DELETE';
            if (is_array($_vars) && count($_vars) > 0) {
                if (strpos($url, '?')) {
                    $_opts[CURLOPT_URL] = $url . '&' . http_build_query($_vars);
                }
                else {
                    $_opts[CURLOPT_URL] = $url . '?' . http_build_query($_vars);
                }
            }
            break;
    }
    $ch = curl_init();
    if (curl_setopt_array($ch, $_opts)) {
        jrCore_start_timer('load_url');
        $res = curl_exec($ch);
        jrCore_stop_timer('load_url');
        // @see: https://curl.se/libcurl/c/libcurl-errors.html
        $err = curl_errno($ch);
        $_lu = curl_getinfo($ch);
        if (empty($err)) {
            curl_close($ch);
            jrCore_save_loaded_url("{$url}:{$_lu['http_code']}:{$_lu['total_time']}:{$_lu['size_download']}");
            return $res;
        }
        $errmsg = curl_error($ch);
        jrCore_save_loaded_url("{$url}:{$_lu['http_code']}:{$_lu['total_time']}:{$_lu['size_download']}:{$errmsg}");
        if ($log_error) {

            // Save info about call to log
            $_log = array();
            if (!empty($_vars)) {
                $_log['_vars'] = $_vars;
            }
            $_log['url']        = $url;
            $_log['error_code'] = $err;
            $_log['error_text'] = $errmsg;

            $url = jrCore_strip_url_params($url, array('auth'));
            jrCore_logger('CRI', "core: load_url {$url} - {$errmsg}", $_log);
        }
        // Remove response headers as could be incorrect (from last call in this process)
        jrCore_delete_flag('jrcore_load_url_response_headers');
    }
    else {
        jrCore_save_loaded_url("{$url}:0:0:0:unable to set cURL options via curl_setopt_array");
        if ($log_error) {
            jrCore_logger('CRI', "core: unable to set cURL options via curl_setopt_array");
        }
    }
    curl_close($ch);
    return false;
}

/**
 * Load multiple URLs in one call asynchronously
 * @param $_urls array URL info
 * @param $agent string Alternate User Agent
 * @param $sleep int Number of microseconds to sleep between each check of curl_multi_exec() to see if URLs have completed
 * @return array|false
 */
function jrCore_load_multiple_urls($_urls, $agent = null, $sleep = 10000)
{
    $_ch   = array();
    $_temp = jrCore_module_meta_data('jrCore');
    if (is_null($agent)) {
        $agent = 'Jamroom v' . $_temp['version'];
    }
    $crl = curl_multi_init();
    foreach ($_urls as $k => $_url) {
        if (!isset($_url['url']) || !jrCore_checktype($_url['url'], 'url')) {
            return false;
        }
        $transfer_timeout = (isset($_url['timeout']) && jrCore_checktype($_url['timeout'], 'number_nz')) ? intval($_url['timeout']) : 30;
        $connect_timeout  = (isset($_url['connect_timeout']) && jrCore_checktype($_url['connect_timeout'], 'number_nz')) ? intval($_url['connect_timeout']) : 10;
        if (isset($_url['port']) && jrCore_checktype($_url['port'], 'number_nz')) {
            $port = intval($_url['port']);
        }
        else {
            $port = (strpos($_url['url'], 'https:') === 0) ? 443 : 80;
        }
        $_opts = array(
            CURLOPT_POST           => false,
            CURLOPT_HEADER         => false,
            CURLOPT_USERAGENT      => $agent,
            CURLOPT_URL            => $_url['url'],
            CURLOPT_PORT           => $port,
            CURLOPT_FRESH_CONNECT  => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FORBID_REUSE   => true,
            CURLOPT_TIMEOUT        => $transfer_timeout,
            CURLOPT_CONNECTTIMEOUT => $connect_timeout,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS      => 3,
            CURLOPT_VERBOSE        => false,
            CURLOPT_FAILONERROR    => false,
            CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
            CURLOPT_HEADERFUNCTION => 'jrCore_save_load_url_response_headers'
        );
        if (!isset($_url['ssl_verify']) || $_url['ssl_verify'] === false) {
            $_opts[CURLOPT_SSL_VERIFYHOST] = false;
            $_opts[CURLOPT_SSL_VERIFYPEER] = false;
        }
        if ($_headers = jrCore_get_flag('jrcore_custom_request_headers')) {
            $_opts[CURLOPT_HTTPHEADER] = $_headers;
            jrCore_delete_flag('jrcore_custom_request_headers');
        }
        // Are we uploading any files?
        if (function_exists('curl_file_create') && isset($_url['_data']) && is_array($_url['_data'])) {
            foreach ($_url['_data'] as $fk => $fv) {
                if (strpos($fv, '@') === 0) {
                    $file = substr($fv, 1);
                    if (file_exists($file)) {
                        $mime = jrCore_mime_type($file);
                        /** @noinspection PhpElementIsNotAvailableInCurrentPhpVersionInspection */
                        $_url['_data'][$fk] = curl_file_create($file, $mime, basename($file));
                    }
                }
            }
        }
        // Check for HTTP Basic Authentication
        if (isset($_url['username']) && strlen($_url['username']) > 0 && isset($_url['password']) && strlen($_url['password']) > 0) {
            $_opts[CURLOPT_USERPWD] = $_url['username'] . ':' . $_url['password'];
        }
        if (isset($_url['method']) && $_url['method'] == 'POST') {
            $_opts[CURLOPT_POST] = true;
        }
        if (isset($_url['_data']) && is_array($_url['_data'])) {
            $_opts[CURLOPT_POSTFIELDS] = $_url['_data'];
        }
        elseif (isset($_url['_data']) && strlen($_url['_data']) > 0) {
            $_opts[CURLOPT_POSTFIELDS] = trim($_url['_data']);
        }
        $_ch[$k] = curl_init();
        curl_setopt_array($_ch[$k], $_opts);
        curl_multi_add_handle($crl, $_ch[$k]);
    }
    if (count($_ch) === 0) {
        return false;
    }

    $active = null;
    // Get content
    jrCore_start_timer('load_url');
    do {
        curl_multi_exec($crl, $active);
        usleep($sleep);
    }
    while ($active > 0);

    $res = array();
    foreach ($_ch as $k => $handle) {
        $res[$k] = curl_multi_getcontent($handle);
        if ($_lu = curl_getinfo($handle)) {
            jrCore_save_loaded_url("{$_urls[$k]['url']}:{$_lu['http_code']}:{$_lu['total_time']}:{$_lu['size_download']}");
        }
    }
    foreach ($_ch as $handle) {
        curl_multi_remove_handle($crl, $handle);
    }
    curl_multi_close($crl);
    jrCore_stop_timer('load_url');
    return $res;
}

/**
 * Save a Loaded URL
 * @param string $url
 * @return bool
 */
function jrCore_save_loaded_url($url)
{
    if (!$_lu = jrCore_get_flag('jrcore_loaded_urls')) {
        $_lu = array();
    }
    $_lu[] = $url;
    return jrCore_set_flag('jrcore_loaded_urls', $_lu);
}

/**
 * Get array of all URLs loaded via jrCore_load_url
 * @return mixed
 */
function jrCore_get_loaded_urls()
{
    return jrCore_get_flag('jrcore_loaded_urls');
}

/**
 * Run strip_tags on a string with special check for <=
 * @param string $string String to strip HTML tags from
 * @return string
 */
function jrCore_strip_tags($string)
{
    // Specifically check for ' <= ' - this is a LESS THAN OR EQUAL but is stripped by strip_tags
    if (strpos($string, ' <= ')) {
        $string = str_replace(' <= ', '', $string);
    }
    return strip_tags($string);
}

/**
 * Strip unsafe HTML tags from a string (recursive)
 * @param mixed $string String or Array to strip tags from
 * @param string $_allowed comma separated list of allowed HTML tags
 * @param bool $strip_embed set to TRUE to strip [jrEmbed] tags
 * @return mixed
 */
function jrCore_strip_html($string, $_allowed = null, $strip_embed = false)
{
    if (is_array($string)) {
        foreach ($string as $k => $v) {
            $string[$k] = jrCore_strip_html($v, $_allowed);
        }
    }
    else {

        if (!is_string($string) || strlen("{$string}") === 0) {
            return $string;
        }

        if ($strip_embed && strpos(' ' . $string, '[jrEmbed')) {
            // We have embed codes in this string - strip
            $string = preg_replace('/\[jrEmbed[^]]+]/', '', $string);
        }

        if ($string == jrCore_strip_tags($string)) {
            // No HTML in string
            return $string;
        }

        // Cleanup any malformed HTML
        $string = jrCore_clean_html($string);

        if (!is_null($_allowed)) {
            if (jrUser_is_master() && jrCore_get_flag('master_html_trusted')) {
                // It's the master and we've been requested to return trusted
                return $string;
            }
            elseif (jrUser_get_profile_home_key('quota_jrCore_allow_all_html') === 'on') {
                // This quota is fully trusted
                return $string;
            }
        }

        $_cs = array();
        // We have to do a quick check for icon_css (i.e. from an embedded item) - if we strip it
        // out then icon's may not show properly for any item that was relying on the CSS
        if (strpos($string, '/icon_css/')) {
            if (preg_match('%/icon_css/([0-9]*)%', $string, $_match)) {
                if ($_match && is_array($_match) && isset($_match[1])) {
                    $url = jrCore_get_module_url('jrCore');
                    foreach ($_match as $k => $v) {
                        if ($k > 0 && !isset($_cs[$v])) {
                            $_tm = array('source' => jrCore_get_base_url() . '/' . $url . '/icon_css/' . $v . '?_v=' . time());
                            jrCore_create_page_element('css_footer_href', $_tm);
                            $_cs[$v] = 1;
                        }
                    }
                    unset($url, $_match);
                }
            }
        }

        // cleanup with HTML purifier
        require_once APP_DIR . '/modules/jrCore/contrib/htmlpurifier/HTMLPurifier.standalone.php';

        $allw       = '';
        $trgt       = false;
        $_xtr       = array();
        $trusted    = false;
        $safe_embed = false;
        if (!is_null($_allowed) && strlen($_allowed) > 0) {
            $_all = explode(',', $_allowed);
            $_att = array();
            $_tmp = array();
            if (is_array($_all)) {
                foreach ($_all as $k => $tag) {
                    $tag = trim($tag);
                    if (strlen($tag) > 0) {

                        // Get Tag Attributes
                        switch ($tag) {
                            // Setup some defaults for ease of use
                            case 'a':
                                $_att[] = 'a.name,a.href,a.target,a.rel,a.title,a.data-lightbox,a.title,a.style,a.id,a.class';
                                $trgt   = true;
                                break;
                            case 'img':
                                $_att[] = 'img.src,img.width,img.min-width,img.max-width,img.height,img.min-height,img.max-height,img.alt,img.style,img.class,img.title';
                                break;
                            case 'script':
                                $_att[]     = 'script.type';
                                $safe_embed = true;
                                break;
                            case 'iframe':
                                $_att[]     = 'iframe.src,iframe.width,iframe.height,iframe.name,iframe.align,iframe.frameborder,iframe.marginwidth,iframe.marginheight';
                                $safe_embed = true;
                                break;
                            case 'object':
                                $_att[]     = 'object.width,object.height,object.style,object.class';
                                $safe_embed = true;
                                break;
                            case 'param':
                                $_att[]     = 'param.name,param.value';
                                $safe_embed = true;
                                break;
                            case 'embed':
                                $_att[] = 'embed.src,embed.type,embed.allowscriptaccess,embed.allowfullscreen,embed.width,embed.height,embed.flashvars';
                                break;
                            case 'form':
                            case 'input':
                                $trusted = true;
                                break;
                            case 'table':
                                $_att[] = 'table.class,table.style,table.border,table.align,table.cellspacing,table.cellpadding';
                                $_xtr[] = 'tbody';
                                $_xtr[] = 'caption';
                                break;
                            case 'tr':
                                $_att[] = 'tr.class,tr.style,tr.align';
                                break;
                            case 'td':
                                $_att[] = 'td.class,td.style,td.align,td.valign,td.colspan,td.rowspan';
                                break;
                            case 'th':
                                $_att[] = 'th.class,th.style,th.align,th.valign,th.colspan,th.rowspan';
                                break;
                            case 'hr':
                                $_att[] = 'hr.id,hr.class,hr.style,hr.align,hr.color,hr.width,hr.noshade';
                                break;
                            case 'p':
                            case 'span':
                            case 'div':
                            case 'h1':
                            case 'h2':
                            case 'h3':
                            case 'h4':
                            case 'h5':
                            case 'h6':
                                $_att[] = "{$tag}.id,{$tag}.class,{$tag}.style,{$tag}.title,{$tag}.align";
                                break;
                            default:
                                // If the tag has a period in it - i.e. "iframe.src"
                                // we need to add tag to tags, and attribute to attributes
                                if (strpos($tag, '.')) {
                                    list($t,) = explode('.', $tag);
                                    $_att[] = $tag;
                                    $tag    = $t;
                                }
                                break;
                        }
                        $_all[$k]   = $tag;
                        $_tmp[$tag] = 1;
                    }
                }

                // Allow other modules to expand on our work
                $_arg = array(
                    'string' => $string
                );
                $_all = jrCore_trigger_event('jrCore', 'allowed_html_tags', $_all, $_arg);

                // Add in extras for proper table support
                if (count($_xtr) > 0) {
                    foreach ($_xtr as $xtra) {
                        $_all[] = $xtra;
                    }
                }

                // Finalize
                $allw = implode(',', $_all);
            }
            unset($_all);

            // now strip our tags
            if (!$trusted && strlen($allw) > 0) {

                // See: http://htmlpurifier.org/live/configdoc/plain.html
                $pc = HTMLPurifier_Config::createDefault();
                $pc->set('Core.NormalizeNewlines', false);
                $pc->set('Cache.SerializerPath', jrCore_get_module_cache_dir('jrCore'));
                $pc->set('HTML.AllowedElements', $allw);
                $pc->set('HTML.SafeEmbed', $safe_embed);
                $pc->set('HTML.MaxImgLength', null);
                $pc->set('CSS.Trusted', true);
                $pc->set('CSS.Proprietary', true);
                $pc->set('CSS.AllowTricky', true);
                $pc->set('CSS.MaxImgLength', null);
                $pc->set('Attr.EnableID', true);
                if (isset($_att) && is_array($_att) && count($_att) > 0) {
                    $pc->set('HTML.AllowedAttributes', implode(',', $_att));
                    unset($_att);
                    if ($trgt) {
                        $pc->set('Attr.AllowedFrameTargets', '_blank,_self,_parent,_top');
                    }
                }
                $pc = jrCore_trigger_event('jrCore', 'html_purifier', $pc);

                // Add support for HTML 5 Elements
                // See: https://github.com/kennberg/php-htmlpurfier-html5/blob/master/htmlpurifier_html5.php
                /** @noinspection PhpUndefinedMethodInspection */
                $pc->set('HTML.DefinitionID', 'html5-definitions');
                /** @noinspection PhpUndefinedMethodInspection */
                $pc->set('HTML.DefinitionRev', 1);

                /** @noinspection PhpUndefinedMethodInspection */
                if ($def = $pc->maybeGetRawHTMLDefinition()) {

                    /** @noinspection PhpUndefinedMethodInspection */
                    $def->addAttribute('a', 'data-lightbox', 'Text');

                    // Common
                    $_temp = array('section', 'nav', 'article', 'aside', 'header', 'footer', 'address');
                    foreach ($_temp as $ctag) {
                        if (isset($_tmp[$ctag])) {
                            /** @noinspection PhpUndefinedMethodInspection */
                            $def->addElement($ctag, 'Block', 'Flow', 'Common');
                        }
                    }

                    // Inline
                    $_temp = array('s', 'var', 'sub', 'sup', 'mark', 'wbr');
                    foreach ($_temp as $ctag) {
                        if (isset($_tmp[$ctag])) {
                            /** @noinspection PhpUndefinedMethodInspection */
                            $def->addElement($ctag, 'Inline', 'Inline', 'Common');
                        }
                    }

                    if (isset($_tmp['hgroup'])) {
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addElement('hgroup', 'Block', 'Required: h1 | h2 | h3 | h4 | h5 | h6', 'Common');
                    }

                    if (isset($_tmp['figure'])) {
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addElement('figure', 'Block', 'Optional: (figcaption, Flow) | (Flow, figcaption) | Flow', 'Common');
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addElement('figcaption', 'Inline', 'Flow', 'Common');
                    }

                    if (isset($_tmp['video'])) {
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addElement('video', 'Block', 'Optional: (source, Flow) | (Flow, source) | Flow', 'Common', array(
                            'src'      => 'URI',
                            'type'     => 'Text',
                            'width'    => 'Length',
                            'height'   => 'Length',
                            'poster'   => 'URI',
                            'preload'  => 'Enum#auto,metadata,none',
                            'controls' => 'Bool',
                        ));
                    }
                    if (isset($_tmp['source'])) {
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addElement('source', 'Block', 'Flow', 'Common', array(
                            'src'  => 'URI',
                            'type' => 'Text',
                        ));
                    }

                    if (isset($_tmp['ins'])) {
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addElement('ins', 'Block', 'Flow', 'Common', array('cite' => 'URI', 'datetime' => 'CDATA'));
                    }
                    if (isset($_tmp['del'])) {
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addElement('del', 'Block', 'Flow', 'Common', array('cite' => 'URI', 'datetime' => 'CDATA'));
                    }

                    if (isset($_tmp['img'])) {
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addAttribute('img', 'data-mce-src', 'Text');
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addAttribute('img', 'data-mce-json', 'Text');
                    }

                    if (isset($_tmp['iframe'])) {
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addAttribute('iframe', 'allowfullscreen', 'Bool');
                    }
                    if (isset($_tmp['table'])) {
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addAttribute('table', 'height', 'Text');
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addAttribute('td', 'border', 'Text');
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addAttribute('th', 'border', 'Text');
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addAttribute('tr', 'width', 'Text');
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addAttribute('tr', 'height', 'Text');
                        /** @noinspection PhpUndefinedMethodInspection */
                        $def->addAttribute('tr', 'border', 'Text');
                    }
                }

                /** @noinspection PhpUndefinedFieldInspection */
                $pc->autoFinalize = true;
                $pf               = new HTMLPurifier($pc);

                $string = preg_replace("@<!-- pagebreak -->@", "#!-- pagebreak --#", $string); // allow pagebreak from TinyMCE editor
                $string = @$pf->purify($string);
                $string = preg_replace("@#!-- pagebreak --#@", "<!-- pagebreak -->", $string);
            }
        }
        else {
            // Strip everything
            // DO NOT rely on strip_tags alone - it will NOT strip out Javascript BETWEEN script tags.
            if (!$pf = jrCore_get_flag('jrCore_strip_html_config')) {
                $pc = HTMLPurifier_Config::createDefault();
                $pc->set('Core.NormalizeNewlines', false);
                $pc->set('Cache.SerializerPath', jrCore_get_module_cache_dir('jrCore'));
                $pc->set('HTML.AllowedElements', '');
                $pf = new HTMLPurifier($pc);
                jrCore_set_flag('jrCore_strip_html_config', $pf);
            }
            $string = @$pf->purify($string);
            $string = jrCore_strip_tags($string);
        }
    }
    return str_replace(' />', '>', $string);
}

/**
 * Recursively run stripslashes() on a string or array
 * @param array|string $data data mixed data to strip slashes from
 * @return array|string
 */
function jrCore_stripslashes($data)
{
    if (is_array($data)) {
        foreach ($data as $k => $v) {
            $data[$k] = jrCore_stripslashes($v);
        }
        return $data;
    }
    return stripslashes($data);
}

/**
 * Get IP Address of a viewer
 * @NOTE: DO NOT USE fdebug() in this function!
 * @param bool $as_long_integer set to TRUE to return as long integer IP
 * @param string $use_ip If provided, value will be used as IP address
 * @return string Returns IP Address.
 */
function jrCore_get_ip($as_long_integer = false, $use_ip = null)
{
    $key = 'jrcore_get_ip_' . intval($as_long_integer);
    if ($use_ip === null && $tmp = jrCore_get_flag($key)) {
        return $tmp;
    }
    if (is_null($use_ip)) {
        // @note: HTTP_CF_RAY is set by Cloudflare - Cloudflare returns
        // the remote IP of the user in REMOTE_ADDR
        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR']) && !isset($_SERVER['HTTP_CF_RAY'])) {
            $use_ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        }
        else {
            // See if we are running in Demo mode (all 1's)
            if ((empty($_SERVER['REMOTE_ADDR'])) || $_SERVER['REMOTE_ADDR'] == $_SERVER['SERVER_ADDR']) {
                if (isset($_SERVER['HTTP_X_FORWARDED'])) {
                    $use_ip = $_SERVER['HTTP_X_FORWARDED'];
                }
                elseif (isset($_SERVER['HTTP_FORWARDED_FOR'])) {
                    $use_ip = $_SERVER['HTTP_FORWARDED_FOR'];
                }
                elseif (isset($_SERVER['HTTP_FORWARDED'])) {
                    $use_ip = $_SERVER['HTTP_FORWARDED'];
                }
                elseif (isset($_SERVER['HTTP_VIA'])) {
                    $use_ip = $_SERVER['HTTP_VIA'];
                }
                elseif (isset($_SERVER['HTTP_X_COMING_FROM'])) {
                    $use_ip = $_SERVER['HTTP_X_COMING_FROM'];
                }
                elseif (isset($_SERVER['HTTP_COMING_FROM'])) {
                    $use_ip = $_SERVER['HTTP_COMING_FROM'];
                }
                elseif (isset($_SERVER['HTTP_CLIENT_IP'])) {
                    $use_ip = $_SERVER['HTTP_CLIENT_IP'];
                }
                else {
                    $use_ip = (!empty($_SERVER['REMOTE_ADDR'])) ? $_SERVER['REMOTE_ADDR'] : '';
                }
            }
            else {
                $use_ip = $_SERVER['REMOTE_ADDR'];
            }
        }
    }

    // 2001:8a0:f442:9700:d909:3ebc:1352:9311, 2001:8a0:f442:9700:d909:3ebc:1352:9311
    if (strpos($use_ip, ',')) {
        list($use_ip,) = explode(',', $use_ip, 2);
    }

    $use_ip = trim($use_ip);
    if (strlen("{$use_ip}") < 7 || !filter_var($use_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
        $use_ip = '0.0.0.0';
    }

    if ($as_long_integer) {
        // Max col value is: 4294967295
        if (strpos($use_ip, ':')) {
            // IP6
            $use_ip = base_convert(substr(md5($use_ip), 8, 8), 16, 10);
        }
        else {
            // IP4
            $use_ip = sprintf('%u', ip2long($use_ip));
        }
    }
    jrCore_set_flag($key, $use_ip);
    return $use_ip;
}

/**
 * Get substring of a string based on position and separator
 * @param string $string Input string to get field from
 * @param string $field numerical position in string to return, or "END" to return last word of string. If $field is negative, then counting begins from the end of the string backwards.
 * @param string $sep Field separator for string
 * @return mixed Returns field from string, or false on error
 */
function jrCore_string_field($string, $field, $sep = ' ')
{
    if ($sep == ' ') {
        // first - convert tabs to spaces
        $string = str_replace("\t", ' ', $string);
    }
    // see if they want the LAST field
    if ($field == 'NF' || $field == 'END') {
        $out = explode($sep, $string);
        return end($out);
    }
    // numerical (positive int)
    elseif ($field > 0) {
        $i = 1;
        foreach (@explode($sep, $string) as $v) {
            if (strlen($v) >= 1) {
                $_out[$i] = trim($v);
                if (isset($_out[$field])) {
                    return $_out[$field];
                }
                $i++;
            }
        }
    }
    // negative (backwards from end of string)
    else {
        $field = str_replace('-', '', $field);
        $i     = 1;
        foreach (@array_reverse(explode($sep, $string)) as $v) {
            if (strlen($v) >= 1) {
                $_out[$i] = trim($v);
                if (isset($_out[$field])) {
                    return $_out[$field];
                }
                $i++;
            }
        }
    }
    return false;
}

/**
 * Display debug information on screen
 * @return bool Returns True/False on Success/Fail.
 */
function debug()
{
    $out = jrCore_get_debug_output(func_get_args());
    if (php_sapi_name() == "cli") {
        echo $out;
    }
    else {
        $out = str_replace("\n", '<br>', jrCore_entity_string($out));
        echo '<style>.jrCore_debug{font-family:Monaco,"Courier New",Courier,monospace;font-size:11px;font-weight:normal;text-transform:none;text-align:left;word-break:break-all;white-space:pre-wrap}</style><div class="jrCore_debug">' . $out . '</div>';
        flush();
    }
    return true;
}

/**
 * Tell fdebug() to only log for specified IP address
 * @param $ip string IP Address
 * @return bool
 */
function fdebug_only_ip($ip)
{
    return jrCore_set_flag('fdebug_only_ip', $ip);
}

/**
 * Don't log fdebug() entries when the URI is an image
 * @return bool
 */
function fdebug_ignore_images()
{
    return jrCore_set_flag('fdebug_ignore_images', 1);
}

/**
 * Don't log fdebug() entries when the URI is an AJAX URI
 * @return bool
 */
function fdebug_ignore_ajax()
{
    return jrCore_set_flag('fdebug_ignore_ajax', 1);
}

/**
 * Log debug info to the data/logs/debug_log
 * @return bool
 */
function fdebug()
{
    // prevent recursion
    if (jrCore_get_flag('jrcore_in_fdebug')) {
        return true;
    }
    jrCore_set_flag('jrcore_in_fdebug', 1);
    $uip = jrCore_get_ip();
    if ($oip = jrCore_get_flag('fdebug_only_ip')) {
        if ($uip != $oip) {
            jrCore_delete_flag('jrcore_in_fdebug');
            return true;
        }
    }

    // Check if we are ignoring images
    if (jrCore_get_flag('fdebug_ignore_images') && jrCore_is_image_request()) {
        jrCore_delete_flag('jrcore_in_fdebug');
        return true;
    }

    // Check if we are ignoring AJAX
    if (jrCore_get_flag('fdebug_ignore_ajax') && jrCore_is_ajax_request()) {
        jrCore_delete_flag('jrcore_in_fdebug');
        return true;
    }

    $out = jrCore_get_debug_output(func_get_args());
    $out = jrCore_trigger_event('jrCore', 'debug_log', $out);
    if (is_string($out)) {
        jrCore_start_timer('filesystem');
        $tmp = fopen(APP_DIR . "/data/logs/debug_log", 'ab');
        if ($tmp) {
            flock($tmp, LOCK_EX);
            fwrite($tmp, $out);
            flock($tmp, LOCK_UN);
            fclose($tmp);
        }
        jrCore_stop_timer('filesystem');
    }
    jrCore_delete_flag('jrcore_in_fdebug');
    return true;
}

/**
 * Get output to use in debug() or fdebug()
 * @param array $_args
 * @return string
 */
function jrCore_get_debug_output($_args)
{
    global $_user;
    if (!is_array($_args)) {
        return '||';
    }
    $uip = jrCore_get_ip();

    // 0.81072800 1393853047
    $now = explode(' ', microtime());
    $msc = $now[0];
    $now = $now[1] + $now[0];

    // Have we run before?
    $beg = jrCore_get_flag('jrcore_process_start_time');
    $dif = round(($now - $beg), 3);
    $mem = jrCore_format_size(memory_get_usage(true));
    $fmt = date('c');
    $out = '';
    $usr = (!empty($_user['user_name'])) ? "-(user: {$_user['user_name']})" : '';
    $sid = session_id();
    $sid = (!empty($sid)) ? $sid : 0;
    $url = jrCore_get_current_url();
    foreach ($_args as $arg) {
        $out .= PHP_EOL . "({$fmt}.{$msc} : {$dif})-(mem: {$mem})-(pid: " . getmypid() . ")-(ip: {$uip})-(sid: {$sid}){$usr}-(url: {$url})" . PHP_EOL;
        if (is_array($arg) || is_object($arg)) {
            $out .= print_r($arg, true);
        }
        else {
            $out .= "|{$arg}|" . PHP_EOL;
        }
    }
    return $out;
}

/**
 * Create a directory if needed and verify Core permissions
 * @param string $dir
 * @param bool $recursive
 * @return bool
 */
function jrCore_create_directory($dir, $recursive = false)
{
    global $_conf;
    jrCore_start_timer('filesystem');
    if (!is_dir($dir)) {
        if (!@mkdir($dir, $_conf['jrCore_dir_perms'], $recursive)) {
            jrCore_stop_timer('filesystem');
            return false;
        }
    }
    else {
        @chmod($dir, $_conf['jrCore_dir_perms']);
    }
    jrCore_stop_timer('filesystem');
    return true;
}

/**
 * Recursively copy one directory to another
 * @param string $source Source Directory (directory to copy to)
 * @param string $destination Destination directory (directory to copy from)
 * @param array $_replace of K/V pairs for replacement within copied files
 * @return bool
 */
function jrCore_copy_dir_recursive($source, $destination, $_replace = null)
{
    global $_conf;
    if (!is_dir($source)) {
        return false;
    }
    if (!is_dir($destination)) {
        jrCore_create_directory($destination, true);
    }
    jrCore_start_timer('filesystem');
    $f = opendir($source);
    if ($f) {
        while ($file = readdir($f)) {
            if ($file == '.' || $file == '..') {
                continue;
            }
            if (is_dir("{$source}/{$file}")) {
                jrCore_copy_dir_recursive("{$source}/{$file}", "{$destination}/{$file}", $_replace);
            }
            else {
                $rep = false;
                switch (jrCore_file_extension($file)) {
                    case 'tpl':
                    case 'css':
                    case 'php':
                    case 'cfg':
                    case 'js':
                    case 'htm':
                    case 'html':
                    case 'xml':
                        $rep = true;
                        break;
                }
                if (!$rep || !is_array($_replace)) {
                    // Straight Copy (possibly to new name)
                    if (copy("{$source}/{$file}", "{$destination}/{$file}")) {
                        chmod("{$destination}/{$file}", $_conf['jrCore_file_perms']);
                    }
                }
                else {
                    // Key => Value replacements in destination
                    $tmp = jrCore_file_get_contents("{$source}/{$file}");
                    $tmp = strtr($tmp, $_replace);
                    $fnm = strtr($file, $_replace);
                    jrCore_write_to_file("{$destination}/{$fnm}", $tmp);
                    unset($tmp);
                }
            }
        }
        closedir($f);
    }
    jrCore_stop_timer('filesystem');
    return true;
}

/**
 * Delete all content from a directory, including sub directories, optionally aged
 * @param string $dir Directory to remove all files and sub directories inside.
 * @param bool $cache_check default directory must be in cache directory
 * @param int $safe_seconds Files/Directories younger than "safe_seconds" will be ignored
 * @param bool $delete_dir set to TRUE to delete directory as well
 * @return bool
 */
function jrCore_delete_dir_contents($dir, $cache_check = true, $safe_seconds = 0, $delete_dir = false)
{
    if (!is_dir($dir)) {
        return false;
    }
    if ($cache_check && strpos($dir, APP_DIR . '/data/cache') !== 0) {
        jrCore_logger('CRI', "core: invalid directory: " . str_replace(APP_DIR . '/', '', $dir) . " - must be a valid directory within the data/cache directory");
        return false;
    }
    // We never delete anything outside the APP_DIR
    if (strpos($dir, APP_DIR) !== 0) {
        jrCore_logger('CRI', "core: attempt to delete directory outside of APP_DIR: {$dir}");
        return false;
    }
    // We do not allow relative dirs...
    if (strpos(' ' . $dir, '..')) {
        jrCore_logger('CRI', "core: attempt to delete directory using relative path: {$dir}");
        return false;
    }
    $secs = false;
    if (intval($safe_seconds) > 0) {
        $secs = (time() - intval($safe_seconds));
    }

    // and now do our deletion
    $cnt = 0;
    jrCore_start_timer('filesystem');
    if ($h = opendir($dir)) {
        while (($file = readdir($h)) !== false) {
            if ($file == '.' || $file == '..') {
                continue;
            }
            if (is_dir("{$dir}/{$file}") && !is_link("{$dir}/{$file}")) {
                $cnt += jrCore_delete_dir_contents("{$dir}/{$file}", false, $safe_seconds, true);
            }
            else {
                if ($secs) {
                    $_tmp = stat("{$dir}/{$file}");
                    if (isset($_tmp['mtime']) && $_tmp['mtime'] < $secs) {
                        unlink("{$dir}/{$file}");  // OK
                        $cnt++;
                    }
                }
                else {
                    unlink("{$dir}/{$file}");  // OK
                    $cnt++;
                }
            }
        }
        closedir($h);
    }
    if ($delete_dir) {
        rmdir($dir);
        $cnt++;
    }
    jrCore_stop_timer('filesystem');
    return $cnt;
}

/**
 * Recursively get all files and directories within a directory
 * @param string $dir Directory to get files for
 * @param bool $finish flag for final dir check
 * @return array|false
 */
function jrCore_get_directory_files($dir, $finish = true)
{
    $_out = false;
    $dir  = rtrim(trim($dir), '/');
    if ($h = opendir($dir)) {
        $_out = array();
        while (false !== ($file = readdir($h))) {
            if ($file == '.' || $file == '..') {
                continue;
            }
            if (is_link($dir . '/' . $file)) {
                $_out[] = "{$dir}/{$file}";
            }
            elseif (is_dir($dir . '/' . $file)) {
                $_tmp = jrCore_get_directory_files("{$dir}/{$file}", false);
                if (isset($_tmp) && is_array($_tmp)) {
                    $_out = array_merge($_out, $_tmp);
                }
            }
            else {
                $_out[] = "{$dir}/{$file}";
            }
        }
        closedir($h);
    }
    if ($finish && isset($_out) && is_array($_out)) {
        foreach ($_out as $k => $full_file) {
            $_out[$full_file] = str_replace("{$dir}/", '', $full_file);
            unset($_out[$k]);
        }
    }
    return $_out;
}

/**
 * Convert a string to a File System safe string
 * @param string $string String to return URL encoded
 * @return string
 */
function jrCore_file_string($string)
{
    return rawurlencode($string);
}

/**
 * Convert a string to a URL Safe string - used to generate slugs
 * @param string $string String to convert URLs in
 * @param string $allowed list of allowed characters
 * @return string
 */
function jrCore_url_string($string, $allowed = '')
{
    // Are we already a url encoded string?
    if (strpos(' ' . $string, '%') && !strpos(' ' . rawurldecode($string), '%')) {
        // We are already rawurlencoded
        return $string;
    }
    setlocale(LC_ALL, 'en_US.UTF8');
    $str = str_replace(array('İ', 'ı'), 'i', trim(urldecode($string)));
    if (class_exists('Transliterator')) {
        $_saved = array(
            '-' => 'xxjrdashxx',
            '_' => 'xxjrdashxx'
        );
        if (!empty($allowed)) {
            $_saved[$allowed] = 'xxjrallowedxx';
        }
        $str = str_replace(array_keys($_saved), $_saved, $str);
        $str = \Transliterator::createFromRules(':: Any-Latin;:: NFD;:: [:Nonspacing Mark:] Remove;:: NFC;:: [:Punctuation:] Remove;:: Lower();[:Separator:] > \'-\'')->transliterate($str);
        $str = rawurlencode(str_replace($_saved, array_keys($_saved), $str));
    }
    else {
        // @ for "Detected an illegal character in input string"
        $str = @iconv('UTF-8', 'ASCII//TRANSLIT', substr(trim($str), 0, 128));
        $str = preg_replace("/[^a-zA-Z0-9{$allowed}\/_| -]/", '', $str);
        $str = strtolower(trim($str, '-'));
        $str = trim(trim(preg_replace("/[\/| -]+/", '-', $str)), '-');
    }
    if (strlen($str) === 0) {
        // We may have removed everything - rawurlencode
        $str = rawurlencode(jrCore_str_to_lower(str_replace(array('"', "'", ' ', '&', '@', '/', '[', ']', '(', ')'), '-', $string)));
    }
    return trim(preg_replace('/[-+_]/', '-', $str), '-');
}

/**
 * Write data to a file with file locking
 * @param string $file File to write to
 * @param string $data Data to write to file
 * @param string $mode Mode - can be "overwrite" or "append"
 * @return bool
 */
function jrCore_write_to_file($file, $data = null, $mode = 'overwrite')
{
    if (is_null($data)) {
        return false;
    }
    $perms = jrCore_get_config_value('jrCore', 'file_perms', 0644);
    ignore_user_abort(true);
    jrCore_start_timer('filesystem');
    if ($mode == 'overwrite') {
        $f = fopen($file, 'wb');
    }
    else {
        $f = fopen($file, 'ab');
    }
    $ret = false;
    if ($f && is_resource($f)) {
        flock($f, LOCK_EX);
        $ret = fwrite($f, $data);
        flock($f, LOCK_UN);
        fclose($f);
        chmod($file, $perms);
    }
    jrCore_stop_timer('filesystem');
    ignore_user_abort(false);
    return $ret;
}

/**
 * Copy a file in chunks to use less memory
 * @param string $source_path
 * @param string $target_path
 * @param int $chunk_size copy chunk size in MB
 * @return bool
 */
function jrCore_chunked_copy($source_path, $target_path, $chunk_size = 1)
{
    if ($source_path == $target_path) {
        return false;
    }
    $ck = (intval($chunk_size) * 1048576);
    jrCore_start_timer('filesystem');
    if (!$fs = fopen($source_path, 'rb')) {
        jrCore_stop_timer('filesystem');
        return false;
    }
    if (!$ft = fopen($target_path, 'w')) {
        jrCore_stop_timer('filesystem');
        return false;
    }
    while (!feof($fs)) {
        fwrite($ft, fread($fs, $ck));
    }
    fclose($fs);
    fclose($ft);
    jrCore_stop_timer('filesystem');
    return true;
}

/**
 * Get file extension from a file
 * Returns file extension in lower case!
 * @param string $file file string file name to return extension for
 * @return string
 */
function jrCore_file_extension($file)
{
    if (strpos($file, '.')) {
        $_tmp = explode('.', trim($file));
        if ($_tmp && is_array($_tmp)) {
            $ext = array_pop($_tmp);
            if (strpos($ext, '?')) {
                list($ext,) = explode('?', $ext, 2);
            }
            return jrCore_str_to_lower(trim($ext));
        }
    }
    return false;
}

/**
 * Get file extension for a given Mime-Type
 * This function relies on the /etc/mime.types file being readable by the web user
 * File extension is returned lower case
 * @param string $type Mime-Type
 * @return bool|string Returns extension if mime-type found, false if not found/known
 */
function jrCore_file_extension_from_mime_type($type)
{
    $ext = false;
    // Some quick ones...
    switch ($type) {
        case 'image/jpeg':
            $ext = 'jpg';
            break;
        case 'image/png':
        case 'image/webp':
            $ext = 'png';
            break;
        case 'image/gif':
            $ext = 'gif';
            break;
    }
    if (!$ext) {
        if (strpos($type, '/') && is_readable('/etc/mime.types')) {
            $_mim = jrCore_get_flag('jrcore_loaded_mime_types');
            if (!$_mim) {
                jrCore_start_timer('filesystem');
                $_mim = file('/etc/mime.types');
                jrCore_stop_timer('filesystem');
                jrCore_set_flag('jrcore_loaded_mime_types', $_mim);
            }
            foreach ($_mim as $line) {
                if (strpos($line, $type) === 0) {
                    $ext = trim(jrCore_string_field($line, 'NF'));
                    if ($ext != $type) {
                        switch ($ext) {
                            case 'jpe':
                            case 'jpeg':
                            case 'jfif':
                                $ext = 'jpg';
                                break;
                        }
                        break;
                    }
                }
            }
        }
    }
    if (!$ext) {
        // Go to our built in mime list
        $_ms = array_flip(jrCore_get_mime_list());
        if (isset($_ms[$type])) {
            return $_ms[$type];
        }
    }
    return $ext;
}

/**
 * Return an array of file extensions => mime types
 * @return array
 */
function jrCore_get_mime_list()
{
    return array(
        'txt'  => 'text/plain',
        'htm'  => 'text/html',
        'html' => 'text/html',
        'php'  => 'text/html',
        'css'  => 'text/css',
        'js'   => 'application/javascript',
        'json' => 'application/json',
        'xml'  => 'application/xml',
        'swf'  => 'application/x-shockwave-flash',

        // images
        'png'  => 'image/png',
        'jpe'  => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'jpg'  => 'image/jpeg',
        'jfif' => 'image/jpeg',
        'gif'  => 'image/gif',
        'bmp'  => 'image/bmp',
        'ico'  => 'image/x-icon',
        'tiff' => 'image/tiff',
        'tif'  => 'image/tiff',
        'svg'  => 'image/svg+xml',
        'svgz' => 'image/svg+xml',
        'webp' => 'image/webp',

        // archives
        'zip'  => 'application/zip',
        'rar'  => 'application/x-rar-compressed',
        'exe'  => 'application/x-msdownload',
        'msi'  => 'application/x-msdownload',
        'cab'  => 'application/vnd.ms-cab-compressed',

        // audio
        'mp3'  => 'audio/mpeg',
        'm4a'  => 'audio/mp4',
        'wma'  => 'audio/x-ms-wma',
        'ogg'  => 'audio/ogg',
        'flac' => 'audio/x-flac',
        'wav'  => 'audio/wav',
        'aac'  => 'application/aac',

        // video
        'flv'  => 'video/x-flv',
        'f4v'  => 'video/x-flv',
        'qt'   => 'video/quicktime',
        'mov'  => 'video/quicktime',
        'ogv'  => 'video/ogg',
        'm4v'  => 'video/x-m4v',
        'mpg'  => 'video/mpeg',
        'mp4'  => 'video/mp4',
        'avi'  => 'video/avi',
        '3gp'  => 'video/3gpp',
        '3g2'  => 'video/3gpp2',
        'wmv'  => 'video/x-ms-wmv',

        // adobe
        'pdf'  => 'application/pdf',
        'psd'  => 'image/vnd.adobe.photoshop',
        'ai'   => 'application/postscript',
        'eps'  => 'application/postscript',
        'ps'   => 'application/postscript',

        // ms office
        'doc'  => 'application/msword',
        'rtf'  => 'application/rtf',
        'xls'  => 'application/vnd.ms-excel',
        'ppt'  => 'application/vnd.ms-powerpoint',

        // open office
        'odt'  => 'application/vnd.oasis.opendocument.text',
        'ods'  => 'application/vnd.oasis.opendocument.spreadsheet'
    );
}

/**
 * Get Mime-Type for a file
 *
 * @param string $file File Name
 * @return string
 */
function jrCore_mime_type($file)
{
    // Go on file extension - fastest
    $_ms = jrCore_get_mime_list();
    $ext = jrCore_file_extension($file);
    if (strlen($ext) > 0 && isset($_ms[$ext])) {
        return $_ms[$ext];
    }

    // mime_content_type is deprecated
    if (is_file($file)) {
        if (function_exists('mime_content_type')) {
            jrCore_start_timer('filesystem');
            $mime = mime_content_type($file);
            jrCore_stop_timer('filesystem');
            return $mime;
        }
        elseif (function_exists('finfo_open')) {
            jrCore_start_timer('filesystem');
            $finfo = @finfo_open(FILEINFO_MIME);
            $mime  = @finfo_file($finfo, $file);
            @finfo_close($finfo);
            jrCore_stop_timer('filesystem');
            if (!empty($mime)) {
                if (strpos($mime, '; ')) {
                    $mime = substr($mime, 0, strpos($mime, '; '));
                }
                return $mime;
            }
        }

        // Last check - see if we can do a system call to get the extension
        if (function_exists('system')) {
            ob_start();
            jrCore_start_timer('filesystem');
            $mime = @system("file -bi {$file}");
            jrCore_stop_timer('filesystem');
            ob_end_clean();
            if (!empty($mime)) {
                return $mime;
            }
        }
    }

    // Default
    return 'application/binary';
}

/**
 * Return the number of seconds OFF from UTC a timezone is
 * @param int $timestamp Epoch time stamp
 * @param bool $adjust set to FALSE to not adjust for DLS
 * @param string $zone Timezone
 * @return int
 */
function jrCore_get_timezone_dls_offset($timestamp, $adjust, $zone = null)
{
    if (empty($zone)) {
        $zone = jrCore_get_config_value('jrCore', 'system_timezone', 'UTC');
    }
    if ($zone != 'UTC') {
        try {
            $tz = new DateTimeZone($zone);
        }
        catch (Exception $e) {
            return 0;
        }
        $_t = $tz->getTransitions($timestamp, $timestamp);
        if (is_array($_t) && !empty($_t[0]['offset'])) {
            // Are we adjust for DLS?
            if (!$adjust && $_t[0]['isdst'] != 1) {
                $add = 0;
                switch ($zone) {
                    case 'America/Caracas':
                    case 'America/St_Johns':
                    case 'Asia/Tehran':
                    case 'Asia/Kabul':
                    case 'Asia/Kolkata':
                    case 'Asia/Rangoon':
                    case 'Australia/Darwin':
                        $add = 1800;
                        break;
                    case 'Asia/Katmandu':
                        $add = 2700;
                        break;
                }
                $_t[0]['offset'] += $add;
            }
            if ($custom = jrCore_get_config_value('jrCore', 'time_offset_seconds', false)) {
                $_t[0]['offset'] += $custom;
            }
            return $_t[0]['offset'];
        }
    }
    return 0;
}

/**
 * Get all installed locales on server
 * @return array
 */
function jrCore_get_installed_locales()
{
    $_lc = false;
    ob_start();
    jrCore_start_timer('filesystem');
    @system('locale -a');
    jrCore_stop_timer('filesystem');
    $out = ob_get_contents();
    ob_end_clean();
    if ($out && strlen($out) > 5) {
        $_lc = array();
        foreach (explode("\n", $out) as $l) {
            if (stripos($l, 'UTF')) {
                $k       = str_ireplace(array('.utf8', '.UTF-8'), '', str_replace('_', '-', $l));
                $_lc[$k] = $l;
            }
        }
    }
    return $_lc;
}

/**
 * Get timezone strings used in Control Panel
 * @return array
 */
function jrCore_get_timezone_strings()
{
    return array(
        'Kwajalein'              => 'GMT -12.00 - Kwajalein',
        'Pacific/Samoa'          => 'GMT -11:00 - Samoa',
        'Pacific/Honolulu'       => 'GMT -10:00 - Hawaii',
        'America/Anchorage'      => 'GMT -09:00 - Alaska',
        'America/Vancouver'      => 'GMT -08:00 - (PST) USA, Canada',
        'America/Denver'         => 'GMT -07:00 - (MST) USA, Canada',
        'America/Chicago'        => 'GMT -06:00 - (CST) USA, Canada, Mexico',
        'America/New_York'       => 'GMT -05:00 - (EST) USA, Canada',
        'America/Caracas'        => 'GMT -04:30 - Caracas',
        'America/Halifax'        => 'GMT -04:00 - (AST) Canada',
        'America/St_Johns'       => 'GMT -03:30 - (NST) Newfoundland',
        'America/Sao_Paulo'      => 'GMT -03:00 - Brazil, Buenos Aires, Sao Paulo',
        'Atlantic/South_Georgia' => 'GMT -02:00 - South Georgia',
        'Atlantic/Azores'        => 'GMT -01:00 - Azores',
        'UTC'                    => 'UTC -00:00 - (UTC) Coordinated Universal Time',
        'Europe/Dublin'          => 'GMT -00:00 - (GMT) Lisbon, London, Dublin',
        'Europe/Belgrade'        => 'GMT +01:00 - Madrid, Paris, Belgrade',
        'Europe/Helsinki'        => 'GMT +02:00 - Cairo, Helsinki, South Africa',
        'Asia/Kuwait'            => 'GMT +03:00 - Baghdad, Moscow, St Petersburg',
        'Asia/Tehran'            => 'GMT +03:30 - Tehran',
        'Asia/Muscat'            => 'GMT +04:00 - Abu Dhabi, Tbilisi',
        'Asia/Kabul'             => 'GMT +04:30 - Kabul',
        'Asia/Yekaterinburg'     => 'GMT +05:00 - Karachi, Lahore, Tashkent',
        'Asia/Kolkata'           => 'GMT +05:30 - (IST) Bangalore, Mumbai, New Delhi',
        'Asia/Katmandu'          => 'GMT +05:45 - Katmandu',
        'Asia/Dhaka'             => 'GMT +06:00 - Dhaka',
        'Asia/Rangoon'           => 'GMT +06:30 - Yangon, Rangoon',
        'Asia/Krasnoyarsk'       => 'GMT +07:00 - Bangkok, Jakarta',
        'Asia/Brunei'            => 'GMT +08:00 - Beijing, Hong Kong, Singapore',
        'Asia/Seoul'             => 'GMT +09:00 - Osaka, Seoul, Tokyo',
        'Australia/Darwin'       => 'GMT +09:30 - Adelaide, Darwin',
        'Australia/Canberra'     => 'GMT +10:00 - Brisbane, Canberra, Vladivostok',
        'Asia/Magadan'           => 'GMT +11:00 - Sydney, Melboune',
        'Pacific/Fiji'           => 'GMT +12:00 - Auckland, Fiji',
        'Pacific/Tongatapu'      => 'GMT +13:00 - Tongatupu'
    );
}

/**
 * Save a URL to a "stack" of URLs by name
 * avoid using as it is not cross-domain compatible
 * @param string $tag Text Tag for memory URL
 * @param string $url URL to remember
 * @return bool
 */
function jrCore_create_memory_url($tag, $url = 'referrer')
{
    if (!$url || strlen($url) === 0 || $url === 'referrer' || !jrCore_is_local_url($url)) {
        $url = jrCore_get_local_referrer();
    }
    if (!$_urls = jrCore_get_user_session_key('jrcore_memory_urls')) {
        $_urls = array();
    }
    $_urls[$tag] = $url;
    return jrCore_set_user_session_key('jrcore_memory_urls', $_urls);
}

/**
 * Get a URL from the memory stack by name
 * @param string $tag Text Tag for memory URL
 * @param bool|string $url Value/URL to return if not set
 * @return string
 */
function jrCore_get_memory_url($tag, $url = 'referrer')
{
    if ($_urls = jrCore_get_user_session_key('jrcore_memory_urls')) {
        if (!empty($_urls[$tag])) {
            return $_urls[$tag];
        }
    }
    return $url;
}

/**
 * Get all memory URLs for a user
 * @return bool|array
 */
function jrCore_get_all_memory_urls()
{
    return jrCore_get_user_session_key('jrcore_memory_urls');
}

/**
 * Delete a URL from the memory stack by name
 * @param string $tag Text Tag for memory URL
 * @return true
 */
function jrCore_delete_memory_url($tag)
{
    if ($_urls = jrCore_get_user_session_key('jrcore_memory_urls')) {
        if (isset($_urls[$tag])) {
            unset($_urls[$tag]);
            jrCore_set_user_session_key('jrcore_memory_urls', $_urls);
        }
    }
    return true;
}

/**
 * Delete all memory URLs
 * @return bool
 */
function jrCore_delete_all_memory_urls()
{
    return jrCore_delete_user_session_key('jrcore_memory_urls');
}

/**
 * Get max allowed upload size as defined by PHP and the user's Quota
 * @param int $quota_max Max as set in Profile Quota
 * @return int Returns Max Upload in bytes
 */
function jrCore_get_max_allowed_upload($quota_max = 0)
{
    // figure max upload form size
    $php_pmax = (int) ini_get('post_max_size');
    $php_umax = (int) ini_get('upload_max_filesize');
    $val      = ($php_pmax > $php_umax) ? $php_umax : $php_pmax;

    // For handling large file uploads we must use the following logic to arrive at our
    // max allowed upload size: Use 1/2 memory_limit, and if $val is smaller use that
    $php_mmax = ceil(intval(str_replace('M', '', ini_get('memory_limit'))) / 2);
    $val      = ($php_mmax > $val) ? $val : $php_mmax;
    $val      = ($val * 1048576);

    // Check if we are getting a quota restricted level
    if (is_numeric($quota_max) && $quota_max > 0 && $quota_max < $val) {
        $val = $quota_max;
    }
    return $val;
}

/**
 * Array of upload sizes to be used in a Select field
 * @return array Array of upload sizes
 */
function jrCore_get_upload_sizes()
{
    $s_max = (int) jrCore_get_max_allowed_upload(false);
    $_qmem = array();
    $_memr = array(1, 2, 4, 8, 16, 24, 32, 48, 64, 72, 96, 100, 128, 160, 200, 256, 300, 350, 384, 400, 500, 512, 600, 640, 700, 768, 800, 896, 1000, 1024, 1536, 2048, 3072, 4096);
    foreach ($_memr as $m) {
        $v = $m * 1048576;
        if ($v < $s_max) {
            $_qmem[$v] = jrCore_format_size($v);
        }
    }
    $_qmem[$s_max] = jrCore_format_size($s_max) . " - max allowed";
    return $_qmem;
}

/**
 * returns a URL if the referring URL is the user's own profile,
 * @param string $default URL to return if referrer is NOT from user's profile
 * @return string
 */
function jrCore_is_profile_referrer($default = 'referrer')
{
    global $_conf, $_user;
    jrCore_delete_flag('jrcore_is_profile_referrer');
    if (isset($_SERVER['HTTP_REFERER']) && !strpos(' ' . $_SERVER['HTTP_REFERER'], $_conf['jrCore_base_url'])) {
        // Is this a mapped domain?
        return jrCore_get_local_referrer();
    }
    elseif (jrUser_is_logged_in()) {
        if ($url = jrUser_get_saved_url_location($_user['_user_id'])) {
            jrCore_set_flag('jrcore_is_profile_referrer', 1);
            return $url;
        }
    }
    $url = jrCore_get_local_referrer();
    if (isset($_user['profile_url']) && strpos($url, "{$_conf['jrCore_base_url']}/{$_user['profile_url']}") === 0) {
        return $url;
    }
    return $default;
}

/**
 * Set a custom Send Header
 * @param string $header Header to set
 * @param string $event Optional CloudClient event
 * @return bool
 */
function jrCore_set_custom_header($header, $event = null)
{
    if (!$_tmp = jrCore_get_flag('jrcore_set_custom_header')) {
        $_tmp = array();
    }
    $_tmp[] = $header;
    jrCore_set_flag('jrcore_set_custom_header', $_tmp);
    if (!empty($event)) {
        jrCore_record_event($event);
    }
    return true;
}

/**
 * Strip all strings that are NOT ASCII characters
 * @param $string
 * @return string
 */
function jrCore_strip_non_ascii($string)
{
    return preg_replace('/[[:^print:]]/', '', $string);
}

/**
 * Strips a string of all non UTF8 characters
 * @param string $string String to strip UTF8 characters from
 * @return string
 */
function jrCore_strip_non_utf8($string)
{
    // strip 2 byte sequences, as well as characters above U+10000
    // $string = iconv('UTF-8', 'UTF-8//IGNORE', $string);
    // @note: iconv() does not work on Alpine linux / non-GNU
    $string = mb_convert_encoding($string, 'UTF-8', 'UTF-8');
    $string = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x19\x7F]+' .
        '|([\xC0\xC1]|[\xF0-\xFF])[\x80-\xBF]*|[\xC2-\xDF]((?![\x80-\xBF])|[\x80-\xBF]{2,})' .
        '|[\xE0-\xEF](([\x80-\xBF](?![\x80-\xBF]))|(?![\x80-\xBF]{2})|[\x80-\xBF]{3,})/S', '', $string);
    // strip long 3 byte sequences and UTF-16 surrogates
    return preg_replace('/\xE0[\x80-\x9F][\x80-\xBF]|\xED[\xA0-\xBF][\x80-\xBF]/S', '', $string);
}

/**
 * Run a system call with a Timeout
 * @param string $cmd Shell command to run
 * @param int $timeout Max length process can run in seconds
 * @param string $stdout
 * @param string $stderr
 * @return bool|string
 */
function jrCore_run_timed_command($cmd, $timeout, &$stdout, &$stderr)
{
    if (!jrCore_checktype($timeout, 'number_nz')) {
        return false;
    }
    $pipes   = array();
    $process = proc_open(
        $cmd,
        array(array('pipe', 'r'), array('pipe', 'w'), array('pipe', 'w')),
        $pipes
    );
    $start   = microtime(true);
    $stdout  = '';
    $stderr  = '';

    if (is_resource($process)) {
        stream_set_blocking($pipes[0], 0);
        stream_set_blocking($pipes[1], 0);
        stream_set_blocking($pipes[2], 0);
        fwrite($pipes[0], '');
        fclose($pipes[0]);
    }
    while (is_resource($process)) {
        $stdout .= stream_get_contents($pipes[1]);
        $stderr .= stream_get_contents($pipes[2]);
        if ((microtime(true) - $start) >= $timeout) {
            proc_close($process);
            return 'SIGTERM';
        }
        $status = proc_get_status($process);
        if (!$status['running']) {
            fclose($pipes[1]);
            fclose($pipes[2]);
            proc_close($process);
            return $status['exitcode'];
        }
        usleep(100000);
    }
    return true;
}

/**
 * Get Server Operating System
 * @return string
 */
function jrCore_get_server_os()
{
    $sos = @php_uname();
    // Are we 64 or 32 bit?
    $bit = 32;
    if (stripos($sos, 'x86_64')) {
        $bit = 64;
    }
    // macOS | Mac OS X
    if (stristr(PHP_OS, 'darwin')) {
        ob_start();
        @system('sw_vers -productVersion');
        $out = ob_get_clean();
        if ($out && strpos($out, '.')) {
            list($maj,) = explode('.', $out, 2);
            if ($maj < 11) {
                $sos = 'Mac OS X ' . $out;
            }
            else {
                $sos = 'macOS ' . $out;
            }
        }
    }
    // Linux standardized to /etc/os-release
    elseif (is_file('/etc/os-release')) {
        if ($_tmp = file('/etc/os-release')) {
            // PRETTY_NAME="Alpine Linux v3.14"
            foreach ($_tmp as $line) {
                if (stripos(' ' . $line, 'PRETTY_NAME')) {
                    list(, $sos) = explode('=', $line);
                    $sos = trim(trim(trim($sos), '"'));
                    break;
                }
            }
        }
    }
    else {
        ob_start();
        @system('lsb_release -a');
        $out = ob_get_clean();
        if ($out && stripos($out, 'description')) {
            $out = explode("\n", $out);
            if ($out && is_array($out)) {
                foreach ($out as $line) {
                    $line = trim($line);
                    if (stripos($line, 'description') === 0) {
                        list(, $sos) = explode(':', $line);
                        $sos = trim($sos);
                        break;
                    }
                }
            }
        }
    }
    return $sos . ' ' . $bit . 'bit';
}

/**
 * Returns RAM usage on server
 * @return mixed
 */
function jrCore_get_system_memory()
{
    if (!function_exists('system')) {
        return false;
    }
    $key = "jrcore_get_system_memory";
    $out = jrCore_is_cached('jrCore', $key);
    if ($out) {
        if ($out == 'bad') {
            return false;
        }
        return json_decode($out, true);
    }

    // See what system we are on
    ob_start();
    $_out = array();
    // Mac OS X
    if (stristr(PHP_OS, 'darwin')) {

        ob_start();
        @system('vm_stat', $ret);
        $out = ob_get_clean();
        if ($ret != 0 || strlen($out) === 0) {
            jrCore_add_to_cache('jrCore', $key, 'bad', 30);
            return false;
        }
        $_tmp = explode("\n", $out);
        if (!isset($_tmp) || !is_array($_tmp)) {
            jrCore_add_to_cache('jrCore', $key, 'bad', 30);
            return false;
        }
        $used = 0;
        $free = 0;
        $inac = 0;
        $spec = 0;
        $page = (strpos($out, '16384 bytes')) ? 16384 : 4096;
        foreach ($_tmp as $line) {
            $line = rtrim(trim($line), '.');
            if (strpos($line, 'free')) {
                $free += (intval(jrCore_string_field($line, 'END')) * $page);
            }
            elseif (strpos($line, 'inactive') || strpos($line, 'purgeable')) {
                $inac += (int) jrCore_string_field($line, 'END') * $page;
            }
            elseif (strpos($line, 'speculative')) {
                $spec += (int) jrCore_string_field($line, 'END') * $page;
            }
            elseif (strpos($line, 'wired') || strpos($line, 'active')) {
                $used += (int) jrCore_string_field($line, 'END') * $page;
            }
        }
        $_out['memory_used'] = $used;
        $_out['memory_free'] = ($free + $spec + $inac);

        // Get total Memory
        ob_start();
        @system('sysctl hw.memsize', $ret);
        $out = ob_get_clean();
        if ($out && strlen($out) > 0) {
            // hw.memsize: 8589934592
            $_out['memory_total'] = intval(jrCore_string_field($out, 'END'));
        }
        if (!isset($_out['memory_total'])) {
            $_out['memory_total'] = ($_out['memory_used'] + $_out['memory_free']);
        }
    }
    else {
        //              total       used       free     shared    buffers     cached
        // Mem:       1033748     997196      36552          0     402108      83156
        // -/+ buffers/cache:     511932     521816
        // Swap:      1379872          0    1379872

        // Ubuntu 14.04
        //            total       used       free     shared    buffers     cached
        // Mem:       1015460     989508     25952    85816     138996      123060

        // Ubuntu 16.04
        //            total       used       free     shared    buff/cache  available
        // Mem:       1015188     230556     19044    44908     765588      701684

        @system('free', $ret);
        $out = ob_get_contents();
        ob_end_clean();
        if ($ret != 0 || strlen($out) === 0) {
            jrCore_add_to_cache('jrCore', $key, 'bad', 30);
            return false;
        }
        $_tmp = explode("\n", $out);
        if (!$_tmp || !is_array($_tmp)) {
            jrCore_add_to_cache('jrCore', $key, 'bad', 30);
            return false;
        }
        $_one                 = explode(' ', preg_replace("/ +/", ' ', $_tmp[1]));
        $_out['memory_total'] = strval($_one[1] * 1024);
        if (stripos($_tmp[0], 'available')) {
            $_out['memory_free'] = strval(($_one[3] + $_one[5]) * 1024);
        }
        else {
            $_out['memory_free'] = strval(($_one[3] + $_one[5] + $_one[6]) * 1024);
        }
        $_out['memory_used'] = ($_out['memory_total'] - $_out['memory_free']);
    }

    // Common
    if (jrCore_checktype($_out['memory_total'], 'number_nz')) {
        $_out['percent_used'] = round(($_out['memory_total'] - $_out['memory_free']) / $_out['memory_total'], 2) * 100;
    }
    else {
        $_out['percent_used'] = 0;
    }
    if ($_out['percent_used'] > 95) {
        $_out['class'] = 'bigsystem-cri';
    }
    elseif ($_out['percent_used'] > 90) {
        $_out['class'] = 'bigsystem-maj';
    }
    elseif ($_out['percent_used'] > 85) {
        $_out['class'] = 'bigsystem-min';
    }
    else {
        $_out['class'] = 'bigsystem-inf';
    }
    jrCore_add_to_cache('jrCore', $key, json_encode($_out), 30);
    return $_out;
}

/**
 * Get information about server disk usage
 * @return array
 */
function jrCore_get_disk_usage()
{
    $key = "jrcore_get_disk_usage";
    $out = jrCore_is_cached('jrCore', $key);
    if ($out) {
        return json_decode($out, true);
    }
    clearstatcache();
    $ts                   = disk_total_space(APP_DIR);
    $fs                   = disk_free_space(APP_DIR);
    $_out                 = array();
    $_out['disk_total']   = $ts;
    $_out['disk_free']    = $fs;
    $_out['disk_used']    = ($ts - $fs);
    $_out['percent_used'] = round(($ts - $fs) / $ts, 2) * 100;
    if ($_out['percent_used'] > 95) {
        $_out['class'] = 'bigsystem-cri';
    }
    elseif ($_out['percent_used'] > 90) {
        $_out['class'] = 'bigsystem-maj';
    }
    elseif ($_out['percent_used'] > 85) {
        $_out['class'] = 'bigsystem-min';
    }
    else {
        $_out['class'] = 'bigsystem-inf';
    }
    jrCore_add_to_cache('jrCore', $key, json_encode($_out), 300);
    return $_out;
}

/**
 * Returns load (run queue) information for server
 * @param int $proc_num Number of processors to determine system load
 * @return array|false
 */
function jrCore_get_system_load($proc_num = 1)
{
    if (!function_exists('sys_getloadavg')) {
        return false;
    }
    $_tm = sys_getloadavg();
    if (!$_tm || !is_array($_tm)) {
        return false;
    }
    if (!jrCore_checktype($proc_num, 'number_nz')) {
        $proc_num = 1;
    }
    $load              = array();
    $load[1]['level']  = $_tm[0];
    $load[5]['level']  = $_tm[1];
    $load[15]['level'] = $_tm[2];
    foreach (array(1, 5, 15) as $ll) {
        $level = number_format(round(($load[$ll]['level'] / $proc_num), 2), 2);
        if ($level > 4) {
            $load[$ll]['class'] = 'bigsystem-cri';
        }
        elseif ($level > 3) {
            $load[$ll]['class'] = 'bigsystem-maj';
        }
        elseif ($level > 2) {
            $load[$ll]['class'] = 'bigsystem-min';
        }
        else {
            $load[$ll]['class'] = 'bigsystem-inf';
        }
        $load[$ll]['level'] = $level;
    }
    return $load;
}

/**
 * Return information about Server Processors
 * @return mixed returns Array with CPU information
 */
function jrCore_get_proc_info()
{
    $key = "jrcore_get_proc_info";
    $out = jrCore_is_cached('jrCore', $key);
    if ($out) {
        return json_decode($out, true);
    }
    $_cpu = array();
    // proc file system
    if (@is_readable('/proc/cpuinfo')) {
        $_tmp = @file("/proc/cpuinfo");
        if (!is_array($_tmp)) {
            return 'unknown CPU';
        }
        $i = 0;
        foreach ($_tmp as $_v) {
            // get our processor
            if (stristr($_v, 'model name') || strstr($_v, 'altivec')) {
                $i++;
                $_cpu[$i]['model'] = trim(substr($_v, strpos($_v, ':') + 1));
            }
            elseif (stristr($_v, 'cpu MHz') || strstr($_v, 'clock')) {
                $_cpu[$i]['mhz'] = round(trim(substr($_v, strpos($_v, ':') + 1))) . " MHz";
            }
            elseif (stristr($_v, 'cache size') || strstr($_v, 'L2 cache')) {
                $_cpu[$i]['cache'] = trim(substr($_v, strpos($_v, ':') + 1));
            }
        }
    }
    // lscpu
    elseif (function_exists('is_executable') && @is_executable('/usr/bin/lscpu')) {
        ob_start();
        @system('/usr/bin/lscpu');
        $out = ob_get_contents();
        ob_end_clean();
        $i                 = 1;
        $ncp               = 0;
        $_cpu[$i]['mhz']   = '';
        $_cpu[$i]['cache'] = '';
        $_cpu[$i]['model'] = '';
        // CPU(s):                2
        foreach (explode("\n", $out) as $line) {
            // Number of procs
            if (strpos($line, 'CPU(') === 0) {
                $ncp = (int) jrCore_string_field($line, 'NF');
            }
            elseif (strpos($line, 'CPU MHz') === 0) {
                $_cpu[$i]['mhz'] = round((jrCore_string_field($line, 'NF') / 1000), 2) . " GHz";
            }
            elseif (strpos($line, 'L2') === 0) {
                $_cpu[$i]['cache'] = trim(jrCore_string_field($line, 'NF'));
            }
        }
        if (jrCore_checktype($ncp, 'number_nz') && $ncp <= 32) {
            while ($i < $ncp) {
                $i++;
                $_cpu[$i] = $_cpu[1];
            }
        }
    }
    // no proc file system - check for sysctl
    elseif (function_exists('is_executable') && @is_executable('/usr/sbin/sysctl')) {

        // CPU model
        ob_start();
        @system('sysctl -n machdep.cpu.brand_string');
        $cpu = trim(ob_get_contents());
        ob_end_clean();

        ob_start();
        @system('/usr/sbin/sysctl -a hw');
        $out = ob_get_contents();
        ob_end_clean();
        $i                 = 1;
        $ncp               = 0;
        $_cpu[$i]['mhz']   = ' ?';
        $_cpu[$i]['cache'] = '';
        $_cpu[$i]['model'] = $cpu;
        foreach (explode("\n", $out) as $line) {

            // Number of procs
            if (strstr($line, 'ncpu') && $ncp === 0) {
                $ncp = (int) jrCore_string_field($line, 'NF');
            }
            elseif (strstr($line, 'hw.cpufrequency:')) {
                $tmp             = explode(' ', $line);
                $tmp             = (int) end($tmp);
                $_cpu[$i]['mhz'] = round(((($tmp / 1000) / 1000) / 1000), 2) . " GHz";
            }
            elseif (strstr($line, 'hw.l2cachesize:')) {
                $tmp               = explode(' ', $line);
                $tmp               = (int) end($tmp);
                $_cpu[$i]['cache'] = round($tmp / 1024) . " Kb";
            }
        }
        if (jrCore_checktype($ncp, 'number_nz') && $ncp <= 32) {
            while ($i < $ncp) {
                $i++;
                $_cpu[$i] = $_cpu[1];
            }
        }

    }
    else {
        return 'unknown CPU';
    }
    jrCore_add_to_cache('jrCore', $key, json_encode($_cpu), 10800);
    return $_cpu;
}

/**
 * Extract a TAR archive
 * @param string $archive Archive name to extra
 * @param string $target Target directory to extract to
 * @return bool
 */
function jrCore_extract_tar_archive($archive, $target)
{
    // we need to include our TAR.php archive
    if (!class_exists('Archive_Tar')) {
        ini_set('include_path', APP_DIR . "/modules/jrCore/contrib/pear");
        require_once APP_DIR . "/modules/jrCore/contrib/pear/Tar.php";
    }
    $tar = new Archive_Tar($archive);
    $tar->setErrorHandling(PEAR_ERROR_PRINT);
    $tar->extract($target);
    ini_restore('include_path');
    return true;
}

/**
 * Create a TAR archive
 * @param string $archive Archive name to extra
 * @param array $_files Array of files to add to TAR archive
 * @return bool
 */
function jrCore_create_tar_archive($archive, $_files)
{
    global $_conf;
    if (!class_exists('Archive_Tar')) {
        ini_set('include_path', APP_DIR . "/modules/jrCore/contrib/pear");
        require_once APP_DIR . "/modules/jrCore/contrib/pear/Tar.php";
    }
    $tar = new Archive_Tar($archive);
    $tar->setErrorHandling(PEAR_ERROR_PRINT);
    $tar->create($_files);
    chmod($archive, $_conf['jrCore_file_perms']);
    return true;
}

/**
 * Create a new ZIP file from an array of files
 * @param $file string Full path of ZIP file to create
 * @param $_files array Array of files to add to zip file
 * @return bool
 */
function jrCore_create_zip_file($file, $_files)
{
    @ini_set('memory_limit', '1024M');
    require_once APP_DIR . "/modules/jrCore/contrib/zip/Zip.php";
    $zip = new Zip();
    $zip->setZipFile($file);
    $cnt = 0;
    foreach ($_files as $filename => $filepath) {
        if (is_file($filepath)) {
            if ($filename === $cnt) {
                $filename = str_replace(APP_DIR . '/', '', $filepath);
            }
            $zip->addFile(jrCore_file_get_contents($filepath), $filename, filemtime($filepath));
        }
        $cnt++;
    }
    $zip->finalize();
    return true;
}

/**
 * ZIP files in a given array and "stream" the resulting ZIP to the browser
 * @param string $name Name of ZIP file to send
 * @param array $_files Array of files to send
 * @return bool
 */
function jrCore_stream_zip($name, $_files)
{
    jrCore_db_close();
    @ini_set('memory_limit', '1024M');
    $tmp = jrCore_get_module_cache_dir('jrCore');
    // Send out our ZIP stream
    require_once APP_DIR . "/modules/jrCore/contrib/zip/ZipStream.php";
    try {
        $zip = new ZipStream($name);
    }
    catch (Exception $e) {
        return false;
    }
    $cnt = 0;
    foreach ($_files as $filename => $filepath) {
        if (is_file($filepath)) {
            $f = fopen($filepath, 'rb');
            if ($filename === $cnt) {
                $filename = basename($filepath);
            }
            $zip->addLargeFile($f, $filename, $tmp);
            fclose($f);
            $cnt++;
        }
    }
    $zip->finalize();
    return true;
}

/**
 * Send a file to a browser that causes a "Save..." dialog
 * @param $file string File to send
 * @return bool
 */
function jrCore_send_download_file($file)
{
    jrCore_db_close();
    if (!is_file($file) || strpos($file, APP_DIR) !== 0 || strpos($file, '..')) {
        return false;
    }
    // Send headers to initiate download prompt
    $size = filesize($file);
    header('Content-Length: ' . $size);
    header('Connection: close');
    header('Content-Type: application/octet-stream');
    header('Content-Transfer-Encoding: binary');
    header('Content-Disposition: attachment; filename="' . basename($file) . '"');

    $handle = fopen($file, 'rb');
    if (!$handle) {
        jrCore_logger('CRI', "core: unable to create file handle for download: {$file}");
        return false;
    }
    $bytes_sent = 0;
    while ($bytes_sent < $size) {
        fseek($handle, $bytes_sent);
        // Read 1 megabyte at a time...
        $buffer     = fread($handle, 1048576);
        $bytes_sent += strlen($buffer);
        echo $buffer;
        flush();
        unset($buffer);
        // Also - check that we have not sent out more data then the allowed size
        if ($bytes_sent >= $size) {
            fclose($handle);
            return true;
        }
    }
    fclose($handle);
    return true;
}

/**
 * Split array into equal number of arrays
 * @param $array array to split
 * @param $size int Size of array
 * @return array
 */
function jrCore_array_split($array, $size)
{
    $listlen = count($array);
    $partlen = floor($listlen / $size);
    $partrem = $listlen % $size;
    $split   = array();
    $mark    = 0;
    for ($px = 0; $px < $size; $px++) {
        $incr       = ($px < $partrem) ? $partlen + 1 : $partlen;
        $split[$px] = array_slice($array, $mark, $incr);
        $mark       += $incr;
    }
    return $split;
}

/**
 * Get skins in categories to show in the ACP
 * @return array
 */
function jrCore_get_acp_skins()
{
    // Expand our skins
    $_rt = jrCore_get_skins();
    $_sk = array();
    $_st = array();
    $_rp = array();
    foreach ($_rt as $skin_dir) {
        $func = "{$skin_dir}_skin_meta";
        if (!function_exists($func)) {
            require_once APP_DIR . "/skins/{$skin_dir}/include.php";
        }
        if (function_exists($func)) {
            $_mt            = $func();
            $_sk[$skin_dir] = $_mt;
            $cat            = 'general';
            if (isset($_mt['category'])) {
                foreach (explode(',', $_mt['category']) as $c) {
                    $c = trim(strtolower($c));
                    if (!isset($_st[$c])) {
                        $_st[$c] = array();
                    }
                    $_st[$c][$skin_dir] = $_sk[$skin_dir]['title'];
                }
            }
            else {
                if (!isset($_st[$cat])) {
                    $_st[$cat] = array();
                }
                $_st[$cat][$skin_dir] = $_sk[$skin_dir]['title'];
            }
        }
    }
    // We need to go through each module and get it's default page
    foreach ($_st as $cat => $_skins) {
        natcasesort($_skins);
        foreach ($_skins as $skin_dir => $title) {
            if (!isset($_rp[$cat])) {
                $_rp[$cat] = array();
            }
            $_rp[$cat][$skin_dir] = $_sk[$skin_dir];
            if (is_file(APP_DIR . "/skins/{$skin_dir}/config.php")) {
                $_rp[$cat][$skin_dir]['skin_index_page'] = 'global';
            }
            else {
                // info
                $_rp[$cat][$skin_dir]['skin_index_page'] = 'info';
            }
        }
    }
    return $_rp;
}

/**
 * Escape a string to sit between single quotes
 * @param string $str
 * @return string
 */
function jrCore_escape_single_quote_string($str)
{
    return addcslashes($str, "'\\");
}

//-----------------------------------------
// USER SESSION KEY wrappers
//-----------------------------------------

/**
 * Get a user session key
 * @param string $key
 * @return bool|mixed
 */
function jrCore_get_user_session_key($key)
{
    return (isset($_SESSION[$key])) ? $_SESSION[$key] : false;
}

/**
 * Set a user session key
 * @param string $key
 * @param mixed $value
 * @return bool
 */
function jrCore_set_user_session_key($key, $value)
{
    global $_user;
    $_user[$key]    = $value;
    $_SESSION[$key] = $value;
    return true;
}

/**
 * Return TRUE if a session key exists
 * @param string $key
 * @return bool
 */
function jrCore_user_session_key_exists($key)
{
    return (isset($_SESSION[$key]));
}

/**
 * Delete a session key if it exists
 * @param string $key
 * @return bool
 */
function jrCore_delete_user_session_key($key)
{
    global $_user;
    if (isset($_user[$key])) {
        unset($_user[$key]);
    }
    if (isset($_SESSION[$key])) {
        unset($_SESSION[$key]);
    }
    return true;
}

//-----------------------------------------
// LOCAL LOCK
//-----------------------------------------

/**
 * Create a LOCAL server lock
 * @param string $module Module to create lock file for
 * @param string $key Unique string that identifies this lock file
 * @param int $expire_seconds Seconds until lock file is no longer valid
 * @return bool
 */
function jrCore_create_local_lock($module, $key, $expire_seconds)
{
    $exp = (int) $expire_seconds;
    if (jrCore_local_cache_is_enabled()) {
        $pfx = jrCore_local_cache_get_key_prefix(false);
        // @note: The reason we have to FETCH this result first
        // is that with apcu_add, the keys is only expired on
        // NEXT ACCESS - there's no internal garbage collection
        // so we get a buildup of expired keys in the cache
        // @see https://github.com/krakjoe/apcu/issues/327
        if ($tmp = apcu_fetch("{$pfx}:l:{$module}_{$key}")) {
            if ($tmp <= time()) {
                apcu_delete("{$pfx}:l:{$module}_{$key}");
            }
        }
        if (apcu_add("{$pfx}:l:{$module}_{$key}", (time() + $exp), $exp)) {
            // We are the first process to create this
            return true;
        }
    }
    else {
        $dir = jrCore_get_module_cache_dir('jrCore');
        $key = str_replace('.', '-', jrCore_file_string($key));
        $lck = "{$dir}/lock.{$module}.{$key}";
        jrCore_start_timer('filesystem');
        if ($f = @fopen($lck, 'x')) {
            fwrite($f, $exp);
            fclose($f);
            jrCore_stop_timer('filesystem');
            return true;
        }
    }
    return false;
}

/**
 * Delete a LOCAL server lock
 * @param string $module
 * @param string $key
 * @return bool
 */
function jrCore_delete_local_lock($module, $key)
{
    if (jrCore_local_cache_is_enabled()) {
        $pfx = jrCore_local_cache_get_key_prefix(false);
        apcu_delete("{$pfx}:l:{$module}_{$key}");
    }
    else {
        $dir = jrCore_get_module_cache_dir('jrCore');
        $key = str_replace('.', '-', jrCore_file_string($key));
        $lck = "{$dir}/lock.{$module}.{$key}";
        jrCore_start_timer('filesystem');
        if (file_exists($lck)) {
            unlink($lck);
        }
        jrCore_stop_timer('filesystem');
    }
    return true;
}

/**
 * Delete expired LOCAL server locks
 * @return int Returns number of deleted expired lock files
 */
function jrCore_delete_expired_local_locks()
{
    $cnt = 0;
    if (jrCore_local_cache_is_enabled()) {
        if ($_tmp = apcu_cache_info()) {
            if (isset($_tmp['cache_list']) && is_array($_tmp['cache_list'])) {
                foreach ($_tmp['cache_list'] as $c) {
                    // [info] => testvar
                    // [ttl] => 10
                    // [num_hits] => 0
                    // [mtime] => 1572268834
                    // [creation_time] => 1572268834
                    // [deletion_time] => 0
                    // [access_time] => 1572268834
                    // [ref_count] => 0
                    // @note: no need to ensure it is a LOCK entry - delete if expired
                    if ($c['ttl'] > 0) {
                        // This entry has a TTL (time to live) - see if it has expired
                        if (($c['ttl'] + $c['creation_time']) < time()) {
                            apcu_delete($c['info']);
                            $cnt++;
                        }
                    }
                }
            }
        }
    }
    else {
        $dir = jrCore_get_module_cache_dir('jrCore');
        jrCore_start_timer('filesystem');
        if ($_fl = glob("{$dir}/lock.*", GLOB_NOSORT)) {
            foreach ($_fl as $f) {
                if ($e = file_get_contents($f)) {
                    if (intval($e) < time()) {
                        unlink($f);
                        $cnt++;
                    }
                }
            }
        }
        jrCore_stop_timer('filesystem');
    }
    return $cnt;
}

//-----------------------------------------
// GLOBAL LOCK
//-----------------------------------------

/**
 * Create a new Global Lock
 * @param string $module
 * @param string $key
 * @param int $expire_seconds
 * @return bool
 */
function jrCore_create_global_lock($module, $key, $expire_seconds)
{
    $key = "{$module}/{$key}";
    if (jrCore_is_cached($module, $key, false, false, true, false)) {
        // This global lock already exists
        return false;
    }
    $exp = (int) $expire_seconds;
    $unq = jrCore_db_escape($key);
    $tbl = jrCore_db_table_name('jrCore', 'global_lock');
    $req = "INSERT IGNORE INTO {$tbl} (lock_unique, lock_expires) VALUES ('{$unq}', (UNIX_TIMESTAMP() + {$exp}))";
    if ($lid = jrCore_db_query($req, 'INSERT_ID')) {
        jrCore_add_to_cache($module, $key, $lid, $expire_seconds, 0, false, false, false);
        return $lid;
    }
    return false;
}

/**
 * Check if a global lock exists
 * @param string $module
 * @param string $key
 * @param int $expire_seconds
 * @return bool
 */
function jrCore_global_lock_exists($module, $key)
{
    return (jrCore_is_cached($module, "{$module}/{$key}", false, false, true, false));
}

/**
 * Delete a global lock
 * @param string $module
 * @param string $key
 * @return bool
 */
function jrCore_delete_global_lock($module, $key)
{
    $key = "{$module}/{$key}";
    $unq = jrCore_db_escape($key);
    $tbl = jrCore_db_table_name('jrCore', 'global_lock');
    $req = "DELETE FROM {$tbl} WHERE lock_unique = '{$unq}'";
    jrCore_db_query($req);
    return jrCore_delete_cache($module, $key, false, false, false);
}

/**
 * Delete expired global locks
 * @return int
 * @note: No need to worry about CACHE here
 */
function jrCore_delete_expired_global_locks()
{
    $tbl = jrCore_db_table_name('jrCore', 'global_lock');
    $req = "SELECT * FROM {$tbl} WHERE lock_expires < UNIX_TIMESTAMP()";
    $_rt = jrCore_db_query($req, 'lock_id', false, 'lock_unique');
    if ($_rt && is_array($_rt)) {
        $req = "DELETE FROM {$tbl} WHERE lock_id IN(" . implode(',', array_keys($_rt)) . ')';
        jrCore_db_query($req);
        return count($_rt);
    }
    return 0;
}

/**
 * Get a hexadecimal color code for a color name
 * @param string $name
 * @return string
 */
function jrCore_get_hex_color_code_for_name($name)
{
    $_colors = array(
        'aliceblue'            => '#f0f8ff',
        'antiquewhite'         => '#faebd7',
        'aqua'                 => '#00ffff',
        'aquamarine'           => '#7fffd4',
        'azure'                => '#f0ffff',
        'beige'                => '#f5f5dc',
        'bisque'               => '#ffe4c4',
        'black'                => '#000000',
        'blanchedalmond'       => '#ffebcd',
        'blue'                 => '#0000ff',
        'blueviolet'           => '#8a2be2',
        'brown'                => '#a52a2a',
        'burlywood'            => '#deb887',
        'cadetblue'            => '#5f9ea0',
        'chartreuse'           => '#7fff00',
        'chocolate'            => '#d2691e',
        'coral'                => '#ff7f50',
        'cornflowerblue'       => '#6495ed',
        'cornsilk'             => '#fff8dc',
        'crimson'              => '#dc143c',
        'cyan'                 => '#00ffff',
        'darkblue'             => '#00008b',
        'darkcyan'             => '#008b8b',
        'darkgoldenrod'        => '#b8860b',
        'darkgray'             => '#a9a9a9',
        'darkgreen'            => '#006400',
        'darkkhaki'            => '#bdb76b',
        'darkmagenta'          => '#8b008b',
        'darkolivegreen'       => '#556b2f',
        'darkorange'           => '#ff8c00',
        'darkorchid'           => '#9932cc',
        'darkred'              => '#8b0000',
        'darksalmon'           => '#e9967a',
        'darkseagreen'         => '#8fbc8f',
        'darkslateblue'        => '#483d8b',
        'darkslategray'        => '#2f4f4f',
        'darkturquoise'        => '#00ced1',
        'darkviolet'           => '#9400d3',
        'deeppink'             => '#ff1493',
        'deepskyblue'          => '#00bfff',
        'dimgray'              => '#696969',
        'dodgerblue'           => '#1e90ff',
        'firebrick'            => '#b22222',
        'floralwhite'          => '#fffaf0',
        'forestgreen'          => '#228b22',
        'fuchsia'              => '#ff00ff',
        'gainsboro'            => '#dcdcdc',
        'ghostwhite'           => '#f8f8ff',
        'gold'                 => '#ffd700',
        'goldenrod'            => '#daa520',
        'gray'                 => '#808080',
        'green'                => '#008000',
        'greenyellow'          => '#adff2f',
        'honeydew'             => '#f0fff0',
        'hotpink'              => '#ff69b4',
        'indianred '           => '#cd5c5c',
        'indigo'               => '#4b0082',
        'ivory'                => '#fffff0',
        'khaki'                => '#f0e68c',
        'lavender'             => '#e6e6fa',
        'lavenderblush'        => '#fff0f5',
        'lawngreen'            => '#7cfc00',
        'lemonchiffon'         => '#fffacd',
        'lightblue'            => '#add8e6',
        'lightcoral'           => '#f08080',
        'lightcyan'            => '#e0ffff',
        'lightgoldenrodyellow' => '#fafad2',
        'lightgrey'            => '#d3d3d3',
        'lightgreen'           => '#90ee90',
        'lightpink'            => '#ffb6c1',
        'lightsalmon'          => '#ffa07a',
        'lightseagreen'        => '#20b2aa',
        'lightskyblue'         => '#87cefa',
        'lightslategray'       => '#778899',
        'lightsteelblue'       => '#b0c4de',
        'lightyellow'          => '#ffffe0',
        'lime'                 => '#00ff00',
        'limegreen'            => '#32cd32',
        'linen'                => '#faf0e6',
        'magenta'              => '#ff00ff',
        'maroon'               => '#800000',
        'mediumaquamarine'     => '#66cdaa',
        'mediumblue'           => '#0000cd',
        'mediumorchid'         => '#ba55d3',
        'mediumpurple'         => '#9370d8',
        'mediumseagreen'       => '#3cb371',
        'mediumslateblue'      => '#7b68ee',
        'mediumspringgreen'    => '#00fa9a',
        'mediumturquoise'      => '#48d1cc',
        'mediumvioletred'      => '#c71585',
        'midnightblue'         => '#191970',
        'mintcream'            => '#f5fffa',
        'mistyrose'            => '#ffe4e1',
        'moccasin'             => '#ffe4b5',
        'navajowhite'          => '#ffdead',
        'navy'                 => '#000080',
        'oldlace'              => '#fdf5e6',
        'olive'                => '#808000',
        'olivedrab'            => '#6b8e23',
        'orange'               => '#ffa500',
        'orangered'            => '#ff4500',
        'orchid'               => '#da70d6',
        'palegoldenrod'        => '#eee8aa',
        'palegreen'            => '#98fb98',
        'paleturquoise'        => '#afeeee',
        'palevioletred'        => '#d87093',
        'papayawhip'           => '#ffefd5',
        'peachpuff'            => '#ffdab9',
        'peru'                 => '#cd853f',
        'pink'                 => '#ffc0cb',
        'plum'                 => '#dda0dd',
        'powderblue'           => '#b0e0e6',
        'purple'               => '#800080',
        'rebeccapurple'        => '#663399',
        'red'                  => '#ff0000',
        'rosybrown'            => '#bc8f8f',
        'royalblue'            => '#4169e1',
        'saddlebrown'          => '#8b4513',
        'salmon'               => '#fa8072',
        'sandybrown'           => '#f4a460',
        'seagreen'             => '#2e8b57',
        'seashell'             => '#fff5ee',
        'sienna'               => '#a0522d',
        'silver'               => '#c0c0c0',
        'skyblue'              => '#87ceeb',
        'slateblue'            => '#6a5acd',
        'slategray'            => '#708090',
        'snow'                 => '#fffafa',
        'springgreen'          => '#00ff7f',
        'steelblue'            => '#4682b4',
        'tan'                  => '#d2b48c',
        'teal'                 => '#008080',
        'thistle'              => '#d8bfd8',
        'tomato'               => '#ff6347',
        'turquoise'            => '#40e0d0',
        'violet'               => '#ee82ee',
        'wheat'                => '#f5deb3',
        'white'                => '#ffffff',
        'whitesmoke'           => '#f5f5f5',
        'yellow'               => '#ffff00',
        'yellowgreen'          => '#9acd32'
    );

    $name = strtolower($name);
    return (isset($_colors[$name])) ? strtoupper($_colors[$name]) : $name;
}
