<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package Dashboard
 * @copyright 2003 - 2022 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Generate "Bigview" dashboard view
 * @param $_post array posted info
 * @param $_user array viewing user info
 * @param $_conf array global config
 */
function jrCore_dashboard_bigview($_post, $_user, $_conf)
{
    global $_mods;
    // Default layout
    $rows = 2;
    $cols = 4;
    // check for custom layout
    if ($_cfg = jrCore_get_config_value('jrCore', 'dashboard_config', false)) {
        $_cfg = json_decode($_cfg, true);
        if (isset($_cfg['rows']) && jrCore_checktype($_cfg['rows'], 'number_nz')) {
            $rows = (int) $_cfg['rows'];
        }
        if (isset($_cfg['cols']) && jrCore_checktype($_cfg['cols'], 'number_nz')) {
            $cols = (int) $_cfg['cols'];
        }
    }
    else {
        $_cfg = array();
    }

    // Our default panel setup
    $_def = jrCore_get_default_dashboard_panels();
    foreach ($_def as $row => $_cols) {
        foreach ($_cols as $col => $_inf) {
            if (!isset($_cfg['_panels'][$row])) {
                $_cfg['_panels'][$row] = array();
            }
            if (!isset($_cfg['_panels'][$row][$col])) {
                $_cfg['_panels'][$row][$col] = $_inf;
            }
        }
    }
    ksort($_cfg['_panels'], SORT_NUMERIC);

    // Get registered Graph functions
    $_tmp = jrCore_get_registered_module_features('jrGraph', 'graph_config');
    $_tmp = jrCore_trigger_event('jrCore', 'dashboard_panels', $_tmp);
    $_url = array();
    if ($_tmp && is_array($_tmp)) {
        foreach ($_tmp as $mod => $_fnc) {
            foreach ($_fnc as $name => $_inf) {
                $_url[$name] = jrCore_get_module_url($mod);
            }
        }
    }

    $_html  = array();
    $_func  = array();
    $width  = round((100 / $cols), 2);
    $inline = false;
    for ($r = 0; $r < $rows; $r++) {
        $dat = array();
        for ($c = 0; $c < $cols; $c++) {
            $dat[$c]['title'] = '';
            if (isset($_cfg['_panels'][$r][$c])) {
                $ttl = $_cfg['_panels'][$r][$c]['t'];
                if (strpos($_cfg['_panels'][$r][$c]['t'], 'item count')) {
                    $mod = trim(jrCore_string_field($_cfg['_panels'][$r][$c]['t'], 1));
                    if (isset($_mods[$mod])) {
                        $ttl = $_mods[$mod]['module_name'] . ' count';
                    }
                }
                $dat[$c]['title'] = '<div class="bignum_stat_cell">' . $ttl;
                $fnc              = $_cfg['_panels'][$r][$c]['f'];
                if (function_exists($fnc)) {
                    $_func[$r][$c] = $fnc($_cfg['_panels'][$r][$c]['t']);
                    $out           = $_func[$r][$c];
                    if (isset($out['graph']) && !jrCore_is_mobile_device()) {
                        $id = "g{$r}{$c}";
                        if (strpos($out['graph'], '/')) {
                            list($mu,) = explode('/', $out['graph'], 2);
                            $mu = $_url[$mu];
                        }
                        else {
                            $mu = $_url["{$out['graph']}"];
                        }
                        if (strlen($mu) > 0) {
                            $_html[] = "<div id=\"{$id}\" style=\"width:750px;height:400px;display:none;bottom:0;\"></div>";
                            if (jrCore_module_is_active('jrGraph')) {
                                $dat[$c]['title'] .= "<div class=\"bignum_stat\"><a href=\"{$_conf['jrCore_base_url']}/{$mu}/graph/{$out['graph']}\" onclick=\"jrCore_dashboard_disable_reload(60);jrGraph_modal_graph('#{$id}','{$mu}','{$out['graph']}'); return false\">" . jrCore_get_icon_html('stats', 16) . '</a></div>';
                            }
                        }
                    }
                }
                $dat[$c]['title'] .= '</div>';
            }
            $dat[$c]['width'] = "{$width}%";
        }
        if (!$inline) {
            jrCore_page_table_header($dat, "bigtable bigtable_r{$rows} bigtable_c{$cols}", false, null, 'jr_bigview');
            $inline = true;
        }
        else {
            jrCore_page_table_header($dat, 'bigtable', true);
        }

        $dat = array();
        for ($c = 0; $c < $cols; $c++) {
            if (isset($_cfg['_panels'][$r][$c])) {
                $out = false;
                if (isset($_func[$r][$c])) {
                    $out = $_func[$r][$c];
                    if ($out && is_array($out)) {
                        $dat[$c]['title'] = $out['title'];
                        if (isset($out['class'])) {
                            $dat[$c]['class'] = "bignum bignum" . ($c + 1) . " {$out['class']}";
                        }
                        else {
                            $dat[$c]['class'] = "bignum bignum" . ($c + 1);
                        }
                    }
                }
                if (!$out) {
                    $dat[$c]['title'] = '!';
                    $dat[$c]['class'] = "bignum bignum" . ($c + 1) . ' error';
                }
            }
            else {
                $dat[$c]['title'] = '?';
                $dat[$c]['class'] = "bignum bignum" . ($c + 1);
            }
            if (isset($dat[$c]['class'])) {
                $dat[$c]['class'] .= "\" id=\"id-{$r}-{$c}";
            }
        }
        jrCore_page_table_row($dat);
    }
    jrCore_page_table_footer();

    $html = jrCore_parse_template('dashboard_panels.tpl', array(), 'jrCore');
    jrCore_page_custom($html);

    if (is_array($_html)) {
        jrCore_page_custom(implode("\n", $_html));
    }

    if (jrUser_is_admin()) {
        $_tmp = array("$('.bignum').click(function(e) { e.stopPropagation(); jrCore_dashboard_disable_reload(60); jrCore_dashboard_panel($(this).attr('id')); return false });");
        jrCore_create_page_element('javascript_ready_function', $_tmp);
    }

    return true;
}

/**
 * Get the default dashboard panel setup
 * @return array
 */
function jrCore_get_default_dashboard_panels()
{
    // Our default panel setup
    return array(
        0 => array(
            0 => array(
                't' => 'total profiles',
                'f' => 'jrProfile_dashboard_panels'
            ),
            1 => array(
                't' => 'signups today',
                'f' => 'jrProfile_dashboard_panels'
            ),
            2 => array(
                't' => 'users online',
                'f' => 'jrUser_dashboard_panels'
            ),
            3 => array(
                't' => 'queue depth',
                'f' => 'jrCore_dashboard_panels'
            )
        ),
        1 => array(
            0 => array(
                't' => 'memory used',
                'f' => 'jrCore_dashboard_panels'
            ),
            1 => array(
                't' => 'disk usage',
                'f' => 'jrCore_dashboard_panels'
            ),
            2 => array(
                't' => 'CPU count',
                'f' => 'jrCore_dashboard_panels'
            ),
            3 => array(
                't' => '5 minute load',
                'f' => 'jrCore_dashboard_panels'
            )
        )
    );
}

/**
 * The Dashboard -> Activity Log
 * @param $_post array Posted parameters
 * @param $_user array Viewing User
 * @param $_conf array Global Config
 * @param $from string "dashboard" or empty
 * @return null
 */
function jrCore_dashboard_activity_log($_post, $_user, $_conf, $from = '')
{
    jrCore_master_log_tabs('activity');
    $url = jrCore_get_module_url('jrCore');

    // construct our query
    $arc = '';
    $tbl = jrCore_db_table_name('jrCore', 'log');
    if (isset($_post['archive'])) {
        if (strlen($_post['archive']) === 6 && jrCore_checktype($_post['archive'], 'number_nz')) {
            $tbl = jrCore_db_table_name('jrCore', 'log_' . $_post['archive']);
            $arc = "/archive={$_post['archive']}";
        }
        else {
            unset($_post['archive']);
        }
    }
    $tb1 = jrCore_db_table_name('jrCore', 'log_debug');
    $tb2 = jrCore_db_table_name('jrCore', 'log_debug_data');
    $req = "SELECT l.*, d.log_log_id AS old_log_id, n.log_log_id AS new_log_id
              FROM {$tbl} l
         LEFT JOIN {$tb1} d ON (d.log_log_id = l.log_id)
         LEFT JOIN {$tb2} n ON (n.log_log_id = l.log_id) ";

    $type = 'all';
    $mod  = 'WHERE';
    if (!empty($_post['type'])) {
        $type = strtolower($_post['type']);
        switch ($type) {
            case 'dbg':
            case 'inf':
            case 'min':
            case 'maj':
            case 'cri':
            case 'err':
                break;
            default:
                $type = 'all';
                break;
        }
    }
    if ($type !== 'all') {
        if ($type == 'err') {
            $req .= "WHERE log_priority IN('min','maj','cri')";
        }
        else {
            $req .= "WHERE log_priority = '{$type}'";
        }
        $mod = 'AND';
    }

    $_ex = false;
    $add = '';
    if (!empty($_post['search_string'])) {
        $str = jrCore_strip_non_ascii(trim(urldecode($_post['search_string'])));
        if (!empty($_post['search_area']) && $_post['search_area'] == 'ip') {
            $str = jrCore_db_escape($str);
            if (substr_count($str, '.') === 3) {
                $req .= "{$mod} l.log_ip = '{$str}' ";
            }
            else {
                $req .= "{$mod} l.log_ip LIKE '%{$str}%' ";
            }
        }
        elseif (!empty($_post['search_area']) && $_post['search_area'] == 'user') {
            $req .= "{$mod} l.log_text LIKE '%[{$str}]%' ";
        }
        else {
            $_post['search_area'] = 'text';
            if (strlen($str) < 4) {
                if (strpos(" {$str}", '%')) {
                    $str = jrCore_db_escape($str);
                    $req .= "{$mod} log_text LIKE '{$str}' ";
                }
                else {
                    $str = jrCore_db_escape($str);
                    $req .= "{$mod} log_text LIKE '%{$str}%' ";
                }
            }
            else {
                if (strpos(" {$str}", '%')) {
                    $str = jrCore_db_escape($str);
                    $req .= "{$mod} MATCH(log_text) AGAINST('{$str}' IN NATURAL LANGUAGE MODE) AND log_text LIKE '{$str}' ";
                }
                else {
                    $str = jrCore_db_escape($str);
                    $req .= "{$mod} MATCH(log_text) AGAINST('{$str}' IN NATURAL LANGUAGE MODE) AND log_text LIKE '%{$str}%' ";
                }
            }
        }
        $_ex = array('search_string' => $_post['search_string']);
        $add = '/search_string=' . str_replace('%2F', '___FS', urlencode($_post['search_string']));
        $mod = 'AND';
    }
    $dat = '';
    if (isset($_post['date']) && strlen($_post['date']) === 8 && jrCore_checktype($_post['date'], 'number_nz')) {
        // We are showing log entries for a specific day
        $dt1 = strtotime("{$_post['date']} 00:00:00");
        $dt2 = strtotime("{$_post['date']} 23:59:59") + 1;
        $req .= "{$mod} l.log_created BETWEEN {$dt1} AND {$dt2} ";
        $dat = "/date={$_post['date']}";
    }
    // @note: must order by log_created on cluster master due to resend logs
    if (jrCore_module_is_active('jrCloudMaster')) {
        $req .= 'ORDER BY l.log_created DESC';
    }
    else {
        $req .= 'ORDER BY l.log_id DESC';
    }

    // find how many lines we are showing
    if (!isset($_post['p']) || !jrCore_checktype($_post['p'], 'number_nz')) {
        $_post['p'] = 1;
    }
    $_rt = jrCore_db_paged_query($req, $_post['p'], 12, 'NUMERIC', 'simplepagebreak');

    $bu = jrCore_get_base_url() . "/{$url}/activity_log";
    if ($from == 'dashboard') {
        $bu = jrCore_get_base_url() . "/{$url}/dashboard/activity";
    }

    // start our html output
    $archive = (!empty($_post['archive'])) ? "/archive={$_post['archive']}" : '';
    $buttons = jrCore_page_button('download', 'Export', "jrCore_confirm('Export the Activity Log?', 'Please be patient - the CSV file could be large and take a minute to create', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/activity_log_download{$archive}')})");

    if ($_archive = jrCore_db_get_all_archive_dates('jrCore', 'log')) {
        $_archive = array_reverse($_archive);
        $_options = array(0 => 'Current');
        foreach ($_archive as $a) {
            // [0] => 202012
            $y            = substr($a, 0, 4);
            $m            = substr($a, 4, 2);
            $_options[$a] = "Archive {$y}-{$m}";
        }
        $selected = (!empty($_post['archive'])) ? $_post['archive'] : 0;
        $onchange = jrCore_strip_url_params(jrCore_get_current_url(), array('archive', 'date'));
        $onchange = "var v=$(this).val(); if (v > 0) { jrCore_window_location('{$onchange}/archive=' + v) } else { jrCore_window_location('{$onchange}') }";
        $buttons  .= jrCore_page_jumper('archive-jumper', $_options, $selected, $onchange, 'form_select_narrow');
    }

    // Get our dates
    if (!$_mn = jrCore_is_cached('jrCore', "activity_log_min_date_{$tbl}", false, false, true, false)) {
        $req = "SELECT FROM_UNIXTIME(log_created, '%Y%m%d') AS log_date FROM {$tbl} GROUP BY log_date";
        $_mn = jrCore_db_query($req, 'log_date', false, 'log_date');
        $sec = (strtotime('tomorrow') - time());
        if (!empty($_mn)) {
            krsort($_mn);
        }
        jrCore_add_to_cache('jrCore', "activity_log_min_date_{$tbl}", $_mn, $sec, 0, false, false, false);
    }
    if (!empty($_mn)) {
        $_options = array(0 => 'All Dates');
        foreach ($_mn as $v) {
            $y            = substr($v, 0, 4);
            $m            = substr($v, 4, 2);
            $d            = substr($v, 6, 2);
            $_options[$v] = "{$y}-{$m}-{$d}";
        }
        $selected = (!empty($_post['date'])) ? $_post['date'] : 0;
        $onchange = jrCore_strip_url_params(jrCore_get_current_url(), array('date'));
        $onchange = "var v=$(this).val(); if (v > 0) { jrCore_window_location('{$onchange}/date=' + v) } else { jrCore_window_location('{$onchange}') }";
        $buttons  .= jrCore_page_jumper('date-jumper', $_options, $selected, $onchange, 'form_select_narrow');
    }

    $_options = array(
        'all' => 'All Priorities',
        'err' => 'Priority Only (min, maj, cri)',
        'dbg' => 'Debug',
        'inf' => 'Informational',
        'min' => 'Minor',
        'maj' => 'Major',
        'cri' => 'Critical'
    );
    $onchange = jrCore_strip_url_params(jrCore_get_current_url(), array('type'));
    $onchange = "jrCore_window_location('{$onchange}/type=' + $(this).val())";
    $buttons  .= jrCore_page_jumper('pri-jumper', $_options, $type, $onchange, 'form_select_narrow');
    $buttons  .= jrCore_page_icon_button('refresh', 'refresh', "window.location.reload()", array('title' => 'Refresh'));

    jrCore_page_banner('activity log', $buttons);
    jrCore_get_form_notice();

    $_areas = array(
        'text' => 'Log Text',
        'user' => 'User Name',
        'ip'   => 'IP Address'
    );
    jrCore_page_search_enable_search_areas($_areas);
    jrCore_page_search('search', "{$bu}{$arc}{$add}{$dat}", null, true, 'Search String');

    $dat = array();
    if (jrUser_is_master()) {
        $dat[0]['title'] = '<input type="checkbox" class="form_checkbox" onclick="$(\'.log_checkbox\').prop(\'checked\',$(this).prop(\'checked\'));">';
        $dat[0]['width'] = '1%;';
        $dat[1]['title'] = '&nbsp;';
        $dat[1]['width'] = '1%;';
        $dat[2]['title'] = 'date';
        $dat[2]['width'] = '4%;';
    }
    else {
        $dat[2]['title'] = 'date';
        $dat[2]['width'] = '6%;';
    }
    $dat[3]['title'] = 'IP';
    $dat[3]['width'] = '1%;';
    $dat[4]['title'] = 'PID';
    $dat[4]['width'] = '1%;';
    $dat[5]['title'] = 'text';
    $dat[6]['title'] = '&nbsp;';
    $dat[6]['width'] = '1%;';
    jrCore_page_table_header($dat);
    unset($dat);

    if (isset($_rt['_items']) && is_array($_rt['_items'])) {

        // LOG LINE
        $murl = jrCore_get_module_url('jrCore');
        $curl = jrCore_get_module_url('jrUser');
        foreach ($_rt['_items'] as $k => $_log) {

            $dat = array();
            if (jrUser_is_master()) {
                $dat[0]['title'] = '<input type="checkbox" class="form_checkbox log_checkbox" name="' . $_log['log_id'] . '">';
                $dat[0]['class'] = 'center';
                $dat[1]['title'] = jrCore_page_icon_button("log-delete-{$_log['log_id']}", 'clear', "jrCore_delete_activity_log({$_log['log_id']})");
            }
            $dat[2]['title'] = jrCore_format_time($_log['log_created']);
            $dat[2]['class'] = 'center nowrap';
            if (!empty($_post['search_string']) && $_post['search_area'] == 'ip') {
                $dat[3]['title'] = "<a onclick=\"popwin('{$_conf['jrCore_base_url']}/{$curl}/whois/{$_log['log_ip']}','{$_log['log_ip']}',950,650,'yes');\">" . jrCore_hilight_string($_log['log_ip'], $_post['search_string']) . '</a>';
            }
            else {
                $dat[3]['title'] = "<a onclick=\"popwin('{$_conf['jrCore_base_url']}/{$curl}/whois/{$_log['log_ip']}','{$_log['log_ip']}',950,650,'yes');\">" . $_log['log_ip'] . '</a>';
            }
            $dat[3]['class'] = 'center';

            $pid = '-';
            if (strpos($_log['log_text'], '%') === 0) {
                preg_match('/^%(\d*)%/', $_log['log_text'], $m);
                if (!empty($m[1])) {
                    $pid              = (int) $m[1];
                    $_log['log_text'] = substr($_log['log_text'], strpos($_log['log_text'], '%:') + 2);
                }
            }
            $dat[4]['title'] = $pid;
            $dat[4]['class'] = 'center';

            if (!empty($_post['search_string']) && $_post['search_area'] != 'ip') {
                $dat[5]['title'] = jrCore_hilight_string($_log['log_text'], $_post['search_string']);
            }
            else {
                $dat[5]['title'] = $_log['log_text'];
            }
            $dat[5]['class'] = "log-inf log-{$_log['log_priority']} word-break";
            $log_id          = false;
            if (!empty($_log['new_log_id'])) {
                $log_id = intval($_log['new_log_id']);
            }
            elseif (!empty($_log['old_log_id'])) {
                $log_id = intval($_log['old_log_id']);
            }
            if (!empty($log_id)) {
                $dat[6]['title'] = jrCore_page_icon_button("log-debug-{$k}", 'search', "window.open('{$_conf['jrCore_base_url']}/{$murl}/log_debug/{$log_id}')");
                $dat[6]['class'] = "log-inf log-{$_log['log_priority']}";
            }
            jrCore_page_table_row($dat, null, $_log);
        }
        $sjs = "var v = $('input:checkbox.log_checkbox:checked').map(function(){ return this.name; }).get().join(',')";
        $tmp = jrCore_page_button("delete", 'delete checked', "jrCore_confirm('Delete checked activity logs?', '', function(){ {$sjs};jrCore_delete_activity_log(v) })");
        if (!empty($_post['search_string'])) {
            $add = '';
            if (!empty($_post['archive'])) {
                $add = "/archive={$_post['archive']}";
            }
            $url = jrCore_get_base_url() . "/{$murl}/activity_log_delete{$add}/search_string=" . rawurlencode($_post['search_string']);
            $tmp .= ' ' . jrCore_page_button("delete_search", 'delete all matching', "jrCore_confirm('Delete all activity logs matching the search?', '', function(){ jrCore_window_location('{$url}') })");
        }

        $dat             = array();
        $dat[1]['title'] = $tmp;
        jrCore_page_table_row($dat);
        jrCore_page_table_pager($_rt, $_ex);
    }
    else {
        $dat = array();
        if (!empty($_post['search_string'])) {
            $dat[1]['title'] = 'No Activity Logs were found to match your search criteria';
        }
        else {
            $dat[1]['title'] = 'There are no Activity Log entries to show';
        }
        $dat[1]['class'] = 'p20 center';
        jrCore_page_table_row($dat);
    }
    jrCore_page_table_footer();
    return true;
}

/**
 * Show Pending Items Dashboard view
 * @param $_post array Global $_post
 * @param $_user array Viewing user array
 * @param $_conf array Global config
 */
function jrCore_dashboard_pending($_post, $_user, $_conf)
{
    // Get our pending items
    $mod = $_post['m'];
    $tbl = jrCore_db_table_name('jrCore', 'pending');
    $req = "SELECT * FROM {$tbl} WHERE pending_linked_item_id = 0 AND pending_module = '" . jrCore_db_escape($mod) . "' ORDER BY pending_id ASC";
    if (!isset($_post['p']) || !jrCore_checktype($_post['p'], 'number_nz')) {
        $_post['p'] = 1;
    }
    $_rt = jrCore_db_paged_query($req, $_post['p'], 12);

    $val = null;
    $_ex = null;
    if (!empty($_post['search_string'])) {
        $val = jrCore_entity_string($_post['search_string']);
        $_ex = array('search_string' => $_post['search_string']);
    }
    jrCore_page_search('search', "{$_conf['jrCore_base_url']}/{$_post['module_url']}/dashboard/pending/m={$mod}", $val);
    $dat = array();
    if ($_rt && is_array($_rt) && isset($_rt['_items'])) {
        $dat[1]['title'] = '<input type="checkbox" class="form_checkbox" onclick="$(\'.pending_checkbox\').prop(\'checked\',$(this).prop(\'checked\'));">';
    }
    else {
        $dat[1]['title'] = '<input type="checkbox" class="form_checkbox" disabled>';
    }
    $dat[1]['width'] = '1%;';
    $dat[2]['title'] = 'date';
    $dat[2]['width'] = '13%;';
    $dat[3]['title'] = 'item';
    $dat[3]['width'] = '40%;';
    $dat[4]['title'] = 'profile';
    $dat[4]['width'] = '10%;';
    $dat[5]['title'] = 'user';
    $dat[5]['width'] = '10%;';
    $dat[6]['title'] = 'approve';
    $dat[6]['width'] = '2%;';
    $dat[7]['title'] = 'reject';
    $dat[7]['width'] = '2%;';
    $dat[8]['title'] = 'delete';
    $dat[8]['width'] = '2%;';

    $dat = jrCore_trigger_event('jrCore', 'pending_items_header', $dat);

    jrCore_page_table_header($dat);
    unset($dat);

    $fnd = false;
    $url = jrCore_get_module_url('jrCore');
    if ($_rt && is_array($_rt) && isset($_rt['_items'])) {

        // Get item data
        $_ids = array();
        foreach ($_rt['_items'] as $_pend) {
            $_ids[] = (int) $_pend['pending_item_id'];
        }
        if ($_ids = jrCore_db_get_multiple_items($mod, $_ids, null, false, '_item_id')) {

            // Get profile data
            $_pr = array();
            foreach ($_ids as $v) {
                $pid       = intval($v['_profile_id']);
                $_pr[$pid] = $pid;
            }
            if (!empty($_pr)) {
                if ($_pr = jrCore_db_get_multiple_items('jrProfile', $_pr, null, false, '_item_id')) {
                    foreach ($_ids as $k => $v) {
                        $pid = intval($v['_profile_id']);
                        if (isset($_pr[$pid])) {
                            $_ids[$k] = array_merge($_pr[$pid], $v);
                        }
                    }
                }
            }

            // Get user data
            $_pr = array();
            foreach ($_ids as $v) {
                $pid       = intval($v['_user_id']);
                $_pr[$pid] = $pid;
            }
            if (!empty($_pr)) {
                if ($_pr = jrCore_db_get_multiple_items('jrUser', $_pr, null, false, '_item_id')) {
                    foreach ($_ids as $k => $v) {
                        $pid = intval($v['_user_id']);
                        if (isset($_pr[$pid])) {
                            $_ids[$k] = array_merge($_pr[$pid], $v);
                        }
                    }
                }
            }
            $_ids = array('_items' => $_ids);
        }

        $mpfx = jrCore_db_get_prefix($mod);
        $murl = jrCore_get_module_url($mod);

        // Did we get a search string?
        if (!empty($_post['search_string'])) {
            foreach ($_rt['_items'] as $pk => $_pend) {
                if (isset($_ids['_items']["{$_pend['pending_item_id']}"])) {
                    $found = false;
                    $_data = $_ids['_items']["{$_pend['pending_item_id']}"];
                    foreach ($_data as $v) {
                        if (stripos(' ' . $v, $_post['search_string'])) {
                            $found = true;
                            break;
                        }
                    }
                    if (!$found) {
                        unset($_rt['_items'][$pk]);
                    }
                }
            }
        }

        if (count($_rt['_items']) > 0) {
            foreach ($_rt['_items'] as $_pend) {
                if (isset($_ids['_items']["{$_pend['pending_item_id']}"])) {

                    $_data = $_ids['_items']["{$_pend['pending_item_id']}"];

                    // Did we get a search string?
                    if (!empty($_post['search_string'])) {
                        $found = false;
                        foreach ($_data as $v) {
                            if (stripos(' ' . $v, $_post['search_string'])) {
                                $found = true;
                                break;
                            }
                        }
                        if (!$found) {
                            continue;
                        }
                    }

                    $dat             = array();
                    $dat[1]['title'] = '<input type="checkbox" class="form_checkbox pending_checkbox" name="' . $_pend['pending_id'] . '">';
                    $dat[1]['class'] = 'center';
                    $dat[2]['title'] = jrCore_format_time($_pend['pending_created']);
                    $dat[2]['class'] = 'center';
                    if (!$title = jrCore_get_item_title($mod, $_data)) {
                        $title = "{$_data['profile_url']}/{$murl}/{$_pend['pending_item_id']}";
                    }
                    if (!empty($_post['search_string'])) {
                        $title = jrCore_hilight_string($title, $_post['search_string']);
                    }
                    $update_url      = jrCore_get_item_url('update', $mod, $_data);
                    $dat[3]['title'] = "<a href=\"{$update_url}\" target=\"_blank\" rel=\"noopener\"><b>{$title}</b></a>";
                    if (isset($_data["{$mpfx}_pending_reason"])) {
                        $dat[3]['title'] .= '<br><small>' . $_data["{$mpfx}_pending_reason"] . '</small>';
                    }
                    $dat[4]['title'] = "<a href=\"{$_conf['jrCore_base_url']}/{$_data['profile_url']}\">@{$_data['profile_name']}</a>";
                    $dat[4]['class'] = 'center';
                    $dat[5]['title'] = $_data['user_name'];
                    $dat[5]['class'] = 'center';
                    $dat[6]['title'] = jrCore_page_button("pending-approve-{$_pend['pending_id']}", 'approve', "jrCore_window_location('{$_conf['jrCore_base_url']}/{$url}/pending_item_approve/{$_pend['pending_module']}/id={$_pend['pending_item_id']}')");
                    $dat[7]['title'] = jrCore_page_button("pending-reject-{$_pend['pending_id']}", 'reject', "jrCore_window_location('{$_conf['jrCore_base_url']}/{$url}/pending_item_reject/{$_pend['pending_module']}/id={$_pend['pending_item_id']}')");
                    $dat[8]['title'] = jrCore_page_button("pending-delete-{$_pend['pending_id']}", 'delete', "jrCore_confirm('Delete this pending item?', 'No user notification will be sent', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$url}/pending_item_delete/{$_pend['pending_module']}/id={$_pend['pending_id']}')})");

                    // Let other modules customize our row if needed
                    $_data['module'] = $mod;
                    $dat             = jrCore_trigger_event('jrCore', 'pending_items_row', $dat, $_data);

                    jrCore_page_table_row($dat, null, $_pend);
                    $fnd = true;
                }
            }
            if ($fnd) {
                $sjs = "var v = $('input:checkbox.pending_checkbox:checked').map(function(){ return this.name; }).get().join(',')";
                $tmp = jrCore_page_button("all", 'approve checked', "{$sjs};jrCore_window_location('{$_conf['jrCore_base_url']}/{$url}/pending_item_approve/all/id=,'+ v)");
                $tmp .= '&nbsp;' . jrCore_page_button("delete", 'delete checked', "jrCore_confirm('Delete checked items?', '', function(){ {$sjs};jrCore_window_location('{$_conf['jrCore_base_url']}/{$url}/pending_item_delete/{$mod}/id='+ v )})");

                $dat             = array();
                $dat[1]['title'] = $tmp;
                jrCore_page_table_row($dat);
                jrCore_page_table_pager($_rt, $_ex);
            }
        }
    }

    if (!$fnd) {
        $dat = array();
        if (!empty($_post['search_string'])) {
            $dat[1]['title'] = 'There were no Pending Items found to match your search criteria';
        }
        else {
            $dat[1]['title'] = 'There are no pending items to show';
        }
        $dat[1]['class'] = 'p20 center';
        jrCore_page_table_row($dat);
    }
    jrCore_page_table_footer();
}

/**
 * Display DS Browser
 * @param $mode string dashboard|admin where browser is being run from
 * @param $_post array Global $_post
 * @param $_user array Viewing user array
 * @param $_conf array Global config
 * @return bool
 */
function jrCore_dashboard_browser($mode, $_post, $_user, $_conf)
{
    global $_mods;

    // Get modules that have registered a custom datastore browser
    $add = '';
    if (jrUser_is_master() && !jrCore_module_is_active('jrCloudClient')) {
        $add .= jrCore_get_activity_indicator('csv-dl', 24);
        $txt = 'Export';
        if (!empty($_post['search_string'])) {
            $txt = 'Export Matching';
        }
        $add .= jrCore_page_button('export', $txt, "jrCore_confirm('Download CSV File?', 'Please be patient - the generated CSV file could be large and take a bit to create', function(){ jrCore_export_csv('{$_post['module']}') })");
    }

    $_tmp = jrCore_get_registered_module_features('jrCore', 'data_browser');

    // Create a Quick Jump list for custom forms for this module
    $_mds = array();
    foreach ($_mods as $mod_dir => $_inf) {
        if (!jrCore_module_is_active($mod_dir) || $mod_dir == 'jrCore' || $mod_dir == 'jrSeamless') {
            continue;
        }
        if (jrCore_db_get_prefix($mod_dir)) {
            $_mds[] = $mod_dir;
        }
    }
    $add .= jrCore_get_module_jumper('data_browser', $_post['module'], "jrCore_window_location('{$_conf['jrCore_base_url']}/'+ $(this).val() +'/dashboard/browser')", $_mds);

    $val = '';
    if (isset($_post['search_string']) && strlen($_post['search_string']) > 0) {
        $val = urldecode($_post['search_string']);
    }

    jrCore_page_banner($_mods["{$_post['module']}"]['module_name'], $add);
    jrCore_get_form_notice();
    jrCore_page_search('search', "{$_conf['jrCore_base_url']}/{$_post['module_url']}/dashboard/browser", $val);

    // See if this module has registered it's own Browser
    if (isset($_tmp["{$_post['module']}"])) {
        $func = array_keys($_tmp["{$_post['module']}"]);
        $func = (string) reset($func);
        if (function_exists($func)) {
            $func($_post, $_user, $_conf);
        }
        else {
            jrCore_page_notice('error', "invalid custom browser function defined for {$_post['module']}");
        }
    }
    else {

        // get our items
        $num = jrCore_get_pager_rows();
        $_pr = array(
            'pagebreak'       => $num,
            'page'            => 1,
            'order_by'        => array(
                '_item_id' => 'desc'
            ),
            'skip_all_checks' => true,
            'ignore_pending'  => true,
            'quota_check'     => false,
            'privacy_check'   => false
        );
        if (isset($_post['p']) && jrCore_checktype($_post['p'], 'number_nz')) {
            $_pr['page'] = (int) $_post['p'];
        }
        // See we have a search condition
        $_ex = false;
        if (isset($_post['search_string']) && strlen($_post['search_string']) > 0) {
            if (!strpos($_post['search_string'], ':')) {
                jrCore_set_form_notice('error', 'You must provide an <b>Item Key</b> to search - see help for detail', false);
                $url = jrCore_strip_url_params(jrCore_get_current_url(), array('search_string'));
                jrCore_location($url);
            }
            $_ex           = array('search_string' => $_post['search_string']);
            $_pr['search'] = array(jrCore_get_ds_search_condition($_post['search_string']));
        }
        $_us = jrCore_db_search_items($_post['module'], $_pr);

        // See if we have detail pages for this module
        $view = false;
        if (is_file(APP_DIR . "/modules/{$_post['module']}/templates/item_detail.tpl")) {
            $view = true;
        }

        // Start our output
        $dat             = array();
        $dat[1]['title'] = 'ID';
        $dat[1]['width'] = '5%';
        $dat[2]['title'] = 'item data';
        $dat[2]['width'] = '78%';
        $dat[3]['title'] = 'action';
        $dat[3]['width'] = '2%';
        jrCore_page_table_header($dat);

        if (isset($_us['_items']) && is_array($_us['_items'])) {
            foreach ($_us['_items'] as $_itm) {
                $dat = array();
                switch ($_post['module']) {
                    case 'jrUser':
                        $iid = $_itm['_user_id'];
                        break;
                    case 'jrProfile':
                        $iid = $_itm['_profile_id'];
                        break;
                    default:
                        $iid = $_itm['_item_id'];
                        break;
                }
                $pfx             = jrCore_db_get_prefix($_post['module']);
                $dat[1]['title'] = $iid;
                $dat[1]['class'] = 'center';
                $_tm             = array();
                ksort($_itm);
                $master_user = false;
                $admin_user  = false;
                $_rep        = array("\n", "\r", "\n\r");
                foreach ($_itm as $k => $v) {
                    if (strpos($k, $pfx) !== 0 && strpos($k, '_') !== 0) {
                        continue;
                    }
                    switch ($k) {
                        case 'user_password':
                        case 'user_old_password':
                        case 'user_validate':
                            break;
                        /** @noinspection PhpMissingBreakStatementInspection */
                        case 'user_group':
                            switch ($v) {
                                case 'master':
                                    $master_user = true;
                                    break;
                                case 'admin':
                                    $admin_user = true;
                                    break;
                            }
                        // We fall through on purpose!
                        default:
                            if (is_array($v)) {
                                $v = json_encode($v);
                            }
                            if (is_numeric($v) && strlen($v) === 10) {
                                if (strpos($k, 'phone')) {
                                    $v = jrCore_format_phone_number($v);
                                }
                                else {
                                    $v = $v . ' - (' . jrCore_format_time($v) . ')';
                                }
                            }
                            else {
                                $v = strip_tags(str_replace($_rep, ' ', $v));
                            }
                            if (strlen($v) > 80) {
                                $v = substr($v, 0, 80) . '...';
                            }
                            if (isset($_post['search_string'])) {
                                // See if we are searching a specific field
                                if (isset($sf)) {
                                    if ($k == $sf) {
                                        $v = jrCore_hilight_string($v, str_replace('%', '', $_post['search_string']));
                                    }
                                }
                                else {
                                    $v = jrCore_hilight_string($v, str_replace('%', '', $_post['search_string']));
                                }
                            }
                            $add = '';
                            if (!empty($_post['search_string']) && stripos($_post['search_string'], "{$k}:") === 0) {
                                $add = ' success';
                            }
                            $_tm[] = "<span class=\"ds_browser_key{$add}\">{$k}</span> <span class=\"ds_browser_value\">{$v}</span>";
                            break;
                    }
                }
                $dat[3]['title'] = '<div class="ds_browser_item">' . implode('<br>', $_tm) . '</div>';
                $_att            = array(
                    'style' => 'width:70px;'
                );

                $_btn = array();
                if ($view && isset($_itm["{$pfx}_title_url"])) {
                    $url    = "{$_conf['jrCore_base_url']}/{$_itm['profile_url']}/{$_post['module_url']}/{$iid}/{$_itm["{$pfx}_title_url"]}";
                    $_btn[] = jrCore_page_button("v{$iid}", 'view', "jrCore_window_location('{$url}')", $_att);
                }

                $url    = "{$_conf['jrCore_base_url']}/{$_post['module_url']}/browser_item_update/id={$iid}";
                $_btn[] = jrCore_page_button("m{$iid}", 'modify', "jrCore_window_location('{$url}')", $_att);

                // Check and see if we are browsing User Accounts - if so, admin users cannot delete
                // admin or master accounts.  Master cannot delete other master accounts.
                $add = false;
                if (jrUser_is_master() && !$master_user) {
                    $add = true;
                }
                elseif (jrUser_is_admin() && !$master_user && !$admin_user) {
                    $add = true;
                }
                if ($add) {
                    $_btn[] = jrCore_page_button("d{$iid}", 'delete', "jrCore_confirm('Delete Item?', 'The item will be permanently deleted!', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/browser_item_delete/id={$iid}') })", $_att);
                }
                $dat[4]['title'] = implode('<br><br>', $_btn);
                $dat[4]['class'] = 'center';
                jrCore_page_table_row($dat, null, $_itm);
            }
            jrCore_page_table_pager($_us, $_ex);
        }
        else {
            $dat = array();
            if (isset($_post['search_string'])) {
                $dat[1]['title'] = 'No Results found for your Search Criteria';
            }
            else {
                $dat[1]['title'] = 'No Items found in this Datastore';
            }
            $dat[1]['class'] = 'p20 center';
            jrCore_page_table_row($dat);
        }
        jrCore_page_table_footer();
    }
    return true;
}

/**
 * Queue Viewer
 * @param $_post array Global $_post
 * @param $_user array Viewing user array
 * @param $_conf array Global config
 * @return bool
 */
function jrCore_dashboard_queue_viewer($_post, $_user, $_conf)
{
    global $_mods;
    $buttons = jrCore_format_time(time()) . '&nbsp;';
    if (jrUser_is_master()) {
        if (!jrCore_queues_are_active()) {
            $buttons .= jrCore_page_button('queues-pause', 'resume all queues', "jrCore_confirm('Resume Queue Workers?', '', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/queue_pause/on') })");
        }
        else {
            $buttons .= jrCore_page_button('queues-pause', 'pause all queues', "jrCore_confirm('Pause Queue Workers?', 'Any existing workers will finish their current job and exit', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/queue_pause/off') })");
        }
    }
    $buttons .= jrCore_page_icon_button('queues-refresh', 'refresh', "location.reload()");

    $add = '';
    if (isset($_post['queue_name']) && strlen($_post['queue_name']) > 0) {
        $add     = ': <span style="text-transform:none">' . $_post['queue_name'] . '</span>';
        $buttons .= jrCore_page_button('all', 'all queues', "jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/dashboard/queue_viewer')");
    }
    jrCore_page_banner("Queue Viewer{$add}", $buttons);

    if (!jrCore_queues_are_active()) {
        jrCore_set_form_notice('error', '<strong>All Queues are Paused</strong> - press the &quot;Resume All Queues&quot; button to start the Queue workers.<br>New queue jobs will not be executed while Queues are paused', false);
    }
    if (!isset($_post['queue_name']) || strlen($_post['queue_name']) === 0) {
        jrCore_set_form_notice('success', 'Queue Workers are background processes that perform queued tasks<br>such as media conversion, notifications, cache cleanup, system backups, etc.', false);
    }
    jrCore_get_form_notice();

    // Queue Counts
    if (isset($_post['queue_name']) && strlen($_post['queue_name']) > 0) {

        // SPECIFIC QUEUE

        $page = 1;
        if (isset($_post['p']) && jrCore_checktype($_post['p'], 'number_nz')) {
            $page = (int) $_post['p'];
        }
        $mod = jrCore_db_escape($_post['queue_module']);
        $nam = jrCore_db_escape($_post['queue_name']);

        $num = 0;
        $tbl = jrCore_db_table_name('jrCore', 'queue_info');
        $req = "SELECT queue_depth FROM {$tbl} WHERE queue_name = '{$mod}_{$nam}'";
        $_cq = jrCore_db_query($req, 'SINGLE');
        if ($_cq && is_array($_cq) && isset($_cq['queue_depth'])) {
            $num = (int) $_cq['queue_depth'];
        }
        $tb1 = jrCore_db_table_name('jrCore', 'queue');
        $tb2 = jrCore_db_table_name('jrCore', 'queue_data');
        $req = "SELECT q.*, d.queue_data, UNIX_TIMESTAMP() AS queue_time FROM {$tb1} q JOIN {$tb2} d ON (d.queue_id = q.queue_id) WHERE q.queue_module = '{$mod}' AND q.queue_name = '{$nam}' ORDER BY q.queue_started DESC, q.queue_sleep ASC";
        $_rt = jrCore_db_paged_query($req, $page, 12, 'NUMERIC', $num);

        $dat = array();
        if ($_rt && is_array($_rt) && isset($_rt['_items'])) {
            $dat[1]['title'] = '<input type="checkbox" class="form_checkbox" onclick="$(\'.tk_checkbox\').prop(\'checked\',$(this).prop(\'checked\'));">';
        }
        else {
            $dat[1]['title'] = '<input type="checkbox" class="form_checkbox" disabled>';
        }
        $dat[1]['width'] = '1%';
        $dat[2]['title'] = 'queue ID';
        $dat[2]['width'] = '5%';
        $dat[3]['title'] = 'worker ID';
        $dat[3]['width'] = '5%';
        $dat[4]['title'] = 'created';
        $dat[4]['width'] = '5%';
        $dat[5]['title'] = 'queue data';
        $dat[5]['width'] = '67%';
        $dat[6]['title'] = 'status';
        $dat[6]['width'] = '6%';
        $dat[7]['title'] = 'latency';
        $dat[7]['width'] = '6%';
        $dat[8]['title'] = 'reset';
        $dat[8]['width'] = '5%';
        $dat[9]['title'] = 'delete';
        $dat[9]['width'] = '5%';
        jrCore_page_table_header($dat);

        if ($_rt && is_array($_rt) && isset($_rt['_items'])) {
            foreach ($_rt['_items'] as $v) {
                $dat             = array();
                $dat[1]['title'] = '<input type="checkbox" class="form_checkbox tk_checkbox" name="' . $v['queue_id'] . '">';
                $dat[1]['class'] = 'center';
                $dat[2]['title'] = $v['queue_id'];
                $dat[2]['class'] = 'center';
                $dat[3]['title'] = (strlen($v['queue_worker']) > 0) ? $v['queue_worker'] : '-';
                $dat[3]['class'] = 'center word-break';
                $dat[4]['title'] = jrCore_format_time($v['queue_created']);
                $dat[4]['class'] = 'center';
                $dat[5]['title'] = '<div class="p5 fixed-width" style="max-height:100px;word-break:break-all;overflow:auto">' . jrCore_strip_html(print_r(json_decode($v['queue_data'], true), true)) . '</div>';
                $cls             = '';
                $val             = (time() - $v['queue_sleep']);
                if ($val > 3600) {
                    $cls = 'error ';
                }
                elseif ($val > 1200) {
                    $cls = 'warning ';
                }
                elseif ($val > 300) {
                    $cls = 'notice ';
                }
                if (strlen($v['queue_worker']) > 0) {
                    $dat[6]['title'] = 'Working<br><small>' . jrCore_number_format(time() - $v['queue_started']) . ' s</small>';
                    $dat[6]['class'] = 'success center';
                    $dat[7]['title'] = '-';
                    $cls             = '';
                }
                elseif ($v['queue_sleep'] > $v['queue_time']) {
                    $dat[6]['title'] = 'Sleeping<br><small>' . jrCore_number_format($v['queue_sleep'] - $v['queue_time']) . ' s</small>';
                    $dat[6]['class'] = 'center';
                    $dat[7]['title'] = '-';
                }
                else {
                    $dat[6]['title'] = 'Pending';
                    $dat[6]['class'] = 'center';
                    $dat[7]['title'] = jrCore_number_format((time() - $v['queue_sleep'])) . ' s';
                }
                if ($v['queue_count'] > 0) {
                    $dat[6]['title'] .= '<br><small>Attempts: ' . $v['queue_count'] . '</small>';
                }
                $dat[7]['class'] = $cls . ' center" style="text-align:center';
                if (strlen($v['queue_worker']) > 0 || $v['queue_sleep'] > $v['queue_time']) {
                    $dat[8]['title'] = jrCore_page_button("r{$v['queue_id']}", 'reset', "jrCore_confirm('Reset Queue Entry?', 'The queue entry can then be worked by a new Queue Worker', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/queue_entry_reset/id={$v['queue_id']}') })");
                }
                else {
                    $dat[8]['title'] = jrCore_page_button("r{$v['queue_id']}", 'reset', 'disabled');
                }
                $dat[8]['class'] = 'center';
                $dat[9]['title'] = jrCore_page_button("d{$v['queue_id']}", 'delete', "jrCore_confirm('Delete Queue Entry?', '', function(){ jrCore_delete_queue_entry('{$v['queue_id']}') })");
                $dat[9]['class'] = 'center';
                jrCore_page_table_row($dat);
            }
            $sjs             = "var v = $('input:checkbox.tk_checkbox:checked').map(function(){ return this.name; }).get().join(',')";
            $dat             = array();
            $dat[1]['title'] = jrCore_page_button("delete", 'delete checked', "jrCore_confirm('Delete Checked Entries?', '', function(){ {$sjs};jrCore_delete_queue_entry(v) })");
            jrCore_page_table_row($dat);
            jrCore_page_table_pager($_rt);
        }
        else {
            $dat             = array();
            $dat[1]['title'] = 'There are no queue entries in this queue';
            $dat[1]['class'] = 'p20 center';
            jrCore_page_table_row($dat);
        }
        jrCore_page_table_footer();
        jrCore_page_cancel_button('referrer');

    }
    else {

        // ALL QUEUE ENTRIES

        // Worker Count
        $tbl = jrCore_db_table_name('jrCore', 'queue_info');
        $req = "SELECT queue_name AS qn, queue_workers AS qw, queue_depth as qd, queue_status as qs FROM {$tbl} ORDER BY queue_depth DESC";
        $_qw = jrCore_db_query($req, 'qn', false, null, false, null, false);

        // Reorder so Cluster entries are at bottom
        $ord = false;
        $_cl = array();
        $_lc = array();
        foreach ($_qw as $qname => $v) {
            if (strpos($qname, '_queue_server_')) {
                $_cl[$qname] = $v;
                $ord         = true;
            }
            else {
                if ($v['qd'] > 0) {
                    $_lc[$qname] = $v;
                }
            }
        }
        $_qw = array_merge($_lc, $_cl);

        $dat             = array();
        $dat[1]['title'] = '&nbsp;';
        $dat[1]['width'] = '1%';
        if (count($_lc) === 0) {
            $dat[2]['title'] = 'cluster module';
        }
        elseif ($ord) {
            $dat[2]['title'] = 'local module';
        }
        else {
            $dat[2]['title'] = 'queue module';
        }
        $dat[2]['width'] = '22%';
        $dat[3]['title'] = 'queue name';
        $dat[3]['width'] = '26%';
        $dat[4]['title'] = 'workers';
        $dat[4]['width'] = '12%';
        $dat[5]['title'] = 'entries';
        $dat[5]['width'] = '12%';
        $dat[6]['title'] = 'latency';
        $dat[6]['width'] = '12%';
        $dat[7]['title'] = 'view';
        $dat[7]['width'] = '5%';
        $dat[8]['title'] = 'empty';
        $dat[8]['width'] = '5%';
        $dat[9]['title'] = 'pause';
        $dat[9]['width'] = '5%';
        jrCore_page_table_header($dat);

        $k   = 0;
        $shw = !(count($_lc) > 0);
        $tbl = jrCore_db_table_name('jrCore', 'queue');
        foreach ($_qw as $v) {

            // Only show Queue if we have entries
            if ($v['qd'] > 0) {

                list($mod, $nam) = explode('_', $v['qn'], 2);

                if (!$shw && $ord && strpos($nam, 'queue_server') === 0) {
                    $dat             = array();
                    $dat[1]['title'] = '&nbsp;';
                    $dat[1]['width'] = '1%';
                    $dat[2]['title'] = 'cluster module';
                    $dat[2]['width'] = '22%';
                    $dat[3]['title'] = 'queue name';
                    $dat[3]['width'] = '26%';
                    $dat[4]['title'] = 'workers';
                    $dat[4]['width'] = '12%';
                    $dat[5]['title'] = 'entries';
                    $dat[5]['width'] = '12%';
                    $dat[6]['title'] = 'latency';
                    $dat[6]['width'] = '12%';
                    $dat[7]['title'] = 'view';
                    $dat[7]['width'] = '5%';
                    $dat[8]['title'] = 'empty';
                    $dat[8]['width'] = '5%';
                    $dat[9]['title'] = 'pause';
                    $dat[9]['width'] = '5%';
                    jrCore_page_table_header($dat, null, true);
                    $shw = true;
                }

                // @note: It is faster for us to use individual queries here for each queue rather than
                // use a GROUP BY in a query that gets all queue entries at once
                $req = "SELECT MIN(queue_sleep) AS q_min FROM {$tbl} WHERE queue_started = 0 AND queue_module = '{$mod}' AND queue_name = '{$nam}' AND queue_sleep > 0";
                $_qm = jrCore_db_query($req, 'SINGLE');

                $dat             = array();
                $dat[1]['title'] = jrCore_get_module_icon_html($mod, 26);
                $dat[2]['title'] = (isset($_mods[$mod]['module_name'])) ? $_mods[$mod]['module_name'] : $mod;
                $dat[2]['class'] = 'center';
                $dat[3]['title'] = $nam;
                $dat[3]['class'] = 'center';
                $dat[4]['title'] = $v['qw'];
                $dat[4]['class'] = 'center';
                $dat[5]['title'] = jrCore_number_format($v['qd']);
                $dat[5]['class'] = 'center';

                $cls = '';
                if (isset($v['qw']) && $v['qw'] >= $v['qd']) {
                    // All Queue entries are being worked
                    $dat[6]['title'] = '-';
                }
                elseif ($_qm && isset($_qm['q_min']) && $_qm['q_min'] > 0 && $_qm['q_min'] > time()) {
                    $dat[6]['title'] = 'Sleeping<br><small>' . jrCore_number_format($_qm['q_min'] - time()) . ' s</small>';
                }
                else {
                    if ($_qm && isset($_qm['q_min']) && $_qm['q_min'] > 0) {
                        $val = (time() - $_qm['q_min']);
                        if ($val > 3600) {
                            $cls = 'error ';
                        }
                        elseif ($val > 1200) {
                            $cls = 'warning ';
                        }
                        elseif ($val > 300) {
                            $cls = 'notice ';
                        }
                        $dat[6]['title'] = jrCore_number_format((time() - $_qm['q_min'])) . ' s';
                    }
                    else {
                        // Q_min is bad
                        $dat[6]['title'] = '?';
                    }
                }
                $dat[6]['class'] = $cls . ' center" style="text-align:center';
                $dat[7]['title'] = jrCore_page_button("view-entries-{$k}", 'view', "jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/dashboard/queue_viewer/queue_module={$mod}/queue_name={$nam}')");
                $dat[7]['class'] = $cls . ' center';
                $dat[8]['title'] = jrCore_page_button("empty-queue-{$k}", 'empty', "jrCore_confirm('Delete All Entries?', '', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/queue_empty_save/queue_module={$mod}/queue_name={$nam}') } )");
                $dat[8]['class'] = 'center ' . $cls;
                if ($v['qs'] == 0) {
                    $dat[9]['title'] = jrCore_page_button("empty-queue-{$k}", 'resume', "jrCore_confirm('Enable this Queue?', 'Queue workers will be able to work entries in this queue', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/queue_status_save/queue_name={$v['qn']}/qs=1') } )");
                    $dat[9]['class'] = 'center error';
                }
                else {
                    $dat[9]['title'] = jrCore_page_button("empty-queue-{$k}", 'pause', "jrCore_confirm('Pause this Queue?', 'Queue entries can be created but will not be worked<br>while this queue is paused', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/queue_status_save/queue_name={$v['qn']}/qs=0') } )");
                    $dat[9]['class'] = 'center ' . $cls;
                }
                jrCore_page_table_row($dat);
                $k++;
            }
        }
        if ($k == 0) {
            $dat             = array();
            $dat[1]['title'] = 'There are no queue entries to show';
            $dat[1]['class'] = 'p20 center';
            jrCore_page_table_row($dat);
        }
        jrCore_page_table_footer();
    }
    return true;
}

/**
 * Show the system Recycle Bin
 * @param $_post array Global $_post
 * @param $_user array Viewing user array
 * @param $_conf array Global config
 * @return bool
 */
function jrCore_dashboard_recycle_bin($_post, $_user, $_conf)
{
    global $_mods;
    // Get all unique module entries in the Recycle Bin
    $btn = null;
    $tbl = jrCore_db_table_name('jrCore', 'recycle');
    $req = "SELECT DISTINCT(r_module) AS m FROM {$tbl} WHERE r_group_id = '1'";
    $_md = jrCore_db_query($req, 'm', false, 'm');
    if ($_md && is_array($_md)) {

        $_options = array(0 => 'All Modules');
        foreach ($_mods as $mod_dir => $_inf) {
            if (!isset($_md[$mod_dir])) {
                continue;
            }
            if (isset($_inf['module_prefix']) && strlen($_inf['module_prefix']) > 0) {
                $_options[$mod_dir] = $_inf['module_name'];
            }
        }
        asort($_options);

        $onchange = jrCore_strip_url_params(jrCore_get_current_url(), array('p', 'm'));
        $onchange = "var i = $(this).val(); if (i == 0) { jrCore_window_location('{$onchange}') } else { jrCore_window_location('{$onchange}/m=' + i) }";
        $selected = (!empty($_post['m'])) ? $_post['m'] : 0;
        $btn      .= jrCore_page_jumper('recycle_bin_browser', $_options, $selected, $onchange);
    }
    $del = (int) jrCore_get_config_value('jrCore', 'enable_rb_delete', 1);
    if ($del === 1) {
        $btn .= jrCore_page_button('e', 'empty recycle bin', "jrCore_confirm('Empty Recycle Bin?', 'This will permanently delete all items in the Recycle Bin', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/empty_recycle_bin') })");
    }
    jrCore_page_banner('Recycle Bin', $btn);
    jrCore_get_form_notice();
    jrCore_page_search('search', "{$_conf['jrCore_base_url']}/{$_post['module_url']}/dashboard/recycle_bin");

    $dat             = array();
    $dat[0]['title'] = '<input type="checkbox" class="form_checkbox" onclick="$(\'.rb_checkbox\').prop(\'checked\',$(this).prop(\'checked\'));">';
    $dat[0]['width'] = '1%;';
    $dat[1]['title'] = 'profile';
    $dat[1]['width'] = '13%';
    $dat[2]['title'] = 'module';
    $dat[2]['width'] = '13%';
    $dat[3]['title'] = 'img';
    $dat[3]['width'] = '5%';
    $dat[4]['title'] = 'item';
    $dat[4]['width'] = '39%';
    $dat[5]['title'] = 'deleted';
    $dat[5]['width'] = '10%';
    $dat[6]['title'] = 'expires';
    $dat[6]['width'] = '10%';
    $dat[7]['title'] = 'content';
    $dat[7]['width'] = '3%';
    $dat[8]['title'] = 'restore';
    $dat[8]['width'] = '3%';
    $dat[9]['title'] = 'delete';
    $dat[9]['width'] = '3%';
    jrCore_page_table_header($dat);

    $page = 1;
    if (isset($_post['p']) && jrCore_checktype($_post['p'], 'number_nz')) {
        $page = (int) $_post['p'];
    }

    $add = '';
    if (isset($_post['m']) && strlen($_post['m'])) {
        $add = " AND r_module = '" . jrCore_db_escape($_post['m']) . "'";
    }
    $sst = '';
    if (isset($_post['search_string']) && strlen($_post['search_string']) > 0) {

        // With a search string we are looking for a PROFILE NAME or an ITEM TITLE
        $_pr = array(
            'search'              => array(
                "profile_name like %{$_post['search_string']}%"
            ),
            'return_item_id_only' => true,
            'skip_triggers'       => true,
            'privacy_check'       => false
        );
        $_pr = jrCore_db_search_items('jrProfile', $_pr);

        $sst = jrCore_db_escape($_post['search_string']);
        if ($_pr && is_array($_pr)) {
            $sst = " AND (r_title LIKE '%{$sst}%' OR r_profile_id IN(" . implode(',', array_keys($_pr)) . "))";
        }
        else {
            $sst = " AND r_title LIKE '%{$sst}%'";
        }
    }

    $req = "SELECT r.* FROM {$tbl} r WHERE (r_module = 'jrProfile' OR r_group_id = '1'){$add}{$sst} ORDER BY r_id DESC";
    $_rt = jrCore_db_paged_query($req, $page, 12);
    if ($_rt && is_array($_rt) && isset($_rt['_items'])) {

        $exp = (int) jrCore_get_config_value('jrCore', 'recycle_bin_expire', 3);
        if ($exp > 0) {
            $exp = ($exp * 86400);
        }

        // Get Profile Names
        $_pr = array();
        $_pi = array();
        foreach ($_rt['_items'] as $_r) {
            $pid       = (int) $_r['r_profile_id'];
            $_pi[$pid] = $pid;
        }
        if (count($_pi) > 0) {
            $_tm = jrCore_db_get_multiple_items('jrProfile', $_pi, array('_profile_id', 'profile_url'));
            if ($_tm && is_array($_tm)) {
                foreach ($_tm as $p) {
                    $pid       = (int) $p['_profile_id'];
                    $_pr[$pid] = $p['profile_url'];
                }
            }
        }

        $murl = jrCore_get_module_url('jrImage');
        foreach ($_rt['_items'] as $k => $_r) {

            $dat             = array();
            $dat[0]['title'] = '<input type="checkbox" class="form_checkbox rb_checkbox" name="' . $_r['r_id'] . '">';
            $dat[0]['class'] = 'center';
            switch ($_r['r_module']) {
                case 'jrUser':
                case 'jrProfile':
                    $dat[1]['title'] = '@' . $_r['r_title'];
                    break;
                default:
                    $dat[1]['title'] = (isset($_pr["{$_r['r_profile_id']}"])) ? "<a href=\"{$_conf['jrCore_base_url']}/" . $_pr["{$_r['r_profile_id']}"] . '">@' . $_pr["{$_r['r_profile_id']}"] . '</a>' : '?';
                    break;
            }
            $dat[1]['class'] = 'center';
            $dat[2]['title'] = $_mods["{$_r['r_module']}"]['module_name'];
            $dat[2]['class'] = 'center';

            $_tmp = json_decode($_r['r_data'], true);
            // Did this item have an image?
            if (strpos($_r['r_data'], '_image_size')) {
                $pfx             = jrCore_db_get_prefix($_r['r_module']);
                $url             = $_conf['jrCore_base_url'] . '/' . $murl . '/rb_image/' . $_r['r_id'] . '/' . $_r['r_module'] . '/' . $pfx . '_image';
                $dat[3]['title'] = '<a href="' . $url . '" data-lightbox="images" title="' . jrCore_entity_string($dat[1]['title']) . '"><img style="max-width:72px" src="' . $url . '">';
            }
            else {
                $dat[3]['title'] = '-';
                $dat[3]['class'] = 'center';
            }

            if ($_r['r_title'] == '?') {
                // If this item does NOT have a title, then it could be a module that
                // adds entries to other items (comments, ratings, likes, etc.) - in
                // this case we want to show the item they are ATTACHED to
                if ($_tmp && is_array($_tmp)) {
                    $pfx = jrCore_db_get_prefix($_r['r_module']);
                    if (isset($_tmp["{$pfx}_title"])) {
                        $dat[4]['title'] = $_tmp["{$pfx}_title"];
                    }
                    elseif (isset($_tmp["{$pfx}_text"])) {
                        if (strlen($_tmp["{$pfx}_text"]) > 140) {
                            $dat[4]['title'] = substr(jrCore_strip_html($_tmp["{$pfx}_text"]), 0, 140) . '...';
                        }
                        else {
                            $dat[4]['title'] = jrCore_strip_html($_tmp["{$pfx}_text"]);
                        }
                    }
                    else {
                        $nam = '';
                        $mod = false;
                        if (isset($_tmp["{$pfx}_module"])) {
                            $mod = $_tmp["{$pfx}_module"];
                            $nam = (isset($_mods[$mod])) ? $_mods[$mod]['module_name'] : $mod;
                        }
                        $iid = false;
                        if (isset($_tmp["{$pfx}_item_id"])) {
                            $iid = (int) $_tmp["{$pfx}_item_id"];
                        }
                        if ($mod && $iid) {
                            $req = "SELECT r_title FROM {$tbl} WHERE r_module = '" . jrCore_db_escape($mod) . "' AND r_item_id = '{$iid}' LIMIT 1";
                            $_in = jrCore_db_query($req, 'SINGLE');
                            if ($_in && isset($_in['r_title'])) {
                                $dat[4]['title'] = $_in['r_title'] . '<br><small>(' . $nam . ')</small>';
                            }
                        }
                    }
                }
                if (!isset($dat[4]['title'])) {
                    $dat[4]['title'] = 'item_id: ' . $_r['r_item_id'] . ' (no title)';
                }
            }
            else {
                $dat[4]['title'] = $_r['r_title'];
            }

            $dat[4]['class'] = 'center';
            $dat[5]['title'] = jrCore_format_time($_r['r_time']);
            $dat[5]['class'] = 'center';
            if ($exp > 0) {
                $dat[6]['title'] = jrCore_format_time($_r['r_time'] + $exp);
            }
            else {
                $dat[6]['title'] = 'never';
            }
            $dat[6]['class'] = 'center';
            $dat[7]['title'] = jrCore_page_button("rb-content-{$k}", 'content', "jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/recycle_bin_item_content/id={$_r['r_id']}')");
            $dat[8]['title'] = jrCore_page_button("rb-restore-{$k}", 'restore', "jrCore_confirm('Restore This Item?', 'Please be patient - restoring could take several minutes for a very large profile', function() { $('.form_button').jrCore_disable_button();jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/restore_recycle_bin_item/id={$_r['r_id']}') })");
            if ($del !== 1) {
                $dat[9]['title'] = jrCore_page_button("rb-delete-{$_r['r_id']}", 'delete', 'disabled');
            }
            else {
                $dat[9]['title'] = jrCore_page_button("rb-delete-{$_r['r_id']}", 'delete', "jrCore_confirm('Delete this Item?', 'The item will be permanently deleted from the system', function(){ jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/recycle_bin_delete/id={$_r['r_id']}') })");
            }
            jrCore_page_table_row($dat);
        }

        $sjs = "var v = $('input:checkbox.rb_checkbox:checked').map(function(){ return this.name }).get().join(',')";
        $tmp = jrCore_page_button("restore-all", 'restore checked', "jrCore_confirm('Restore the selected entries?', 'Please be patient - restoring the items could take several minutes depending on their size', function(){ {$sjs}; $('.form_button').jrCore_disable_button(); jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/restore_recycle_bin_item/id=' + v) })");
        if ($del === 1) {
            $tmp .= ' ' . jrCore_page_button("delete-all", 'delete checked', "jrCore_confirm('Delete the selected entries?', 'Please be patient - deleting the items could take several minutes depending on their size', function(){ {$sjs}; $('.form_button').jrCore_disable_button(); jrCore_window_location('{$_conf['jrCore_base_url']}/{$_post['module_url']}/recycle_bin_delete/id=' + v) })");
        }
        $dat             = array();
        $dat[1]['title'] = $tmp;
        jrCore_page_table_row($dat);
        jrCore_page_table_pager($_rt);
    }
    else {
        $dat = array();
        if (isset($_post['search_string']) && strlen($_post['search_string']) > 0) {
            $dat[1]['title'] = 'No items in the Recycle Bin matched your search';
        }
        else {
            $dat[1]['title'] = 'There are no items in the Recycle Bin';
        }
        $dat[1]['class'] = 'p20 center';
        jrCore_page_table_row($dat);
    }
    jrCore_page_table_footer();
    return true;
}

/**
 * Get Available DS count options for dashboard
 */
function jrCore_get_dashboard_ds_count_options()
{
    global $_mods;
    // Add in Daily/Monthly DS counts
    $_tmp = array('jrCore' => array());
    foreach (jrCore_get_datastore_modules() as $m => $p) {
        $name                  = $_mods[$m]['module_name'] . ' - daily';
        $_tmp['jrCore'][$name] = 'jrCore_dashboard_ds_counts';
        $name                  = $_mods[$m]['module_name'] . ' - monthly';
        $_tmp['jrCore'][$name] = 'jrCore_dashboard_ds_counts';
        $name                  = $_mods[$m]['module_name'] . ' - total';
        $_tmp['jrCore'][$name] = 'jrCore_dashboard_ds_counts';
    }
    $_new = array();
    $keys = array_keys($_tmp['jrCore']);
    natsort($keys);
    foreach ($keys as $k) {
        $_new[$k] = $_tmp['jrCore'][$k];
    }
    $_tmp['jrCore'] = $_new;
    return $_tmp;
}

/**
 * Dashboard Panels
 * @param $panel
 * @return array
 */
function jrCore_dashboard_ds_counts($panel)
{
    global $_mods;
    $out = array('title' => 0);
    if ($_opts = jrCore_get_dashboard_ds_count_options()) {
        if (isset($_opts['jrCore'][$panel])) {
            list($name,) = explode('-', $panel, 2);
            $name = trim($name);
            foreach ($_mods as $dir => $m) {
                if ($m['module_name'] == $name) {
                    if (strpos($panel, 'total')) {
                        $cnt = jrCore_db_get_ds_row_count($dir);
                    }
                    else {
                        $days = 1;
                        if (strpos($panel, 'monthly')) {
                            $days = 30;
                        }
                        $_sp = array(
                            'search'          => array(
                                "_created > " . (time() - ($days * 86400))
                            ),
                            'skip_all_checks' => true,
                            'return_count'    => true,
                            'limit'           => 100000
                        );
                        $cnt = jrCore_db_search_items($dir, $_sp);
                    }
                    $out = array('title' => jrCore_number_format($cnt));
                }
            }
        }
    }
    return $out;
}

/**
 * User Profiles Dashboard Panels
 * @param $panel
 * @return array|false
 */
function jrCore_dashboard_panels($panel)
{
    global $_mods;
    // The panel being asked for will come in as $panel
    $out = false;
    switch ($panel) {

        case 'pending items':
            $tbl = jrCore_db_table_name('jrCore', 'pending');
            $req = "SELECT COUNT(pending_id) AS cnt FROM {$tbl} WHERE pending_linked_item_id = 0";
            $_rt = jrCore_db_query($req, 'SINGLE');
            $cnt = ($_rt && is_array($_rt) && isset($_rt['cnt'])) ? intval($_rt['cnt']) : 0;
            $out = array(
                'title' => jrCore_number_format($cnt)
            );
            break;

        case 'installed modules':
            $out = array(
                'title' => count($_mods)
            );
            break;

        case 'installed skins':
            $out = array(
                'title' => count(jrCore_get_skins())
            );
            break;

        case 'queue depth':
            $tbl = jrCore_db_table_name('jrCore', 'queue_info');
            $req = "SELECT SUM(queue_depth) AS c FROM {$tbl}";
            $_rt = jrCore_db_query($req, 'SINGLE');
            $num = ($_rt && is_array($_rt)) ? intval($_rt['c']) : 0;
            $out = array(
                'title' => jrCore_number_format($num)
            );
            break;

        case 'memory used':
            $_rm = jrCore_get_system_memory();
            if (isset($_rm['percent_used']) && is_numeric($_rm['percent_used'])) {
                $out = array(
                    'title' => $_rm['percent_used'] . "%<br><span>" . jrCore_format_size($_rm['memory_used']) . " of " . jrCore_format_size($_rm['memory_total']) . '</span>',
                    'class' => (isset($_rm['class']) ? $_rm['class'] : 'bigsystem-inf')
                );
            }
            else {
                $out = array(
                    'title' => '?',
                    'class' => 'bigsystem-inf'
                );
            }
            break;

        case 'disk usage':
            $_ds = jrCore_get_disk_usage();
            if (isset($_ds['percent_used']) && is_numeric($_ds['percent_used'])) {
                $out = array(
                    'title' => $_ds['percent_used'] . "%<br><span>" . jrCore_format_size($_ds['disk_used']) . " of " . jrCore_format_size($_ds['disk_total']) . '</span>',
                    'class' => (isset($_ds['class']) ? $_ds['class'] : 'bigsystem-inf')
                );
            }
            else {
                $out = array(
                    'title' => '?',
                    'class' => 'bigsystem-inf'
                );
            }
            break;

        case 'CPU count':
            $_pc = jrCore_get_proc_info();
            if ($_pc && is_array($_pc)) {
                $num = count($_pc);
                jrCore_set_flag('jrCore_dashboard_cpu_num', $num);
                $out = array(
                    'title' => "{$num}<span>{$_pc[1]['model']}</span>",
                    'class' => 'bigsystem-inf'
                );
            }
            else {
                $out = array(
                    'title' => '?',
                    'class' => 'bigsystem-inf'
                );
            }
            break;

        case '1 minute load':
        case '5 minute load':
        case '15 minute load':
            $min = (int) jrCore_string_field($panel, 1);
            if (!$num = jrCore_get_flag('jrCore_dashboard_cpu_num')) {
                $num = jrCore_get_proc_info();
                if ($num && is_array($num)) {
                    $num = count($num);
                }
            }
            $_ll = jrCore_get_system_load($num);
            if (isset($_ll) && is_array($_ll)) {
                $out = array(
                    'title' => "{$_ll[$min]['level']}<br><span>{$_ll[1]['level']}, {$_ll[5]['level']}, {$_ll[15]['level']}</span>",
                    'class' => $_ll[$min]['class']
                );
            }
            else {
                $out = array(
                    'title' => '?',
                    'class' => 'bigsystem-inf'
                );
            }
            break;

        default:

            // All other "DS" Counts
            if (strpos($panel, 'item count')) {
                $mod = trim(jrCore_string_field($panel, 1));
                $out = array(
                    'title' => jrCore_db_get_ds_row_count($mod),
                    'graph' => "{$mod}|ds_items_by_day"
                );
            }
            break;

    }
    return ($out) ?: false;
}
